package com.bizunited.platform.kuiper.starter.common.excel;

import com.bizunited.platform.kuiper.starter.common.constant.ExcelConstants;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelMigrateException;
import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.poi.ss.usermodel.DateUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 基于 apache poi的excel
 *
 * @author Keller
 * @create 2020/8/23
 */
public class ExcelUtils {

  private static final Logger LOGGER = LoggerFactory.getLogger(ExcelUtils.class);


  private ExcelUtils() {
  }

  public static ConvertUtilsBean convertUtilsBean = new ConvertUtilsBean();

  static {
    convertUtilsBean.register(new Converter() {

      @SuppressWarnings("unchecked")
      @Override
      public <T> T convert(Class<T> type, Object value) {
        if (value != null) {
          return (T) DateUtil.getJavaDate(Double.parseDouble(value.toString()));
        }
        return null;
      }
    }, Date.class);
    convertUtilsBean.register(new Converter() {

      @SuppressWarnings("unchecked")
      @Override
      public <T> T convert(Class<T> type, Object value) {
        if (value != null) {
          String src = value.toString();
          if (src.matches("\\d+(\\.\\d+)?")) {
            BigDecimal bigDecimal = new BigDecimal(value.toString());
            return (T) new Integer(bigDecimal.intValue());
          }
        }
        return null;
      }
    }, Integer.class);
  }

  /**
   * 转换成指定类型的对象
   *
   * @param o
   * @param c
   * @return
   */
  public static Object convert(Object o, Class<?> c) throws ExcelMigrateException {
    if (o == null) {
      return "";
    }
    return convertUtilsBean.convert(o.toString(), c);
  }

  /**
   * 数组是否为空
   */
  public static boolean isEmpty(Object[] os) {
    boolean isEmpty = true;
    if (ArrayUtils.isNotEmpty(os)) {
      for (Object o : os) {
        if (o != null && !"".equals(o.toString())) {
          isEmpty = false;
          break;
        }
      }
    }
    return isEmpty;
  }

  /**
   * 根据字母返回列数
   */
  public static int getColumnIndex(String c) {
    // DCBA
    int res = ExcelConstants.columnMap.get(c.charAt(c.length() - 1));
    if (c.length() > 1) {
      for (int i = 1; i < c.length(); i++) {
        char r = c.charAt(i);
        res = res + new Double(Math.pow(26, c.length() - i)).intValue() * ExcelConstants.columnMap.get(r);
      }
    }
    return res;
  }

  /**
   * 字母列转换为数字 Excel column index begin 1
   *
   * @param columnIndex
   * @return
   */
  public static String excelColIndexToStr(int columnIndex) {
    if (columnIndex <= 0) {
      return null;
    }
    String columnStr = "";
    columnIndex--;
    do {
      if (columnStr.length() > 0) {
        columnIndex--;
      }
      columnStr = ((char) (columnIndex % 26 + (int) 'A')) + columnStr;
      columnIndex = ((columnIndex - columnIndex % 26) / 26);
    } while (columnIndex > 0);
    return columnStr;
  }

  private static boolean match(String regex, String str) {
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(str);
    return matcher.matches();
  }

  /**
   * 数据写入对应javabean
   *
   * @param clazz
   * @param values
   * @return
   * @throws InstantiationException
   * @throws IllegalAccessException
   */
  public static Object setValue(Class<?> clazz, Object[] values) throws InstantiationException, IllegalAccessException {
    Object o = clazz.newInstance();
    Field[] fields = clazz.getDeclaredFields();
    for (Field field : fields) {
      NebulaExcelColumn nebulaExcelColumn = field.getAnnotation(NebulaExcelColumn.class);
      if (nebulaExcelColumn == null) {
        continue;
      }
      field.setAccessible(true);
      Class<?> fieldType = field.getType();
      Method fieldSetMet = null;
      try {
        fieldSetMet = clazz.getMethod("set" + StringUtils.capitalize(field.getName()), field.getType());
      } catch (NoSuchMethodException e) {
        throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法不存在，请检查:" + field.getName(), e);
      }
      Object value = values[nebulaExcelColumn.order()];
      String tmp = "";
      //null 情况下 返回空字符串
      if (value != null) {
        tmp = String.valueOf(value);
      }

      //要做用户自定义正则表达式
      if (StringUtils.isNoneBlank(nebulaExcelColumn.regex())) {
        boolean check = match(nebulaExcelColumn.regex(), tmp);
        if (!check) {
          throw new ExcelMigrateException(nebulaExcelColumn, "规则验证失败 字段【" + field.getName() + "】regex:" + nebulaExcelColumn.regex() + " 值:" + tmp);
        }
      }
      if(!nebulaExcelColumn.nullable() && StringUtils.isBlank(tmp)){
        throw new ExcelMigrateException(nebulaExcelColumn, "非空规则验证失败 字段【" + field.getName()+"】");
      }
       //字符串类型处理
      if (String.class.isAssignableFrom(fieldType)) {
        try {
          fieldSetMet.invoke(o, tmp);
        } catch (IllegalAccessException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法无法访问，请检查:" + field.getName(), e);
        } catch (InvocationTargetException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法调用失败，请检查:" + field.getName(), e);
        }
        //日期格式处理
      } else if (Date.class.isAssignableFrom(fieldType)) {
        if(StringUtils.isBlank(tmp)){
          throw new ExcelMigrateException("Date类型数据为空，请检查");
        }
        String dateFormat = nebulaExcelColumn.format();
        Date date = null;
        try {
          if (StringUtils.isNoneBlank(dateFormat)) {
            SimpleDateFormat sdf = new SimpleDateFormat(dateFormat);
            date = sdf.parse(tmp);
          } else {
            date = DateUtils.parseDate(tmp, "yyyy-MM-dd HH:mm:ss");
          }
        } catch (Exception e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "日期格式转换异常：日期数据(" + tmp + ") 期望输入格式（" + dateFormat + "）", e);
        }
        try {
          fieldSetMet.invoke(o, date);
        } catch (IllegalAccessException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法无法访问，请检查:" + field.getName(), e);
        } catch (InvocationTargetException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法调用失败，请检查:" + field.getName(), e);
        }
        //整型处理
      } else if (Integer.class.isAssignableFrom(fieldType) || int.class.isAssignableFrom(fieldType)) {
        if(StringUtils.isBlank(tmp)){
          throw new ExcelMigrateException("Integer类型数据为空，请检查");
        }
        Integer intval = Integer.parseInt(tmp);
        try {
          fieldSetMet.invoke(o, intval);
        } catch (IllegalAccessException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法无法访问，请检查:" + field.getName(), e);
        } catch (InvocationTargetException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法调用失败，请检查:" + field.getName(), e);
        }
        //Bigdecimal处理
      } else if (BigDecimal.class.isAssignableFrom(fieldType)) {
        if(StringUtils.isBlank(tmp)){
          throw new ExcelMigrateException("BigDecimal类型数据为空，请检查");
        }
        BigDecimal intval = new BigDecimal(tmp);
        try {
          fieldSetMet.invoke(o, intval);
        } catch (IllegalAccessException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法无法访问，请检查:" + field.getName(), e);
        } catch (InvocationTargetException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法调用失败，请检查:" + field.getName(), e);
        }
        //Long类型处理
      } else if (Long.class.isAssignableFrom(fieldType) || long.class.isAssignableFrom(fieldType)) {
        if(StringUtils.isBlank(tmp)){
          throw new ExcelMigrateException("Long类型数据为空，请检查");
        }
        Long temp = Long.parseLong(tmp);
        try {
          fieldSetMet.invoke(o, temp);
        } catch (IllegalAccessException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法无法访问，请检查:" + field.getName(), e);
        } catch (InvocationTargetException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法调用失败，请检查:" + field.getName(), e);
        }
        //Double类型处理
      } else if (Double.class.isAssignableFrom(fieldType) || double.class.isAssignableFrom(fieldType)) {
        if(StringUtils.isBlank(tmp)){
          throw new ExcelMigrateException("Double类型数据为空，请检查");
        }
        Double temp = null;
        if (StringUtils.endsWith(tmp, "%")) {
          //存在%的格式数据处理
          tmp = StringUtils.substringBefore(tmp, "%");
          temp = Double.parseDouble(tmp) / 100;
        } else {
          temp = Double.parseDouble(tmp);
        }
        try {
          fieldSetMet.invoke(o, temp);
        } catch (IllegalAccessException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法无法访问，请检查:" + field.getName(), e);
        } catch (InvocationTargetException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法调用失败，请检查:" + field.getName(), e);
        }
        //Boolean类型处理
      } else if (Boolean.class.isAssignableFrom(fieldType) || boolean.class.isAssignableFrom(fieldType)) {
        if(StringUtils.isBlank(tmp)){
          throw new ExcelMigrateException("Boolean类型数据为空，请检查");
        }
        Boolean temp = Boolean.parseBoolean(tmp);
        try {
          fieldSetMet.invoke(o, temp);
        } catch (IllegalAccessException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法无法访问，请检查:" + field.getName(), e);
        } catch (InvocationTargetException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法调用失败，请检查:" + field.getName(), e);
        }
        //Float类型处理
      } else if (Float.class.isAssignableFrom(fieldType) || float.class.isAssignableFrom(fieldType)) {
        if(StringUtils.isBlank(tmp)){
          throw new ExcelMigrateException("Float类型数据为空，请检查");
        }
        Float temp = null;
        //存在%的格式数据处理
        if (StringUtils.endsWith(tmp, "%")) {
          tmp = StringUtils.substringBefore(tmp, "%");
          temp = Float.parseFloat(tmp) / 100;
        } else {
          temp = Float.parseFloat(tmp);
        }
        try {
          fieldSetMet.invoke(o, temp);
        } catch (IllegalAccessException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法无法访问，请检查:" + field.getName(), e);
        } catch (InvocationTargetException e) {
          throw new ExcelMigrateException(nebulaExcelColumn, "字段属性set方法调用失败，请检查:" + field.getName(), e);
        }
        //其他定义类型的统一处理
      } else {
        LOGGER.error(String.format("not supper type: %s"), fieldType);
        throw new ExcelMigrateException(nebulaExcelColumn, "字段类型无效 字段:" + field.getName() + " 类型:" + fieldType);
      }
    }
    return o;
  }

  public static Object setValue(Object[] values) {
    Set<Object> o = Arrays.stream(values).collect(Collectors.toSet());
    return o;
  }
}