package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.FormDetailsExportBoxEntity;
import com.bizunited.platform.kuiper.starter.service.FormDetailsExportBoxService;
import com.bizunited.platform.kuiper.starter.vo.FormDetailsExportBoxVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 基于表单引擎，列表引擎，对业务数据进行异步的导出功能
 * 实现工具箱方式导出文件。1、导出数据。2、下载文件
 *
 * @author Keller
 * @create 2020/8/24
 */
@Api(value = "FormDetailsExportBoxController")
@RestController
@RequestMapping("/v1/kuiper/formDetailsExportBox")
public class FormDetailsExportBoxController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(FormDetailsExportBoxController.class);

  @Autowired
  private FormDetailsExportBoxService formDetailsExportBoxService;

  /**
   * 创建导出数据记录
   *
   * @return
   */
  @ApiOperation(value = "创建导出数据记录")
  @PostMapping("/export")
  public ResponseModel export(
          @Valid FormDetailsExportBoxVo formDetailsExportBoxVo,
          HttpServletRequest request) {

    // 构建可能的params信息
    Map<String, Object> params = new HashMap<>();
    Enumeration<String> names = request.getParameterNames();
    if (names != null) {
      while (names.hasMoreElements()) {
        String name = names.nextElement();
        String[] values = request.getParameterValues(name);
        if (values != null && values.length == 1) {
          params.put(name, values[0]);
        } else if (values != null && values.length > 1) {
          params.put(name, values);
        }
      }
    }

    try {
      return this.buildHttpResultW(formDetailsExportBoxService.export(formDetailsExportBoxVo, params), "creator", "executor");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取所有导出记录信息（分页）
   *
   * @return
   */
  @ApiOperation(value = "查询导出工具箱列表")
  @GetMapping("/queryAll")
  public ResponseModel queryAll(@ApiParam("分页信息") @PageableDefault(value = 5) Pageable pageable) {
    try {
      Page<FormDetailsExportBoxEntity> entitys = formDetailsExportBoxService.queryPage(pageable);
      return this.buildHttpResultW(entitys);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查看导出单条信息")
  @GetMapping("/{id}")
  public ResponseModel query(@PathVariable("id") @ApiParam(value = "主键", required = true) String id) {
    try {
      FormDetailsExportBoxEntity entity = formDetailsExportBoxService.findById(id);
      return this.buildHttpResultW(entity);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 清除导出记录
   *
   * @return
   */
  @ApiOperation(value = "清除导出记录")
  @DeleteMapping("/{id}")
  public ResponseModel deleteById(@PathVariable("id") @ApiParam(value = "主键", required = true) String id) {
    try {
      formDetailsExportBoxService.deleteById(id);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 取消当前导出任务
   *
   * @return
   */
  @ApiOperation(value = "取消当前导入进程")
  @PutMapping("/{id}")
  public ResponseModel cancel(@PathVariable("id") @ApiParam(value = "主键", required = true) String id) {
    try {
      formDetailsExportBoxService.cancel(id);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 下载导出数据
   *
   * @return
   */
  @ApiOperation(value = "下载导出数据")
  @GetMapping("/download/{id}")
  public void download(@PathVariable("id") @ApiParam(value = "主键", required = true) String id,
                       HttpServletRequest request,
                       HttpServletResponse response) {
    try {
      byte[] bytes = formDetailsExportBoxService.download(id);
      FormDetailsExportBoxEntity entity = formDetailsExportBoxService.findById(id);
      Validate.isTrue(ArrayUtils.isNotEmpty(bytes), "没有任何导出信息");

      //获取下载文件名
      String fileName = entity.getDownloadFileName();
      writeResponseFile(request, response, bytes, fileName);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      printResponseModel(buildHttpResultForException(e), response);
    }
  }

  @ApiOperation(value = "通过该接口，系统将可以返回该进程中已经被扫描到的所有业务数据导入处理器的类名信息")
  @GetMapping(value = "/findProcessClassNames")
  public ResponseModel findProcessClassNames() {
    try {
      Set<String> results = this.formDetailsExportBoxService.findProcessClassNames();
      return this.buildHttpResultW(results, new String[]{});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "清除导入列表所有数据")
  @DeleteMapping(value = "/deleteAll")
  public ResponseModel clearAll() {
    try {
      formDetailsExportBoxService.deleteAll();
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
