package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.util.FileUtils;
import com.bizunited.platform.kuiper.entity.FormDetailsImportBoxDetailsEntity;
import com.bizunited.platform.kuiper.entity.FormDetailsImportBoxEntity;
import com.bizunited.platform.kuiper.starter.common.enums.ExcelTypeEnum;
import com.bizunited.platform.kuiper.starter.common.excel.ExcelExportWrapper;
import com.bizunited.platform.kuiper.starter.common.excel.ExcelImportWrapper;
import com.bizunited.platform.kuiper.starter.common.excel.ExcelImportWrapperBuilder;
import com.bizunited.platform.kuiper.starter.common.excel.ExcelUtils;
import com.bizunited.platform.kuiper.starter.common.excel.IRecordInterceptor;
import com.bizunited.platform.kuiper.starter.common.excel.NebulaExcelImport;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelMigrateException;
import com.bizunited.platform.kuiper.starter.repository.FormDetailsImportBoxDetailsRepository;
import com.bizunited.platform.kuiper.starter.repository.FormDetailsImportBoxRepository;
import com.bizunited.platform.kuiper.starter.service.FormDetailesImportBoxExtService;
import com.bizunited.platform.kuiper.starter.service.FormDetailsImportBoxTaskService;
import com.bizunited.platform.kuiper.starter.service.instances.imports.FormDetailsImportBoxProcess;
import com.bizunited.platform.venus.common.service.file.FileRelativeTemplate;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.hibernate.HibernateException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DataAccessException;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.AnnotatedType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Date;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author Keller
 * @create 2020/8/25
 */
@Service("FormDetailsImportBoxTaskServiceImpl")
public class FormDetailsImportBoxTaskServiceImpl implements FormDetailsImportBoxTaskService {

  private static final Logger LOGGER = LoggerFactory.getLogger(FormDetailsImportBoxTaskServiceImpl.class);

  private static final String FORM_DETAILS = "formdetails";

  @Autowired
  private FormDetailsImportBoxRepository formDetailsImportBoxRepository;
  @Autowired
  private FormDetailsImportBoxDetailsRepository formDetailsImportBoxDetailsRepository;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private FileRelativeTemplate fileRelativeTemplate;
  @Autowired(required = false)
  private FormDetailesImportBoxExtService formDetailesImportBoxExtService;
  @Autowired
  private PlatformContext platformContext;

  @Value("${venus.file.fileRoot}")
  private String fileRoot;

  @Async("fromDetailsMigrateExecutor")
  @Override
  public void importProcess(FormDetailsImportBoxEntity entity, FormDetailsImportBoxProcess<Object, Object> process, String account, Map<String, Object> params) {
    /*
     * 1.初始化excel参数，通过NebulaExcelImport注解获取（或默认参数）
     * 2.按行读取excel，封装excelbean数据
     * 3.调用业务process方法封装与效验excelbean数据
     * 4.获取业务process返回数据进行二次数据封装
     * 5.调用业务execute业务数据写入功能
     */
    LOGGER.debug("异步执行导入");

    //1.初始化excel参数，通过NebulaExcelImport注解获取（或默认参数）
    AtomicInteger startRow = new AtomicInteger(0);
    AtomicInteger startColumn = new AtomicInteger(0);
    AtomicInteger endColumn = new AtomicInteger(0);
    AtomicInteger sheetIndex = new AtomicInteger(0);

    //获取导入excel对应的bean文件
    Class<?> excelBean = getInterfaceGeneric(process, 1, FormDetailsImportBoxProcess.class);

    //获取bean标注信息
    NebulaExcelImport nebulaExcelImport = null;
    if (excelBean != null) {
      nebulaExcelImport = excelBean.getAnnotation(NebulaExcelImport.class);
    }
    //可以支持不添加ExcelImport注解模式会封装成Object[] 传入业务处理 暂不开放
    Validate.notNull(nebulaExcelImport, "excelProperty bean未添加NebulaExcelImport注解,请检查");

    //扩展后续支撑无需标注信息情况下的数据导入
    startRow.addAndGet(nebulaExcelImport.startRow());
    startColumn.addAndGet(nebulaExcelImport.startColumn());
    endColumn.addAndGet(nebulaExcelImport.endColumn());
    sheetIndex.addAndGet(nebulaExcelImport.sheetIndex());

    //初始化统计信息
    AtomicInteger totalNum = new AtomicInteger(0);
    AtomicInteger successNum = new AtomicInteger(0);
    AtomicInteger failedNum = new AtomicInteger(0);

    //判断endColumn有效性
    if (endColumn.intValue() <= startColumn.intValue()) {
      //使用excelProperty字段数量来设置读取数据结束列数量
      endColumn = new AtomicInteger(excelBean.getDeclaredFields().length);
    }

    //加载导入文件
    byte[] fileByte = venusFileService.readFileContent(entity.getRelativeLocal(), entity.getFileName());
    Validate.isTrue(fileByte != null && fileByte.length > 0, "导入文件为空");

    //加载模板文件用于写入错误数据
    byte[] templateByte = venusFileService.readFileContent(entity.getTemplateFileRelativeLocal(), entity.getTemplateFileName());
    Validate.isTrue(templateByte != null && templateByte.length > 0, "导入模板文件为空");

    //未解决POI bug导入文件写入临时文件
    File tmpFile = FileUtils.writeLocalFile(fileByte, fileRoot, ExcelTypeEnum.XLSX.getValue());

    //创建错误信息文件
    String relativePath = fileRelativeTemplate.generatRelativePath(FORM_DETAILS, "");
    File errorFile = createLocalExcelFile(templateByte, relativePath);

    try (InputStream is = new ByteArrayInputStream(fileByte);
         InputStream ist = new ByteArrayInputStream(templateByte);
         OutputStream os = new FileOutputStream(errorFile);
    ) {
      //初始化excel错误信息写入封装器
      ExcelExportWrapper excelExportWrapper = new ExcelExportWrapper(ist);
      //初始化excel读取封装器
      ExcelImportWrapper excelImportWrapper = ExcelImportWrapperBuilder.create(is, tmpFile).setDomReader(false).builder();
      //单行读取数据并传入业务处理
      excelImportWrapper.readSheet(sheetIndex.intValue(), startRow.intValue(), startColumn.intValue(), endColumn.intValue(), new IRecordInterceptor<Object[]>() {

        @Override
        @Transactional(rollbackOn = Exception.class)
        public void handle(Object[] t, int row) {
          FormDetailsImportBoxEntity status = checkImportStatus(entity.getId());
          if (status.getExecuteResult() != 1) {
            LOGGER.info(String.format("导入进程被暂停，导入ID：%s", entity.getId()));
            throw new IllegalArgumentException(String.format("导入进程被暂停，导入ID：%s", entity.getId()));
          }
          try {
            Object o = null;
            Object bean = null;

            //2.按行读取excel，封装excelbean数据
            if (excelBean != null) {
              o = ExcelUtils.setValue(excelBean, t);
            } else {
              o = ExcelUtils.setValue(t);
            }
            //3.调用业务process方法封装与效验excelbean数据
            bean = process.process(o, params, entity.getTemplateCode(), entity.getTemplateVersion());
            Validate.notNull(bean, "本次导入操作未返回任何数，请检查业务数据导入前的校验逻辑!!");

            //4.获取业务process返回数据进行二次数据封装
            if (StringUtils.isNoneBlank(entity.getTemplateCode())) {
              Validate.notNull(formDetailesImportBoxExtService, "请导入表单引擎引用后，进行状态表单相关操作!");
              formDetailesImportBoxExtService.fillFormInstanceId(entity, bean, account);
            }

            //5.调用业务execute业务数据写入功能
            process.execute(bean, params);

            successNum.getAndIncrement();
          } catch (InstantiationException | IllegalAccessException e) {
            LOGGER.error(e.getMessage(), e);
            throw new IllegalArgumentException("ExcelBean默认构造函数初始化失败，请检查！", e);
          } catch (ExcelMigrateException | DataAccessException | HibernateException e) {
            LOGGER.info(e.getMessage(), e);
            errorRecord(entity, row, e.getMessage());
            Object[] v = ArrayUtils.add(t, e.getMessage());
            excelExportWrapper.paint(0, failedNum.getAndIncrement() + startRow.intValue(), startColumn.intValue(), v);
          } catch (Exception e) {
            //针对Exception未约定的异常进行拦截并记录
            LOGGER.error(e.getMessage(), e);
            errorRecord(entity, row, e.getMessage());
            Object[] v = ArrayUtils.add(t, e.getMessage());
            excelExportWrapper.paint(0, failedNum.getAndIncrement() + startRow.intValue(), startColumn.intValue(), v);
          } finally {
            //单条记录处理完成
            LOGGER.info("完成单条记录处理完成");
            //检查是否存在失败记录，更新导入表(批量执行 暂无此需求暂不扩展实现)
            totalNum.getAndIncrement();
          }
        }
      });
      excelExportWrapper.export(os);
    } catch (FileNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("导入文件路径不存在");
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("导入文件读取失败");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      throw e;
    } finally {
      org.apache.commons.io.FileUtils.deleteQuietly(tmpFile);
      //更新导入记录
      complateRecord(entity, successNum.intValue(), failedNum.intValue(), totalNum.intValue(), errorFile.getName(), relativePath);
    }
    //错误文件信息上传至venus
    try {
      venusFileService.saveFile(relativePath, errorFile.getName(), errorFile.getName(), org.apache.commons.io.FileUtils.readFileToByteArray(errorFile));
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("导出错误文件");
    }
    LOGGER.info("导入工作完成");
  }

  @Override
  @Async("fromDetailsMigrateExecutor")
  public void checkProcess(FormDetailsImportBoxEntity entity, FormDetailsImportBoxProcess<Object, Object> process, String account, Map<String, Object> params) {
    /*
     * 1.初始化excel参数，通过NebulaExcelImport注解获取（或默认参数）
     * 2.按行读取excel，封装excelbean数据
     * 3.调用业务process方法封装与效验excelbean数据
     */
    LOGGER.debug("异步excel校验");

    //获取导入excel对应的bean文件
    Class<?> excelBean = getInterfaceGeneric(process, 1, FormDetailsImportBoxProcess.class);

    //获取bean标注信息
    NebulaExcelImport nebulaExcelImport = null;
    if (excelBean != null) {
      nebulaExcelImport = excelBean.getAnnotation(NebulaExcelImport.class);
    }
    //可以支持不添加ExcelImport注解模式会封装成Object[] 传入业务处理 暂不开放
    Validate.notNull(nebulaExcelImport, "excelProperty bean未添加NebulaExcelImport注解,请检查");

    //1.初始化excel参数，通过NebulaExcelImport注解获取（或默认参数）
    AtomicInteger startRow = new AtomicInteger(nebulaExcelImport.startRow());
    AtomicInteger startColumn = new AtomicInteger(nebulaExcelImport.startColumn());
    AtomicInteger endColumn = new AtomicInteger(nebulaExcelImport.endColumn());
    AtomicInteger sheetIndex = new AtomicInteger(nebulaExcelImport.sheetIndex());

    //判断endColumn有效性
    if (endColumn.intValue() <= startColumn.intValue()) {
      //使用excelProperty字段数量来设置读取数据结束列数量
      endColumn = new AtomicInteger(excelBean.getDeclaredFields().length);
    }

    //加载导入文件
    byte[] fileByte = venusFileService.readFileContent(entity.getRelativeLocal(), entity.getFileName());
    Validate.isTrue(fileByte != null && fileByte.length > 0, "导入文件为空");
    try (InputStream is = new ByteArrayInputStream(fileByte)) {
      //未解决POI bug导入文件写入临时文件
      File tmpFile = FileUtils.writeLocalFile(fileByte, fileRoot, ExcelTypeEnum.XLSX.getValue());
      //初始化excel读取封装器
      ExcelImportWrapper excelImportWrapper = ExcelImportWrapperBuilder.create(is, tmpFile).setDomReader(false).builder();
      //单行读取数据并传入业务处理
      excelImportWrapper.readSheet(sheetIndex.intValue(), startRow.intValue(), startColumn.intValue(), endColumn.intValue(), new IRecordInterceptor<Object[]>() {
        @Override
        @Transactional(rollbackOn = Exception.class)
        public void handle(Object[] t, int row) {
          FormDetailsImportBoxEntity status = checkImportStatus(entity.getId());
          Validate.isTrue(status.getExecuteResult() != 0 && status.getExecuteResult() != 2 && status.getExecuteResult() != 3, String.format("导入进程被暂停，导入ID：%s", entity.getId()));
          try {
            Object o;
            //2.按行读取excel，封装excelbean数据
            if (excelBean != null) {
              o = ExcelUtils.setValue(excelBean, t);
            } else {
              o = ExcelUtils.setValue(t);
            }
            //3.调用业务process方法封装与效验excelbean数据
            process.process(o, params, entity.getTemplateCode(), entity.getTemplateVersion());
          } catch (InstantiationException | IllegalAccessException e) {
            faild(entity);
            LOGGER.error(e.getMessage(), e);
            throw new IllegalArgumentException("ExcelBean默认构造函数初始化失败，请检查！", e);
          } catch (ExcelMigrateException | DataAccessException | HibernateException e) {
            faild(entity);
            LOGGER.info(e.getMessage(), e);
            throw new IllegalArgumentException("导入错误，请检查！", e);
          } catch (Exception e) {
            faild(entity);
            //针对Exception未约定的异常进行拦截并记录
            LOGGER.error(e.getMessage(), e);
            throw new IllegalArgumentException("导入错误，请检查！", e);
          }
        }
      });
    } catch (FileNotFoundException e) {
      faild(entity);
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("导入文件路径不存在");
    } catch (IOException e) {
      faild(entity);
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("导入文件读取失败");
    } catch (Exception e) {
      faild(entity);
      LOGGER.error(e.getMessage(), e);
      throw e;
    }
    complateCheck(entity);
    LOGGER.info("导入校验工作完成");
  }

  @Override
  @Async("fromDetailsMigrateExecutor")
  public void asyncImport(FormDetailsImportBoxEntity entity, FormDetailsImportBoxProcess<Object, Object> process, String account, Map<String, Object> params) {
    // 设置当前导入记录主键信息
    params.put("id", entity.getId());
    process.importExt(params, entity.getTemplateCode(), entity.getTemplateVersion());
  }

  /**
   * 创建本地excel文件
   *
   * @param bytes
   * @param relativePath
   * @return
   */
  private File createLocalExcelFile(byte[] bytes, String relativePath) {
    String renameImage = UUID.randomUUID().toString();
    String fileRename = StringUtils.join(renameImage, ".", "xlsx");
    File folder = new File(StringUtils.join(fileRoot, relativePath));
    if (!folder.exists()) {
      folder.mkdirs();
    }
    //保存错误文件至指定文件夹
    File file = new File(folder, fileRename);
    try {
      Validate.isTrue(file.createNewFile(), "创建本地文件失败：%s", file.getAbsolutePath());
    } catch (IOException e) {
      throw new IllegalArgumentException(String.format("创建本地文件失败：%s", e.getMessage()));
    }
    if (bytes != null) {
      ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
      try (FileOutputStream fos = new FileOutputStream(file)) {
        int maxLen = 4096;
        int realLen;
        byte[] contents = new byte[maxLen];
        while ((realLen = bais.read(contents, 0, maxLen)) != -1) {
          fos.write(contents, 0, realLen);
        }
      } catch (FileNotFoundException e) {
        throw new IllegalArgumentException(String.format("未找到本地文件：%s", file.getAbsolutePath()));
      } catch (IOException e) {
        throw new IllegalArgumentException(String.format("读取本地文件失败：%s", e.getMessage()));
      }
    }
    return file;
  }

  /**
   * 错误记录信息
   *
   * @param entity
   * @param row
   * @param message
   * @return
   */
  @Transactional(rollbackOn = Exception.class)
  FormDetailsImportBoxDetailsEntity errorRecord(FormDetailsImportBoxEntity entity, int row, String message) {
    Validate.notNull(entity, "导入记录不能为空");
    FormDetailsImportBoxDetailsEntity detailsEntity = new FormDetailsImportBoxDetailsEntity();
    detailsEntity.setFormDetailsImportBoxEntity(entity);
    detailsEntity.setRowNum(row);
    detailsEntity.setErrorMsg(message);
    detailsEntity.setProjectName(platformContext.getAppName());
    detailsEntity = formDetailsImportBoxDetailsRepository.saveAndFlush(detailsEntity);
    return detailsEntity;
  }

  /**
   * 导入完成进度保存
   *
   * @param entity
   * @param successNum
   * @param failedNum
   * @param totalNum
   * @param errorFileName
   * @param errorFileRelativeLocal
   * @return
   */
  @Transactional(rollbackOn = Exception.class)
  FormDetailsImportBoxEntity complateRecord(FormDetailsImportBoxEntity entity, int successNum, int failedNum, int totalNum, String errorFileName, String errorFileRelativeLocal) {
    Validate.notNull(entity, "导入记录不能为空");
    entity.setExecuteEndTime(new Date());
    entity.setSuccessNum(successNum);
    if ((successNum + failedNum) != totalNum) {
      //修复错误数据数量（在未处理的业务异常情况下）
      failedNum = totalNum - successNum;
    }
    entity.setFailedNum(failedNum);
    entity.setTotalNum(totalNum);
    entity.setErrorFileName(errorFileName);
    entity.setErrorFileRelativeLocal(errorFileRelativeLocal);
    entity.setProjectName(platformContext.getAppName());
    if (failedNum == 0) {
      entity.setExecuteResult(2);
    } else {
      entity.setExecuteResult(0);
    }
    entity.setProjectName(platformContext.getAppName());
    entity = this.formDetailsImportBoxRepository.saveAndFlush(entity);
    return entity;
  }

  /**
   * excel验证完成
   *
   * @param entity
   * @return
   */
  @Transactional(rollbackOn = Exception.class)
  FormDetailsImportBoxEntity complateCheck(FormDetailsImportBoxEntity entity) {
    Validate.notNull(entity, "导入记录不能为空");
    entity.setExecuteEndTime(new Date());
    entity.setProjectName(platformContext.getAppName());
    entity.setExecuteResult(5);
    entity = this.formDetailsImportBoxRepository.saveAndFlush(entity);
    return entity;
  }

  /**
   * excel验证失败
   *
   * @param entity
   * @return
   */
  @Transactional(rollbackOn = Exception.class)
  FormDetailsImportBoxEntity faild(FormDetailsImportBoxEntity entity) {
    Validate.notNull(entity, "导入记录不能为空");
    entity.setExecuteEndTime(new Date());
    entity.setProjectName(platformContext.getAppName());
    entity.setExecuteResult(0);
    entity = this.formDetailsImportBoxRepository.saveAndFlush(entity);
    return entity;
  }

  /**
   * 检查导入状态
   *
   * @param id
   * @return
   */
  private FormDetailsImportBoxEntity checkImportStatus(String id) {
    Validate.notBlank(id, "导入记录id不能为空");
    Optional<FormDetailsImportBoxEntity> op = this.formDetailsImportBoxRepository.findById(id);
    FormDetailsImportBoxEntity entity = op.orElse(null);
    Validate.notNull(entity, "导入数据记录错误");
    return entity;
  }

  /**
   * 获取指定接口泛型类型
   *
   * @param obj            实现接口的对象
   * @param index          第几位参数
   * @param interfaceClass 指定接口
   * @return
   */
  private Class<?> getInterfaceGeneric(Object obj, Integer index, Class<?> interfaceClass) {
    Class<?> clazz = null;
    if (AopUtils.isCglibProxy(obj) || AopUtils.isAopProxy(obj)) {
      //针对代理类的处理
      clazz = AopUtils.getTargetClass(obj);
    } else {
      clazz = obj.getClass();
    }
    AnnotatedType[] annotatedTypes = clazz.getAnnotatedInterfaces();
    if (annotatedTypes == null) {
      return null;
    }
    for (AnnotatedType annotatedType : annotatedTypes) {
      Type type = annotatedType.getType();
      if ((type instanceof ParameterizedType) && interfaceClass.getName().equals(((ParameterizedType) type).getRawType().getTypeName())) {
        Type[] params = ((ParameterizedType) type).getActualTypeArguments();
        if (params == null || params.length < index) {
          continue;
        }
        try {
          Class<?> cc = ((Class<?>) params[index]);
          ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
          return Class.forName(cc.getName(), false, classLoader);
        } catch (ClassNotFoundException e) {
          LOGGER.error(e.getMessage());
          throw new IllegalArgumentException(e);
        }
      }
    }
    return null;
  }
}
