package com.bizunited.platform.kuiper.starter.common.excel.reader;

import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.eventusermodel.EventWorkbookBuilder;
import org.apache.poi.hssf.eventusermodel.FormatTrackingHSSFListener;
import org.apache.poi.hssf.eventusermodel.HSSFEventFactory;
import org.apache.poi.hssf.eventusermodel.HSSFListener;
import org.apache.poi.hssf.eventusermodel.HSSFRequest;
import org.apache.poi.hssf.eventusermodel.MissingRecordAwareHSSFListener;
import org.apache.poi.hssf.eventusermodel.dummyrecord.LastCellOfRowDummyRecord;
import org.apache.poi.hssf.eventusermodel.dummyrecord.MissingCellDummyRecord;
import org.apache.poi.hssf.model.HSSFFormulaParser;
import org.apache.poi.hssf.record.BOFRecord;
import org.apache.poi.hssf.record.BlankRecord;
import org.apache.poi.hssf.record.BoolErrRecord;
import org.apache.poi.hssf.record.BoundSheetRecord;
import org.apache.poi.hssf.record.FormulaRecord;
import org.apache.poi.hssf.record.LabelRecord;
import org.apache.poi.hssf.record.LabelSSTRecord;
import org.apache.poi.hssf.record.NumberRecord;
import org.apache.poi.hssf.record.Record;
import org.apache.poi.hssf.record.SSTRecord;
import org.apache.poi.hssf.record.StringRecord;
import org.apache.poi.hssf.usermodel.HSSFDataFormatter;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Keller
 * @create 2020/8/24
 */
public class ExcelXlsReader implements HSSFListener {

  public static final Logger LOGGER = LoggerFactory.getLogger(ExcelXlsReader.class);

  /**
   * 最小处理列数
   */
  private int minColums = -1;

  /**
   * POI文件处理
   */
  private POIFSFileSystem fs;

  /**
   * 总行数
   */
  private int totalRows = 0;

  /**
   * 上一行row的序号
   */
  private int lastRowNumber;

  /**
   * 上一单元格的序号
   */
  private int lastColumnNumber;

  /**
   * 是否输出formula，还是它对应的值
   */
  private boolean outputFormulaValues = true;

  /**
   * 用于转换formulas
   */
  private EventWorkbookBuilder.SheetRecordCollectingListener workbookBuildingListener;

  /**
   * excel2003工作簿
   */
  private HSSFWorkbook stubWorkbook;

  /**
   * SSTR读取模式单条记录
   */
  private SSTRecord sstRecord;

  /**
   * 格式转换监听器
   */
  private FormatTrackingHSSFListener formatListener;

  /**
   * 日期格式化
   */
  private final HSSFDataFormatter formatter = new HSSFDataFormatter();

  /**
   * excel所有sheet数组（已排序）
   */
  private BoundSheetRecord[] orderedBSRs;

  /**
   * excel所有sheet数组容器
   */
  @SuppressWarnings("rawtypes")
  private ArrayList boundSheetRecords = new ArrayList<>();

  /**
   * 下一行索引
   */
  private int nextRow;

  /**
   * 下一列号
   */
  private int nextColumn;

  /**
   * 输出下一个单元格字符串值标识
   */
  private boolean outputNextStringRecord;

  /**
   * 当前行索引
   */
  private int curRow = 0;

  /**
   * 记录一行单元格数据
   */
  private List<String> cellList = new ArrayList<String>();

  /**
   * 判断整行是否为空行的标记
   */
  private boolean flag = false;

  /**
   * 遍历excel下所有的sheet
   *
   * @param fileName
   * @throws Exception
   */
  public int process(String fileName) {
    try {
      this.fs = new POIFSFileSystem(new FileInputStream(fileName));
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("excel 文件读取错误,请检查！");
    }
    MissingRecordAwareHSSFListener listener = new MissingRecordAwareHSSFListener(this);
    formatListener = new FormatTrackingHSSFListener(listener);
    HSSFEventFactory factory = new HSSFEventFactory();
    HSSFRequest request = new HSSFRequest();
    if (outputFormulaValues) {
      request.addListenerForAllRecords(formatListener);
    } else {
      workbookBuildingListener = new EventWorkbookBuilder.SheetRecordCollectingListener(formatListener);
      request.addListenerForAllRecords(workbookBuildingListener);
    }
    try {
      factory.processWorkbookEvents(request, fs);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("excel 文件读取错误,请检查！");
    }
    //返回该excel文件的总行数，不包括首列和空行
    return totalRows;
  }

  /**
   * HSSFListener 监听方法，处理Record
   * 处理每个单元格
   *
   * @param record
   */
  @SuppressWarnings("unchecked")
  @Override
  public void processRecord(Record record) {
    int thisRow = -1;
    int thisColumn = -1;
    String thisStr = null;
    String value = null;
    switch (record.getSid()) {
      case BoundSheetRecord.sid:
        boundSheetRecords.add(record);
        break;
      case BOFRecord.sid:
        //开始处理每个sheet
        BOFRecord br = (BOFRecord) record;
        if (br.getType() == BOFRecord.TYPE_WORKSHEET) {
          //如果有需要，则建立子工作簿
          if (workbookBuildingListener != null && stubWorkbook == null) {
            stubWorkbook = workbookBuildingListener.getStubHSSFWorkbook();
          }

          if (orderedBSRs == null) {
            orderedBSRs = BoundSheetRecord.orderByBofPosition(boundSheetRecords);
          }
        }
        break;
      case SSTRecord.sid:
        sstRecord = (SSTRecord) record;
        break;
      case BlankRecord.sid:
        //单元格为空白
        BlankRecord brec = (BlankRecord) record;
        thisRow = brec.getRow();
        thisColumn = brec.getColumn();
        thisStr = "";
        cellList.add(thisColumn, thisStr);
        break;
      case BoolErrRecord.sid:
        //单元格为布尔类型
        BoolErrRecord berec = (BoolErrRecord) record;
        thisRow = berec.getRow();
        thisColumn = berec.getColumn();
        thisStr = String.valueOf(berec.getBooleanValue());
        cellList.add(thisColumn, thisStr);
        //如果里面某个单元格含有值，则标识该行不为空行
        checkRowIsNull(thisStr);
        break;
      case FormulaRecord.sid:
        //单元格为公式类型
        FormulaRecord frec = (FormulaRecord) record;
        thisRow = frec.getRow();
        thisColumn = frec.getColumn();
        if (outputFormulaValues) {
          if (Double.isNaN(frec.getValue())) {
            outputNextStringRecord = true;
            nextRow = frec.getRow();
            nextColumn = frec.getColumn();
          } else {
            thisStr = '"' + HSSFFormulaParser.toFormulaString(stubWorkbook, frec.getParsedExpression()) + '"';
          }
        } else {
          thisStr = '"' + HSSFFormulaParser.toFormulaString(stubWorkbook, frec.getParsedExpression()) + '"';
        }
        cellList.add(thisColumn, thisStr);
        //如果里面某个单元格含有值，则标识该行不为空行
        checkRowIsNull(thisStr);
        break;
      case StringRecord.sid:
        //单元格中公式的字符串
        if (outputNextStringRecord) {
          StringRecord srec = (StringRecord) record;
          thisStr = srec.getString();
          thisRow = nextRow;
          thisColumn = nextColumn;
          outputNextStringRecord = false;
        }
        break;
      case LabelRecord.sid:
        LabelRecord lrec = (LabelRecord) record;
        curRow = thisRow = lrec.getRow();
        thisColumn = lrec.getColumn();
        value = lrec.getValue().trim();
        if (StringUtils.isBlank(value)) {
          value = "";
        }
        cellList.add(thisColumn, value);
        //如果里面某个单元格含有值，则标识该行不为空行
        checkRowIsNull(value);
        break;
      case LabelSSTRecord.sid:
        //单元格为字符串类型
        LabelSSTRecord lsrec = (LabelSSTRecord) record;
        curRow = thisRow = lsrec.getRow();
        thisColumn = lsrec.getColumn();
        if (sstRecord == null) {
          cellList.add(thisColumn, "");
        } else {
          value = sstRecord.getString(lsrec.getSSTIndex()).toString().trim();
          if (StringUtils.isBlank(value)) {
            value = "";
          }
          cellList.add(thisColumn, value);
          //如果里面某个单元格含有值，则标识该行不为空行
          checkRowIsNull(value);
        }
        break;
      case NumberRecord.sid:
        //单元格为数字类型
        NumberRecord numrec = (NumberRecord) record;
        curRow = thisRow = numrec.getRow();
        thisColumn = numrec.getColumn();
        //第一种方式
        //value = formatListener.formatNumberDateCell(numrec).trim();//这个被写死，采用的m/d/yy h:mm格式，不符合要求
        //第二种方式，参照formatNumberDateCell里面的实现方法编写
        Double valueDouble = ((NumberRecord) numrec).getValue();
        String formatString = formatListener.getFormatString(numrec);
        if (formatString.contains("m/d/yy")) {
          formatString = "yyyy-MM-dd hh:mm:ss";
        }
        int formatIndex = formatListener.getFormatIndex(numrec);
        value = formatter.formatRawCellContents(valueDouble, formatIndex, formatString).trim();
        if (StringUtils.isBlank(value)) {
          value = "";
        }
        //向容器加入列值
        cellList.add(thisColumn, value);
        //如果里面某个单元格含有值，则标识该行不为空行
        checkRowIsNull(value);
        break;
      default:
        break;
    }

    //遇到新行的操作
    if (thisRow != -1 && thisRow != lastRowNumber) {
      lastColumnNumber = -1;
    }

    //空值的操作
    if (record instanceof MissingCellDummyRecord) {
      MissingCellDummyRecord mc = (MissingCellDummyRecord) record;
      curRow = thisRow = mc.getRow();
      thisColumn = mc.getColumn();
      cellList.add(thisColumn, "");
    }

    //更新行和列的值
    if (thisRow > -1) {
      lastRowNumber = thisRow;
    }
    if (thisColumn > -1) {
      lastColumnNumber = thisColumn;
    }

    //行结束时的操作
    if (record instanceof LastCellOfRowDummyRecord) {
      if (minColums > 0 && lastColumnNumber == -1) {
        //列值重新置空
        lastColumnNumber = 0;
      }
      lastColumnNumber = -1;

      if (flag && curRow != 0) {
        //该行不为空行且该行不是第一行，发送（第一行为列名，不需要）
        totalRows++;
      }
      //清空容器
      cellList.clear();
      flag = false;
    }
  }

  /**
   * 如果里面某个单元格含有值，则标识该行不为空行
   *
   * @param value
   */
  public void checkRowIsNull(String value) {
    if (StringUtils.isNoneBlank(value)) {
      flag = true;
    }
  }
}
