package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.kuiper.entity.FormDetailsExportBoxEntity;
import com.bizunited.platform.kuiper.starter.repository.FormDetailsExportBoxRepository;
import com.bizunited.platform.kuiper.starter.service.FormDetailsExportBoxService;
import com.bizunited.platform.kuiper.starter.service.FormDetailsExportBoxTaskService;
import com.bizunited.platform.kuiper.starter.service.instances.imports.FormDetailsExportBoxProcess;
import com.bizunited.platform.kuiper.starter.vo.FormDetailsExportBoxVo;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskRejectedException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.transaction.Transactional;
import java.security.Principal;
import java.util.Date;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author Keller
 * @create 2020/8/25
 */
@Service("FormDetailsExportBoxServiceImpl")
public class FormDetailsExportBoxServiceImpl implements FormDetailsExportBoxService {

  private static final Logger LOGGER = LoggerFactory.getLogger(FormDetailsExportBoxServiceImpl.class);

  @Autowired
  private FormDetailsExportBoxRepository formDetailsExportBoxRepository;

  @Autowired
  private VenusFileService venusFileService;

  @Autowired
  private UserService userService;

  @Autowired
  private FormDetailsExportBoxTaskService formDetailsExportBoxTaskService;

  @Autowired
  private PlatformContext platformContext;

  @Override
  public FormDetailsExportBoxEntity export(FormDetailsExportBoxVo formDetailsExportBoxVo, Map<String, Object> params) {

    //参数验证
    this.executeValidation(formDetailsExportBoxVo);

    // 获取当前进程中的用户信息
    Principal principal = SecurityUtils.getPrincipal();
    // 检查操作者信息
    Validate.notNull(principal, "当前操作者信息必须传入");
    String account = principal.getName();
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser, "错误的操作者信息，请检查!!");

    //保存参数与业务数据
    FormDetailsExportBoxEntity entity = new FormDetailsExportBoxEntity();
    entity.setProcessClassName(formDetailsExportBoxVo.getProcessClassName());
    entity.setExportField(formDetailsExportBoxVo.getExportField());
    entity.setExportParam(formDetailsExportBoxVo.getExportParam());
    entity.setTemplateName(formDetailsExportBoxVo.getTemplateName());
    entity.setTemplateVersion(formDetailsExportBoxVo.getTemplateVersion());
    entity.setTemplateCode(formDetailsExportBoxVo.getTemplateCode());
    entity.setListTemplateName(formDetailsExportBoxVo.getListTemplateName());
    entity.setListTemplateVersion(formDetailsExportBoxVo.getListTemplateVersion());
    entity.setListTemplateCode(formDetailsExportBoxVo.getListTemplateCode());
    entity.setDataViewCode(formDetailsExportBoxVo.getDataViewCode());
    entity.setProjectName(platformContext.getAppName());

    // 检验处理器信息
    Validate.notBlank(formDetailsExportBoxVo.getProcessClassName(), "指定的process Class name必须传入，请检查!!");
    Class<?> processClass = null;
    try {
      processClass = ApplicationContextUtils.getApplicationContext().getClassLoader().loadClass(formDetailsExportBoxVo.getProcessClassName());
    } catch (ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("在当前进程的spring ioc容器中，未发现指定的process Class[%s]，请检查!!", formDetailsExportBoxVo.getProcessClassName()));
    }
    Validate.isTrue(FormDetailsExportBoxProcess.class.isAssignableFrom(processClass), "当前给定的processClassName并没有正确实现FormDetailsImportBoxProcess接口，请检查!!");

    // 获得处理类bean
    FormDetailsExportBoxProcess<?> process = (FormDetailsExportBoxProcess<?>) ApplicationContextUtils.getApplicationContext().getBean(processClass);

    //封装基础的参数进入params中
    Validate.notNull(params,"传递参数对象为空，请检查！");
    params.put("account", account);
    params.put("request", getRequest());
    params.put("cookies", getRequest().getCookies());

    //异步执行导出任务
    try {
      formDetailsExportBoxTaskService.exportProcess(entity, process, account, params);
    } catch (TaskRejectedException e) {
      LOGGER.warn(e.getMessage(), e);
      failedExport(entity);
      throw new IllegalArgumentException("超出导出任务并发执行数");
    }

    entity.setExecuteStartTime(new Date());
    entity.setExecuteResult(2);
    entity.setExecutor(account);
    entity.setDelete(false);
    entity.setProjectName(platformContext.getAppName());
    entity = this.formDetailsExportBoxRepository.saveAndFlush(entity);
    return entity;
  }

  /**
   * 导出执行参数验证
   * @param formDetailsExportBoxVo
   */
  private void executeValidation(FormDetailsExportBoxVo formDetailsExportBoxVo){
    Validate.notBlank(formDetailsExportBoxVo.getProcessClassName(), "指定的process Class name必须传入，请检查!!");
  }

  @Transactional(rollbackOn = Exception.class)
  FormDetailsExportBoxEntity failedExport(FormDetailsExportBoxEntity entity) {
    Validate.notBlank(entity.getId(), "导出记录不能为空！");
    entity.setExecuteResult(0);
    entity.setDelete(true);
    entity.setProjectName(platformContext.getAppName());
    return this.formDetailsExportBoxRepository.saveAndFlush(entity);
  }

  @Override
  public FormDetailsExportBoxEntity findById(String id) {
    Optional<FormDetailsExportBoxEntity> op = this.formDetailsExportBoxRepository.findById(id);
    FormDetailsExportBoxEntity current = op.orElse(null);
    Validate.notNull(current, "导出记录不存在");
    Validate.isTrue(SecurityUtils.getUserAccount().equals(current.getExecutor()), "无该数据操作权限！");
    return current;
  }

  @Override
  public Page<FormDetailsExportBoxEntity> queryPage(Pageable pageable) {
    return this.formDetailsExportBoxRepository.queryPageByDelete(false, SecurityUtils.getUserAccount(), platformContext.getAppName(), pageable);
  }

  @Override
  public void deleteById(String id) {
    Validate.notBlank(id, "删除信息主键为空！");
    Optional<FormDetailsExportBoxEntity> op = this.formDetailsExportBoxRepository.findById(id);
    FormDetailsExportBoxEntity current = op.orElse(null);
    Validate.notNull(current, "删除的记录不存在");
    Validate.isTrue(SecurityUtils.getUserAccount().equals(current.getExecutor()), "无该数据操作权限！");
    current.setDelete(true);
    this.formDetailsExportBoxRepository.saveAndFlush(current);
  }

  @Override
  @Transactional(rollbackOn = Exception.class)
  public void deleteAll() {
    this.formDetailsExportBoxRepository.deleteAll(platformContext.getAppName(), SecurityUtils.getUserAccount());
  }

  @Override
  public void cancel(String id) {
    Validate.notBlank(id, "导出记录主键为空！");
    Optional<FormDetailsExportBoxEntity> op = this.formDetailsExportBoxRepository.findById(id);
    FormDetailsExportBoxEntity current = op.orElse(null);
    Validate.notNull(current, "导出记录不存在");
    Validate.isTrue(SecurityUtils.getUserAccount().equals(current.getExecutor()), "无该数据操作权限！");
    current.setExecuteResult(0);//取消状态
    this.formDetailsExportBoxRepository.saveAndFlush(current);
  }

  @Override
  public byte[] download(String id) {
    Validate.notBlank(id, "导出记录主键为空！");
    Optional<FormDetailsExportBoxEntity> op = this.formDetailsExportBoxRepository.findById(id);
    FormDetailsExportBoxEntity current = op.orElse(null);
    Validate.notNull(current, "导出记录不存在");
    return venusFileService.readFileContent(current.getRelativeLocal(), current.getFileName());
  }

  /**
   * 获取系统中的导出处理类
   *
   * @return
   */
  @SuppressWarnings("rawtypes")
  @Override
  public Set<String> findProcessClassNames() {
    Map<String, FormDetailsExportBoxProcess> map = ApplicationContextUtils.getApplicationContext().getBeansOfType(FormDetailsExportBoxProcess.class);
    return map.values().stream().map(process -> {
      //针对代理类的处理
      if (AopUtils.isCglibProxy(process) || AopUtils.isAopProxy(process)) {
        return AopUtils.getTargetClass(process).getName();
      }
      return process.getClass().getName();
    }).collect(Collectors.toSet());
  }

  private HttpServletRequest getRequest() {
    HttpServletRequest request = null;
    ServletRequestAttributes servletRequestAttributes = null;
    RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
    if (requestAttributes != null) {
      servletRequestAttributes = (ServletRequestAttributes) requestAttributes;
      request = servletRequestAttributes.getRequest();
    }
    Validate.notNull(request, "无法获取当前request");
    return request;
  }
}
