package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.kuiper.entity.FormDetailsImportBoxDetailsEntity;
import com.bizunited.platform.kuiper.entity.FormDetailsImportBoxEntity;
import com.bizunited.platform.kuiper.starter.repository.FormDetailsImportBoxDetailsRepository;
import com.bizunited.platform.kuiper.starter.repository.FormDetailsImportBoxRepository;
import com.bizunited.platform.kuiper.starter.service.FormDetailsImportBoxService;
import com.bizunited.platform.kuiper.starter.service.FormDetailsImportBoxTaskService;
import com.bizunited.platform.kuiper.starter.service.instances.imports.FormDetailsImportBoxProcess;
import com.bizunited.platform.kuiper.starter.vo.FormDetailsImportBoxVo;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskRejectedException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.transaction.Transactional;
import java.security.Principal;
import java.util.Date;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 基于表单引擎，列表引擎，对业务数据通过xls/xlsx文件的方式，提供工具箱导入方式默认实现
 *
 * @author Keller
 * @create 2020/8/23
 */
@Service("FormDetailsImportBoxServiceImpl")
public class FormDetailsImportBoxServiceImpl implements FormDetailsImportBoxService {

  private static final Logger LOGGER = LoggerFactory.getLogger(FormDetailsImportBoxServiceImpl.class);

  private static final Integer MAX_DETAILES_RECORD = 50;

  @Autowired
  private FormDetailsImportBoxRepository formDetailsImportBoxRepository;
  @Autowired
  private FormDetailsImportBoxDetailsRepository formDetailsImportBoxDetailsRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private FormDetailsImportBoxTaskService formDetailsImportBoxTaskService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private PlatformContext platformContext;

  /**
   * 创建excel导入记录
   *
   * @param entity
   * @param account
   * @return
   */
  @Transactional(rollbackOn = Exception.class)
  FormDetailsImportBoxEntity startImport(FormDetailsImportBoxEntity entity, String account) {
    Validate.notNull(entity.getCreateTime(), "保存记录时，导入时间不能为空！");
    Validate.notBlank(entity.getRelativeLocal(), "导入文件在本地路径不能为空！");
    Validate.notBlank(entity.getFileName(), "导入文件重命名后的文件名字不能为空！");
    Validate.notNull(entity.getCreator(), "导入文件上传人不能为空！");
    Validate.notNull(entity.getProcessClassName(), "导入处理类不能为空");
    Validate.notNull(entity, "开始导入记录不能为空");
    entity.setExecutor(account);
    entity.setExecuteStartTime(new Date());
    //开始导入状态
    entity.setExecuteResult(1);
    entity.setProjectName(platformContext.getAppName());
    return this.formDetailsImportBoxRepository.saveAndFlush(entity);
  }

  /**
   * 创建开始excel验证信息
   *
   * @param entity
   * @param account
   * @return
   */
  @Transactional(rollbackOn = Exception.class)
  FormDetailsImportBoxEntity startCheck(FormDetailsImportBoxEntity entity, String account) {
    Validate.notNull(entity.getCreateTime(), "保存记录时，导入时间不能为空！");
    Validate.notBlank(entity.getRelativeLocal(), "导入文件在本地路径不能为空！");
    Validate.notBlank(entity.getFileName(), "导入文件重命名后的文件名字不能为空！");
    Validate.notNull(entity.getCreator(), "导入文件上传人不能为空！");
    Validate.notNull(entity.getProcessClassName(), "导入处理类不能为空");
    Validate.notNull(entity, "开始导入记录不能为空");
    entity.setExecutor(account);
    entity.setExecuteStartTime(new Date());
    //开始验证状态
    entity.setExecuteResult(4);
    entity.setProjectName(platformContext.getAppName());
    return this.formDetailsImportBoxRepository.saveAndFlush(entity);
  }

  /**
   * 导入失败操作
   *
   * @param entity
   * @return
   */
  @Transactional(rollbackOn = Exception.class)
  FormDetailsImportBoxEntity failedImport(FormDetailsImportBoxEntity entity) {
    Validate.notBlank(entity.getId(), "导入记录不能为空！");
    entity.setExecuteResult(0);
    entity.setExecuteEndTime(new Date());
    entity.setDelete(true);
    entity.setProjectName(platformContext.getAppName());
    return this.formDetailsImportBoxRepository.saveAndFlush(entity);
  }

  /**
   * 导入成功操作
   *
   * @param entity
   * @return
   */
  @Transactional(rollbackOn = Exception.class)
  FormDetailsImportBoxEntity successImport(FormDetailsImportBoxEntity entity) {
    Validate.notBlank(entity.getId(), "导入记录不能为空！");
    entity.setExecuteResult(2);
    entity.setExecuteEndTime(new Date());
    entity.setDelete(true);
    entity.setProjectName(platformContext.getAppName());
    return this.formDetailsImportBoxRepository.saveAndFlush(entity);
  }

  /**
   * 创建导入数据信息
   *
   * @param formDetailsImportBoxVo
   * @return
   */
  private FormDetailsImportBoxEntity createImportData(FormDetailsImportBoxVo formDetailsImportBoxVo, Map<String, Object> params) {
    // 初始化导入记录数据
    FormDetailsImportBoxEntity entity = new FormDetailsImportBoxEntity();
    entity.setCreateTime(new Date());
    entity.setCreator(SecurityUtils.getUserAccount());
    entity.setOriginalFileName(formDetailsImportBoxVo.getOriginalFileName());
    entity.setDelete(false);
    //导入excel以及模板信息
    entity.setFileName(formDetailsImportBoxVo.getFileName());
    entity.setRelativeLocal(formDetailsImportBoxVo.getRelativePath());
    entity.setFileSize(formDetailsImportBoxVo.getFileSize());
    entity.setTemplateName(formDetailsImportBoxVo.getTemplateName());
    entity.setTemplateCode(formDetailsImportBoxVo.getTemplateCode());
    entity.setTemplateVersion(formDetailsImportBoxVo.getTemplateVersion());
    entity.setProcessClassName(formDetailsImportBoxVo.getProcessClassName());
    entity.setBtnCode(formDetailsImportBoxVo.getBtnCode());
    entity.setListTemplateName(formDetailsImportBoxVo.getListTemplateName());
    entity.setListTemplateCode(formDetailsImportBoxVo.getListTemplateCode());
    entity.setListTemplateVersion(formDetailsImportBoxVo.getListTemplateVersion());
    entity.setTemplateFileName(formDetailsImportBoxVo.getTemplateFileName());
    entity.setTemplateFileRelativeLocal(formDetailsImportBoxVo.getTemplateFileRelativePath());
    entity.setProjectName(platformContext.getAppName());
    if (params != null && !params.isEmpty()) {
      entity.setParams(JsonUtils.obj2JsonString(params));
    }
    return entity;
  }

  @SuppressWarnings("unchecked")
  @Override
  public FormDetailsImportBoxEntity execute(FormDetailsImportBoxVo formDetailsImportBoxVo, Map<String, Object> params) {
    // 参数效验
    this.executeValidation(formDetailsImportBoxVo);

    // 获取当前进程中的用户信息
    Principal principal = SecurityUtils.getPrincipal();
    // 检查操作者信息
    Validate.notNull(principal, "当前操作者信息必须传入");
    String account = principal.getName();
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser, "错误的操作者信息，请检查!!");

    Class<?> processClass = null;
    try {
      processClass = ApplicationContextUtils.getApplicationContext().getClassLoader().loadClass(formDetailsImportBoxVo.getProcessClassName());
    } catch (NullPointerException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("在当前进程的spring ioc容器中，未发现指定的process Class[%s]，请检查!!", formDetailsImportBoxVo.getProcessClassName()));
    }
    Validate.notNull(processClass, "导入逻辑处理类为空，请检查！");
    Validate.isTrue(FormDetailsImportBoxProcess.class.isAssignableFrom(processClass), "当前给定的processClassName并没有正确实现FormDetailsImportBoxProcess接口，请检查!!");

    // 获得处理类bean
    FormDetailsImportBoxProcess<Object, Object> process = (FormDetailsImportBoxProcess<Object, Object>) ApplicationContextUtils.getApplicationContext().getBean(processClass);

    FormDetailsImportBoxEntity entity = this.createImportData(formDetailsImportBoxVo, params);
    //更新记录表开始导入工作
    this.startImport(entity, account);
    //完成基础信息检查后进行异步数据导入工作

    //封装基础的参数进入params中
    Validate.notNull(params, "传递参数对象为空，请检查！");
    params.put("account", account);
    params.put("request", getRequest());
    params.put("cookies", getRequest().getCookies());

    try {
      this.formDetailsImportBoxTaskService.importProcess(entity, process, account, params);
    } catch (TaskRejectedException e) {
      LOGGER.warn(e.getMessage(), e);
      failedImport(entity);
      throw new IllegalArgumentException("超出导入任务并发执行数");
    }
    return entity;
  }

  @SuppressWarnings("unchecked")
  @Override
  public FormDetailsImportBoxEntity excelCheck(FormDetailsImportBoxVo formDetailsImportBoxVo, Map<String, Object> params) {
    /*
     * 导入excel验证
     * 1、验证参数
     * 2、验证当前用户信息
     * 3、创建导入记录信息
     * 4、调用异步读取excel并调用业务扩展验证数据方法
     */
    //验证参数
    this.executeValidation(formDetailsImportBoxVo);
    // 获取当前进程中的用户信息
    Principal principal = SecurityUtils.getPrincipal();
    // 检查操作者信息
    Validate.notNull(principal, "当前操作者信息必须传入");
    String account = principal.getName();
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser, "错误的操作者信息，请检查!!");

    Class<?> processClass = null;
    try {
      processClass = ApplicationContextUtils.getApplicationContext().getClassLoader().loadClass(formDetailsImportBoxVo.getProcessClassName());
    } catch (NullPointerException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("在当前进程的spring ioc容器中，未发现指定的process Class[%s]，请检查!!", formDetailsImportBoxVo.getProcessClassName()));
    }
    Validate.notNull(processClass, "导入逻辑处理类为空，请检查！");
    Validate.isTrue(FormDetailsImportBoxProcess.class.isAssignableFrom(processClass), "当前给定的processClassName并没有正确实现FormDetailsImportBoxProcess接口，请检查!!");

    // 获得处理类bean
    FormDetailsImportBoxProcess<Object, Object> process = (FormDetailsImportBoxProcess<Object, Object>) ApplicationContextUtils.getApplicationContext().getBean(processClass);

    // 初始化导入记录数据
    FormDetailsImportBoxEntity entity = this.createImportData(formDetailsImportBoxVo, params);

    //更新记录excel验证工作
    this.startCheck(entity, account);

    //封装基础的参数进入params中
    Validate.notNull(params, "传递参数对象为空，请检查！");
    params.put("account", account);
    params.put("request", getRequest());
    params.put("id", entity.getId());
    params.put("cookies", getRequest().getCookies());

    try {
      this.formDetailsImportBoxTaskService.checkProcess(entity, process, account, params);
    } catch (TaskRejectedException e) {
      LOGGER.warn(e.getMessage(), e);
      failedImport(entity);
      throw new IllegalArgumentException("超出导入任务并发执行数", e);
    }
    return entity;
  }

  @SuppressWarnings("unchecked")
  @Override
  public FormDetailsImportBoxEntity excelImport(String id, Map<String, Object> params) {
    FormDetailsImportBoxEntity entity = this.formDetailsImportBoxRepository.findById(id).orElse(null);
    Validate.notNull(entity, "导入excel数据编号错误，请检查！");
    Validate.isTrue(entity.getExecuteResult() == 5, "当前导入记录状态错误，请检查！");
    // 获取当前进程中的用户信息
    Principal principal = SecurityUtils.getPrincipal();
    // 检查操作者信息
    Validate.notNull(principal, "当前操作者信息必须传入");
    String account = principal.getName();
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser, "错误的操作者信息，请检查!!");
    //开始导入
    this.startImport(entity, account);

    Class<?> processClass = null;
    try {
      processClass = ApplicationContextUtils.getApplicationContext().getClassLoader().loadClass(entity.getProcessClassName());
    } catch (NullPointerException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("在当前进程的spring ioc容器中，未发现指定的process Class[%s]，请检查!!", entity.getProcessClassName()));
    }
    Validate.notNull(processClass, "导入逻辑处理类为空，请检查！");
    Validate.isTrue(FormDetailsImportBoxProcess.class.isAssignableFrom(processClass), "当前给定的processClassName并没有正确实现FormDetailsImportBoxProcess接口，请检查!!");

    // 获得处理类bean
    FormDetailsImportBoxProcess<Object, Object> process = (FormDetailsImportBoxProcess<Object, Object>) ApplicationContextUtils.getApplicationContext().getBean(processClass);
    String paramsJson = entity.getParams();
    if(StringUtils.isNotBlank(paramsJson)){
      Map<String,Object> entityParams = JsonUtils.json2Obj(paramsJson,Map.class);
      params.putAll(entityParams);
    }
    params.put("account", account);
    params.put("request", getRequest());
    params.put("cookies", getRequest().getCookies());
    try {
      this.formDetailsImportBoxTaskService.asyncImport(entity, process, account, params);
    } catch (Exception e) {
      //记录导入失败
      this.failedImport(entity);
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("执行数据导入错误,错误信息[%s]，请检查!!", e.getMessage()));
    }
    //记录导入成功
    this.successImport(entity);
    return entity;
  }

  /**
   * 导入执行参数验证
   *
   * @param formDetailsImportBoxVo
   */
  private void executeValidation(FormDetailsImportBoxVo formDetailsImportBoxVo) {
    Validate.notBlank(formDetailsImportBoxVo.getOriginalFileName(), "导入文件名原名称为空！");
    Validate.notBlank(formDetailsImportBoxVo.getFileName(), "导入文件名为空！");
    Validate.notBlank(formDetailsImportBoxVo.getRelativePath(), "导入文件服务器地址空！");
    Validate.notBlank(formDetailsImportBoxVo.getProcessClassName(), "指定的process Class name必须传入，请检查!!");
    Validate.notBlank(formDetailsImportBoxVo.getTemplateFileName(), "导入模板文件名为空！");
    Validate.notBlank(formDetailsImportBoxVo.getTemplateFileRelativePath(), "导入模板文件相对路径为空！");
  }

  @Override
  public Page<FormDetailsImportBoxEntity> queryPage(Pageable pageable) {
    return this.formDetailsImportBoxRepository.queryPageByDelete(false, SecurityUtils.getUserAccount(), platformContext.getAppName(), pageable);
  }

  @Override
  public Page<FormDetailsImportBoxEntity> queryPageByExecuteResult(Pageable pageable, Integer[] results) {
    Validate.isTrue(ArrayUtils.isNotEmpty(results), "导入记录查询状态值为空，请检查！");
    return this.formDetailsImportBoxRepository.queryPageByDeleteAndExecuteResult(results, false, SecurityUtils.getUserAccount(), platformContext.getAppName(), pageable);
  }

  @Override
  public FormDetailsImportBoxEntity findById(String id) {
    Optional<FormDetailsImportBoxEntity> op = this.formDetailsImportBoxRepository.findById(id);
    FormDetailsImportBoxEntity current = op.orElse(null);
    Validate.notNull(current, "未查询到导入记录");
    Validate.isTrue(SecurityUtils.getUserAccount().equals(current.getExecutor()), "无该数据操作权限！");
    return current;
  }

  @Override
  public void deleteById(String id) {
    Validate.notBlank(id, "删除信息主键为空！");
    Optional<FormDetailsImportBoxEntity> op = this.formDetailsImportBoxRepository.findById(id);
    FormDetailsImportBoxEntity current = op.orElse(null);
    Validate.notNull(current, "删除的记录不存在");
    Validate.isTrue(SecurityUtils.getUserAccount().equals(current.getExecutor()), "无该数据操作权限！");
    current.setDelete(true);
    this.formDetailsImportBoxRepository.saveAndFlush(current);
  }

  @Override
  @Transactional(rollbackOn = Exception.class)
  public void deleteAll() {
    this.formDetailsImportBoxRepository.deleteAll(platformContext.getAppName(), SecurityUtils.getUserAccount());
  }

  @Override
  public void cancel(String id) {
    Validate.notBlank(id, "导入记录主键为空！");
    Optional<FormDetailsImportBoxEntity> op = this.formDetailsImportBoxRepository.findById(id);
    FormDetailsImportBoxEntity current = op.orElse(null);
    Validate.notNull(current, "导入记录不存在");
    Validate.isTrue(SecurityUtils.getUserAccount().equals(current.getExecutor()), "无该数据操作权限！");
    //取消状态
    current.setExecuteResult(3);
    this.formDetailsImportBoxRepository.saveAndFlush(current);
  }

  @Override
  public Page<FormDetailsImportBoxDetailsEntity> queryPageById(String id) {
    Validate.notBlank(id, "导入记录主键为空！");
    Pageable pageable = PageRequest.of(0, MAX_DETAILES_RECORD);
    Page<FormDetailsImportBoxDetailsEntity> page = formDetailsImportBoxDetailsRepository.findByFormDetailsImportBoxId(id, platformContext.getAppName(), pageable);
    return page;
  }

  @Override
  public byte[] exportErrorFile(String id) {
    Validate.notBlank(id, "导入记录主键为空！");
    Optional<FormDetailsImportBoxEntity> op = this.formDetailsImportBoxRepository.findById(id);
    FormDetailsImportBoxEntity entity = op.orElse(null);
    Validate.notNull(entity, "无导入错误文件");
    return venusFileService.readFileContent(entity.getErrorFileRelativeLocal(), entity.getErrorFileName());
  }

  /**
   * 获取系统中的导入处理类
   *
   * @return
   */
  @SuppressWarnings("rawtypes")
  @Override
  public Set<String> findProcessClassNames() {
    Map<String, FormDetailsImportBoxProcess> map = ApplicationContextUtils.getApplicationContext().getBeansOfType(FormDetailsImportBoxProcess.class);
    return map.values().stream().map(process -> {
      if (AopUtils.isCglibProxy(process) || AopUtils.isAopProxy(process)) {
        //针对代理类的处理
        return AopUtils.getTargetClass(process).getName();
      }
      return process.getClass().getName();
    }).collect(Collectors.toSet());
  }

  private HttpServletRequest getRequest() {
    HttpServletRequest request = null;
    ServletRequestAttributes servletRequestAttributes = null;
    RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
    if (requestAttributes != null) {
      servletRequestAttributes = (ServletRequestAttributes) requestAttributes;
      request = servletRequestAttributes.getRequest();
    }
    Validate.notNull(request, "无法获取当前request");
    return request;
  }
}
