package com.bizunited.platform.kuiper.service;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.entity.InstanceViewEntity;
import java.security.Principal;
import java.util.Date;
import java.util.Map;

import com.bizunited.platform.kuiper.entity.TemplateEntity;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

/**
 * 表单引擎实例相关服务
 * @author yinwenjie
 */
public interface InstanceService {
  /**
   * 只需要传入一个表单模板编号（code），表单引擎就可以根据这个code寻找到相关的最新版本，并基于这个最新版本的模板自动创建表单实例
   * @param templateCode 指定的模板业务编号（code）
   * @param taskCode 指定的当前create活动对应的唯一外部业务编号（可能传入）
   * @param principal 当前操作者
   */
  public InstanceEntity create(String templateCode, String taskCode , Principal principal);
  /**
   * 创建一个新的表单实例，以及表单实例中第一个名叫create的活动实例。并且这个名叫create的活动实例的taskCode按照传入的taskCode为准
   * @param instance 定的实例基本信息——主要要有其中关联的模板数据层编号信息 
   * @param taskCode 指定的当前create活动对应的唯一外部业务编号（可能传入）
   * @param principal 当前操作者
   * @return
   */
  public InstanceEntity create(InstanceEntity instance , String taskCode, Principal principal);

  /**
   * 只需要传入一个表单模板的业务编号（code）和版本号（cversion），表单引擎就可以根据这个code和cversion寻找到相关的模版，并基于这个模板自动创建表单实例
   * @param templateCode 指定的模板业务编号（code）
   * @param cversion 指定的模板版本号（cversion）
   * @param taskCode 指定的当前create活动对应的唯一外部业务编号（可能传入）
   * @param principal 当前操作者
   * @return
   */
  InstanceEntity create(String templateCode, String cversion, String taskCode, Principal principal);
  /**
   * 传入一个表单模板的业务编号（code）
   * @param templateCode 指定的模板业务编号（code）
   * @param taskCode 指定的可能传入的，将要创建的create名称的活动实例的唯一外部业务编号
   * @param principal 当前操作者
   * @param serviceName 服务源的唯一名称
   * @param formData json形式的业务数据结构
   * @param params 外参，http请求的url后边携带的？xxx=yyy&aaa=bbb 这类参数（没有就可以不传）
   * @return 当前创建的表单实例以及相关的名字为“create”的活动实例将会被返回给调用者
   */
  public InstanceEntity create(String templateCode, String taskCode , Principal principal , String serviceName , JSONObject formData , Map<String , Object> params);
  /**
   * 根据表单实例的数据层编号，查询这个实例的详情——包括实例的直接关联信息
   * @param instanceId 指定的表单实例编号
   * @return
   */
  public InstanceEntity findDetailsById(String instanceId);
  /**
   * 按条件搜索，目前支持模板code、模板版本信息、实例域信息、实例项目名，不支持模糊查询
   * @param code 模板code
   * @param instanceId 实例ID
   * @param cversion 可能的模板版本信息
   * @param domain 可能的业务域查询条件
   * @param templateName 模板名称
   * @param projectName 可能的项目名查询条件
   * @param beginCreateTime 可能的实例创建时间（起）
   * @param endCreateTime 可能的实例创建时间（止）
   * @param userName 实例创建者真实姓名
   * @param pageable 分页设置信息
   * @return
   */
  public Page<InstanceViewEntity> findByConditions(String code, String instanceId, String cversion, String domain,String templateName, String projectName,
    Date beginCreateTime , Date endCreateTime , String userName ,  Pageable pageable); 
  
  /**
   * 根据实例id，查询实例信息
   * @param id
   * @return
   */
  public InstanceEntity findById(String id);
  /**
   * 根据指定的表单模板编号信息，查询这个表单模板下涉及的表单实例数量
   * @param templateId 指定的表单模板编号信息
   * @return
   */
  public int countByTemplateId(String templateId);

  /**
   * 根据模版ID分页查询实例
   * @param templateId
   * @param pageable
   * @return
   */
  Page<InstanceEntity> findByTemplateId(String templateId, Pageable pageable);

  /**
   * 更新模版实例为当前版本模版
   * @param template
   */
  void updateTemplate(TemplateEntity template);
}
