package com.bizunited.platform.kuiper.service;

import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.common.model.MigrateImportModel;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import com.bizunited.platform.kuiper.entity.TemplateEntity;

public interface TemplateService {
  
  /**
   * 该接口按照表单模板的数据库编号，返回表单模板的基本信息（注意只有基本信息，不包括任何的关联信息）
   * @param id 指定的表单模板数据层编号
   */
  public TemplateEntity findById(String id);
  
  /**
   * 该接口按照表单模板的业务编号，返回表单模板的基本信息（注意只有基本信息，不包括任何的关联信息）
   * @param code 指定的表单模板业务编号
   * @return 可能存在多个则返回集合，其它情况
   */
  public List<TemplateEntity> findByCode(String code);
  /**
   * 查询指定模板的默认版本信息（只有基本信息，没有关联信息）
   * @param code 指定的模板业务编号
   * @return
   */
  public TemplateEntity findByDefaultVersion(String code);
  /**
   * 该接口查询指定模板下的属性结构信息。注意，只包括属性结构，也就是主模型上的一般属性、关联属性、分组属性和明细编辑属性。
   * 如果要获取模板下的事件、布局等信息，请调用其它相关接口
   * @param id 指定的表单模板的数据层编号
   * @return 如果查询到，则只可能有一个对象；其它情况返回null
   */
  public TemplateEntity findDetailsById(String id);
  /**
   * 按照表单样式所使用的
   * @param formStyle 当前样式说明信息
   * @return 返回查询到的符合条件的信息，其它情况返回null
   */
  public List<TemplateEntity> findByFormStyle(String formStyle);
  
  /**
   * 按条件搜索，目前支持的查询条件是：表单名、业务领域、项目名、模板版本号、模板code、模板状态tstatus（1正常；0禁止/废除）
   * @param conditions 条件信息
   * @param pageable 分页设置信息
   * @return
   */
  public Page<TemplateEntity> findByConditions(Map<String, Object> conditions, Pageable pageable);
  /**
   * 查询一批指定的id编号的表单模板信息在系统中真实存在的数量
   * @param ids
   * @return
   */
  public int countByIds(String[] ids);
  /**
   * 保存一个新的表单模板，因为表单模板有初始化过程，所以这里的保存功能肯定都是修改功能。<p>
   * 首先就是说表单模板对应的模型是不能改变的，所以表单模板保存功能对表单模型无效；另外表单模板保存功能主要就是对数据源、事件、布局、可见性信息进行修改<p>
   * 另外表单模板的保存功能会自动判断数据源、事件、布局、可见性的信息变化，自行进行这些数据项的新增、修改、删除操作。
   * @param template 新的表单模板信息，必须至少包括、数据源、事件、布局、可见性等信息，另外注意，在添加是表单模板的id（主键信息）必须传入
   * @return 当创建成功后，表单模板的基本信息将被返回（注意，不包括任何关联信息），
   */
  public TemplateEntity update(TemplateEntity template);
  /**
   * 该方法设定指定的表单模板版本为这个表单code下唯一默认的表单模板。<br>
   * 在表单模板的信息中，相同code下只有一个指定的版本可以是默认的版本号（也就是defaultVersion属性为true）
   * @param templateId 指定的版本信息
   */
  public void updateDefaultVersion(String templateId);
  /**
   * 更新模板状态信息，如果当前模板被禁用，那么就不能继续使用该模板创建表单实例了；<br>
   * 另一个方面是，某一个模板编号下的默认版本不允许被禁用
   * @param templateId 指定的模板编号
   * @param tstatus 状态符1：正常；0：禁用/废除
   */
  public void updateStatusById(String templateId , Integer tstatus);
  /**
   *
   * 根据表单ID，获取表单属性的特定JSON结构（备注：此接口针对远端服务源-出参快速设定）
   * @param templateEntityId
   * @return
   */
  JSONArray importTemplate(String templateEntityId);

  /**
   * 根据模版编号和版本获取表单模版
   * @param templateCode
   * @param cversion
   * @return
   */
  TemplateEntity findByCodeAndCversion(String templateCode, String cversion);

  /**
   * 根据模版编码和版本号查询模版详情
   * @param code
   * @param version
   * @return
   */
  TemplateEntity findDetailsByCodeAndVersion(String code, String version);

  /**
   * 根据模版编码查询默认版本号模版详情
   * @param code
   * @return
   */
  TemplateEntity findDetailsByCodeAndDefaultVersion(String code);

  /**
   * 根据条件查询所有模版
   * @param condition
   * @return
   */
  List<TemplateEntity> findAllByConditions(TemplateEntity condition);

  /**
   * 根据ID集合查询模版
   * @param ids
   * @return
   */
  List<TemplateEntity> findDetailsByIds(String[] ids);

  /**
   * 导入表单模版
   * @param importModel
   */
  void importData(MigrateImportModel importModel);
}