package com.bizunited.platform.kuiper.starter.common.excel;

import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelMigrateException;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelWraperException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.FileMagic;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * excel导出封装
 *
 * @author Keller
 * @create 2020/8/24
 */
public class ExcelExportWrapper {
  public static final Logger LOGGER = LoggerFactory.getLogger(ExcelExportWrapper.class);

  /**
   * 日期时间格式
   */
  private static final String DATE_TIME_FORMAT = "yyyy-MM-dd HH:mm:ss";

  /**
   * 日期格式
   */
  private static final String DATE_FORMAT = "yyyy-MM-dd";

  /**
   * SXSSFWorkbook 模式下缓存在内存中的行数，超过该值flush到硬盘上
   */
  private static final Integer ROW_KEEP_INMEMORY = 100;

  /**
   * excel列默认宽度
   */
  private static final Integer DEFAULT_COLUMN_WIDTH = 20;

  private static final String DEFAULT_CHARACTER = "utf-8";

  /**
   * poi excel workbook
   */
  private Workbook wb;
  /**
   * 默认单元格样式信息
   */
  private CellStyle defaultCellStyle = null;
  /**
   * 单元格样式信息
   */
  private Map<String, CellStyle> cellStyleCache = new HashMap<>();
  /**
   * 日期格式缓存
   */
  private Map<String, Short> formatCache = new HashMap<>();
  /**
   * 自定义列宽
   */
  private Map<Sheet, Map<Integer, Integer>> columnWidth = new HashMap<>();
  /**
   * 时间格式
   */
  private DataFormat dataFormat;
  /**
   * 用于自动列宽
   */
  private boolean autoColumnWidth = true;
  /**
   * 时间格式缓存
   */
  private Map<String, String> dataFormatCache = new HashMap<>();

  /**
   * 通过输入流初始化
   *
   * @param inp
   */
  public ExcelExportWrapper(InputStream inp) {
    try (InputStream is = FileMagic.prepareToCheckMagic(inp)) {
      FileMagic fm = FileMagic.valueOf(is);
      switch (fm) {
        case OLE2:
          wb = new HSSFWorkbook(is);
          break;
        case OOXML:
          XSSFWorkbook xssfWorkbook = new XSSFWorkbook(is);
          wb = new SXSSFWorkbook(xssfWorkbook, ROW_KEEP_INMEMORY);
          ((SXSSFWorkbook) wb).setCompressTempFiles(true);
          break;
        default:
          throw new ExcelWraperException("Excel文件格式错误");
      }
    } catch (IOException e) {
      throw new ExcelWraperException("Excel文件读取错误");
    }
    init(wb);
  }

  /**
   * 通过外部传入workbook初始化
   *
   * @param wb
   */
  public ExcelExportWrapper(Workbook wb) {
    init(wb);
  }

  /**
   * excel初始化
   *
   * @param wb
   */
  private void init(Workbook wb) {
        /*
        1、初始化wb
        2、设置常用日期类型缓存
        3、设置默认单元格样式
         */
    try {
      if (wb != null) {
        this.wb = wb;
        dataFormat = wb.createDataFormat();
        formatCache.put(DATE_TIME_FORMAT, dataFormat.getFormat(DATE_TIME_FORMAT));
        formatCache.put(DATE_FORMAT, dataFormat.getFormat(DATE_FORMAT));
        formatCache.put("@", dataFormat.getFormat("@"));

        CellStyle defauleS = wb.createCellStyle();
        defauleS.setVerticalAlignment(VerticalAlignment.CENTER);
        defauleS.setWrapText(true);
        setDefaultCellStyle(defauleS);
        setDataFormat("java.util.Date", DATE_TIME_FORMAT);
        setDataFormat("java.lang.CharSequence", "@");
      }
    } catch (Exception e) {
      throw new ExcelWraperException("创建excel失败,请检查文件格式和内容。");
    }
  }

  public Workbook getWorkbook() {
    return wb;
  }

  /***
   * 创建sheet
   *
   * @param sheetName sheet名称
   * @return
   */
  public Sheet buildSheet(String sheetName) {
    return buildSheet(sheetName, DEFAULT_COLUMN_WIDTH);
  }

  /***
   * 创建sheet
   *
   * @param sheetName          sheet名称
   * @param defaultColumnWidth 默认列宽
   * @return
   */
  public Sheet buildSheet(String sheetName, int defaultColumnWidth) {
    // 创建sheet
    if (StringUtils.isBlank(sheetName)) {
      sheetName = UUID.randomUUID().toString();
    }
    sheetName = sheetName.replaceAll("[\u0000\u0003:\\\\*\\?/\\[\\]<>\"\\|]", "");
    sheetName = sheetName.length() > 31 ? sheetName.substring(0, 28) + "..." : sheetName;
    Sheet sheet = getWorkbook().createSheet(sheetName);
    sheet.setDefaultColumnWidth(defaultColumnWidth);
    return sheet;
  }

  /**
   * 获取已经存在的sheet
   *
   * @param name sheet名称
   * @return
   */
  public Sheet getSheet(String name) {
    Validate.notBlank(name, "sheet名称为空");
    return getWorkbook().getSheet(name);
  }

  /***
   * 获取已经存在的sheet
   *
   * @param sheetIndex sheet索引
   * @return
   */
  public Sheet getSheet(int sheetIndex) {
    Validate.isTrue(sheetIndex > -1, "sheet索引小于0");
    Sheet sheet = getWorkbook().getSheetAt(sheetIndex);
    return sheet;
  }

  /**
   * 从第一行第一列开始绘制数据
   *
   * @param sheet sheet对象
   * @param list  数据值
   */
  public void paint(Sheet sheet, Object[] list) {
    paint(sheet, 0, 0, list, null, null);
  }

  /**
   * 从第一行第一列开始绘制数据
   *
   * @param sheet     sheet对象
   * @param list      数据值
   * @param cellstyle 单元格样式
   */
  public void paint(Sheet sheet, Object[] list, CellStyle cellstyle) {
    paint(sheet, 0, 0, list, cellstyle, null);
  }

  /**
   * 从某个位置开始绘制数据，行列起始位置索引为0
   *
   * @param sheet       sheet对象
   * @param rowIndex    从那一行开始
   * @param columnIndex 从那一列开始
   * @param list        数据值
   */
  public void paint(Sheet sheet, int rowIndex, int columnIndex, Object[] list) {
    paint(sheet, rowIndex, columnIndex, list, null, null);
  }

  /**
   * 从某个位置开始绘制数据，行列起始位置索引为0
   *
   * @param sheetIndex  sheet对象
   * @param rowIndex    从那一行开始
   * @param columnIndex 从那一列开始
   * @param list        数据值
   */
  public void paint(int sheetIndex, int rowIndex, int columnIndex, Object[] list) {
    paint(getSheet(sheetIndex), rowIndex, columnIndex, list, null, null);
  }

  /**
   * 从某个位置开始绘制数据，行列起始位置索引为0
   *
   * @param sheet       sheet对象
   * @param rowIndex    从那一行开始
   * @param columnIndex 从那一列开始
   * @param list        数据值
   * @param cellstyle   单元格样式
   */
  public void paint(Sheet sheet, int rowIndex, int columnIndex, Object[] list, CellStyle cellstyle) {
    paint(sheet, rowIndex, columnIndex, list, cellstyle, null);
  }

  /**
   * 从某个位置开始绘制数据，行列起始位置索引为0
   *
   * @param sheet       sheet对象
   * @param rowIndex    从那一行开始
   * @param columnIndex 从那一列开始
   * @param list        数据值
   * @param cellstyle   单元格样式
   * @param formats     值展示的格式
   */
  public void paint(Sheet sheet, int rowIndex, int columnIndex, Object[] list, CellStyle cellstyle, String[] formats) {
    Row row = sheet.getRow(rowIndex);
    if (row == null) {
      row = sheet.createRow(rowIndex);
    }
    if (list != null && list.length != 0) {
      for (int i = 0; i < list.length; i++) {
        Object v = list[i];
        Cell cell = row.getCell(i + columnIndex);
        if (cell == null) {
          cell = row.createCell(i + columnIndex);
        }
        paintCell(cell, v, cellstyle, formats != null && formats.length > i && StringUtils.isNotBlank(formats[i]) ? formats[i] : "");
      }
    }
  }

  /**
   * 从某个位置开始绘制数据，行列起始位置索引为0
   *
   * @param cell      单元格
   * @param val       值
   * @param cellstyle 单元格样式
   * @param format    值展示的格式
   */
  public void paintCell(Cell cell, Object val, CellStyle cellstyle, String format) {
    CellStyle deCellStyle = cellstyle;
    if (deCellStyle == null) {
      deCellStyle = defaultCellStyle;
    }
    if (val != null) {
      if (StringUtils.isNotBlank(format)) {
        deCellStyle = getFormatCellStyle(deCellStyle, format);
      } else if (Date.class.isAssignableFrom(val.getClass())) {
        deCellStyle = getFormatCellStyle(deCellStyle, dataFormatCache.get("java.lang.Date"));
      } else if (CharSequence.class.isAssignableFrom(val.getClass())) {
        deCellStyle = getFormatCellStyle(deCellStyle, dataFormatCache.get("java.lang.CharSequence"));
      }
    }
    cell.setCellStyle(deCellStyle);
    fillCellValue(cell, val);
  }

  /**
   * 获取公司单元格样式
   *
   * @param cellstyle
   * @param format
   * @return
   */
  private CellStyle getFormatCellStyle(CellStyle cellstyle, String format) {
    CellStyle deCellStyle = cellstyle;
    if (StringUtils.isNotBlank(format)) {
      String key = StringUtils.join(deCellStyle.toString(), "_", format);
      if (cellStyleCache.containsKey(key)) {
        deCellStyle = cellStyleCache.get(key);
      } else {
        deCellStyle = buildFormtCellStyle(deCellStyle, format);
        cellStyleCache.put(key, deCellStyle);
      }
    }
    return deCellStyle;
  }

  /**
   * 构建公式单元格样式
   *
   * @param copy
   * @param format
   * @return
   */
  private CellStyle buildFormtCellStyle(CellStyle copy, String format) {
    CellStyle style = copy;
    short desf = 0;
    if (formatCache.containsKey(format)) {
      desf = formatCache.get(format);
    } else {
      desf = dataFormat.getFormat(format);
      formatCache.put(format, desf);
    }
    if (desf != 0) {
      style = wb.createCellStyle();
      style.cloneStyleFrom(copy);
      style.setDataFormat(desf);
    }
    return style;
  }

  /**
   * 为指定区域的单元格绘制样式
   *
   * @param sheet     sheet对象
   * @param region    区域
   * @param cellstyle 单元格样式
   */
  public void paintStyle(Sheet sheet, CellRegion region, CellStyle cellstyle) {
    if (region != null && sheet != null && cellstyle != null) {
      for (int i = region.getFirstRow(); i <= region.getFootRow(); i++) {
        Row row = sheet.getRow(i);
        //验证row不为空
        Validate.notNull(row, String.format("行%d不存在，无法绘制样式。", i));

        for (int j = region.getFirstCol(); j <= region.getFootCol(); j++) {
          Cell cell = row.getCell(j);
          //验证cell不为空
          Validate.notNull(cell, String.format("单元格(%d,%d)不存在，无法绘制样式。", i, j));
          cell.setCellStyle(cellstyle);
        }
      }
    }
  }

  /**
   * 合并区域
   *
   * @param list  需要合并的所有区域
   * @param sheet sheet对象
   */
  public void mergeRegion(List<CellRegion> list, Sheet sheet) {
    if (!list.isEmpty()) {
      for (CellRegion cellRegion : list) {
        sheet.addMergedRegion(new CellRangeAddress(cellRegion.getFirstRow(), cellRegion.getFootRow(), cellRegion.getFirstCol(), cellRegion.getFootCol()));
      }
    }
  }

  /**
   * 合并区域
   *
   * @param cellRegion 需要合并的区域
   * @param sheet      sheet对象
   */
  public void mergeRegion(CellRegion cellRegion, Sheet sheet) {
    sheet.addMergedRegion(new CellRangeAddress(cellRegion.getFirstRow(), cellRegion.getFootRow(), cellRegion.getFirstCol(), cellRegion.getFootCol()));
  }

  /***
   * 导出excel
   *
   * @param os 输出流
   */
  public void export(OutputStream os) {
    try {
      exportBeforeHandle();
      wb = getWorkbook();
      wb.write(os);
      if (wb instanceof SXSSFWorkbook) {
        ((SXSSFWorkbook) wb).dispose();
      }
    } catch (IOException e) {
      throw new ExcelMigrateException("excel写入出错");
    }
  }

  /**
   * 非字符串值的单元格赋值
   *
   * @param cell
   * @param o
   */
  private void fillCellValue(Cell cell, Object o) {
    //初始化宽度值
    int length = DEFAULT_COLUMN_WIDTH;
    try {
      if (o != null) {
        if (o instanceof Integer) {
          cell.setCellValue(Integer.parseInt(String.valueOf(o)));
          length = o.toString().getBytes(DEFAULT_CHARACTER).length;
        } else if (o instanceof Float) {
          cell.setCellValue(Float.parseFloat(String.valueOf(o)));
          length = o.toString().getBytes(DEFAULT_CHARACTER).length;
        } else if (o instanceof Double) {
          cell.setCellValue(Double.parseDouble(String.valueOf(o)));
          length = o.toString().getBytes(DEFAULT_CHARACTER).length;
        } else if (o instanceof Long) {
          cell.setCellValue(Long.parseLong(String.valueOf(o)));
          length = o.toString().getBytes(DEFAULT_CHARACTER).length;
        } else if (o instanceof Short) {
          cell.setCellValue(Short.parseShort(String.valueOf(o)));
          length = o.toString().getBytes(DEFAULT_CHARACTER).length;
        } else if (o instanceof Boolean) {
          cell.setCellValue(Boolean.parseBoolean(String.valueOf(o)));
          length = o.toString().getBytes(DEFAULT_CHARACTER).length;
        } else if (o instanceof RichTextString) {
          cell.setCellValue((RichTextString) o);
          length = ((RichTextString) o).getString().getBytes(DEFAULT_CHARACTER).length;
        } else if (Date.class.isAssignableFrom(o.getClass())) {
          cell.setCellValue((Date) o);
        } else if (o instanceof ExcelFormula) {
          cell.setCellFormula(((ExcelFormula) o).getFormula());
          length = 0;
        } else if (o instanceof BigDecimal) {
          cell.setCellValue(((BigDecimal) o).doubleValue());
          length = o.toString().getBytes(DEFAULT_CHARACTER).length;
        } else {
          cell.setCellValue(o.toString());
          length = o.toString().getBytes(DEFAULT_CHARACTER).length;
        }
      } else {
        cell.setCellValue("");
      }
    } catch (UnsupportedEncodingException e) {
      LOGGER.error(e.getMessage(), e);
      cell.setCellValue("");
    }
    setMaxWidth(cell.getSheet(), cell.getColumnIndex(), length);
  }

  /**
   * 默认单元格样式
   *
   * @return
   */
  public CellStyle getDefaultCellStyle() {
    return defaultCellStyle;
  }

  public boolean isAutoColumnWidth() {
    return autoColumnWidth;
  }

  public void setAutoColumnWidth(boolean autoColumnWidth) {
    this.autoColumnWidth = autoColumnWidth;
  }

  /**
   * 设置最大宽度
   *
   * @param sheet
   * @param i
   * @param length
   */
  private void setMaxWidth(Sheet sheet, int i, int length) {
    if (!isAutoColumnWidth()) {
      return;
    }
    Map<Integer, Integer> rowWidth = columnWidth.get(sheet);
    if (rowWidth == null) {
      rowWidth = new HashMap<>(20);
      columnWidth.put(sheet, rowWidth);
    }
    Integer width = rowWidth.get(i);
    if (width != null) {
      if (length > width) {
        rowWidth.put(i, length);
      }
    } else {
      rowWidth.put(i, length);
    }
  }

  /**
   * 完成到出前的自动宽度设置
   */
  private void exportBeforeHandle() {
    if (!isAutoColumnWidth()) {
      return;
    }
    for (Map.Entry<Sheet, Map<Integer, Integer>> e : columnWidth.entrySet()) {
      Sheet sheet = e.getKey();
      Map<Integer, Integer> rowWidth = e.getValue();
      if (rowWidth != null && !rowWidth.isEmpty()) {
        for (Map.Entry<Integer, Integer> row : rowWidth.entrySet()) {
          int deswidth = ((row.getValue() * 7 + 5) / 7 + 1);
          sheet.setColumnWidth(row.getKey(), (deswidth > 255 ? 255 : deswidth) * 256);
        }
      }
    }
  }

  public void setDataFormat(String name, String format) {
    dataFormatCache.put(name, format);
  }

  public void setDefaultCellStyle(CellStyle cellStyle) {
    this.defaultCellStyle = cellStyle;
    getFormatCellStyle(cellStyle, DATE_TIME_FORMAT);
    getFormatCellStyle(cellStyle, DATE_FORMAT);
  }
}
