package com.bizunited.platform.kuiper.starter.common.excel.reader.impl;


import com.bizunited.platform.kuiper.starter.common.enums.ExcelTypeEnum;
import com.bizunited.platform.kuiper.starter.common.excel.IObgainWorkbook;
import com.bizunited.platform.kuiper.starter.common.excel.IRecordInterceptor;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelWraperException;
import com.bizunited.platform.kuiper.starter.common.excel.reader.IExcelReader;
import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.lang3.Validate;
import org.apache.poi.hssf.usermodel.HSSFFormulaEvaluator;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFFormulaEvaluator;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;

/**
 * 基于dom方式读取excel(适合小数据量文件)
 *
 * @author Keller
 * @create 2020/8/24
 */

public class DomExcelReader implements IExcelReader, IObgainWorkbook {

  /**
   * 值转换工具bean
   */
  public final static ConvertUtilsBean convertUtilsBean = new ConvertUtilsBean();

  /**
   * excel格式处理
   */
  FormulaEvaluator eva = null;

  /**
   * excel wb
   */
  private Workbook wb;

  /**
   * 初始化读取器
   *
   * @param fileType 文件类型
   * @param input    输入流
   */
  public DomExcelReader(ExcelTypeEnum fileType, InputStream input) {
      /*
      初始化创建Excel操作Workbook
      1.根据Excel类型 XLS/XLSX 创建不同的Workbook处理类
      2.读取输入流创建Workbook实例
       */
    try {
      if (fileType == ExcelTypeEnum.XLS) {
        wb = new HSSFWorkbook(input);
        eva = new HSSFFormulaEvaluator((HSSFWorkbook) wb);
      } else if (fileType == ExcelTypeEnum.XLSX) {
        wb = new XSSFWorkbook(input);
        eva = new XSSFFormulaEvaluator((XSSFWorkbook) wb);
      } else {
        throw new ExcelWraperException("创建excel失败,请检查文件格式。");
      }
    } catch (IOException e) {
      throw new ExcelWraperException("创建excel失败,请检查文件路径。");
    }
  }

  @Override
  public void readSheet(Integer sheetIndex, int rowBeginIndex, int columnBeginIndex, int columnEndIndex, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    Sheet sheet = wb.getSheetAt(sheetIndex);
    Validate.notNull(sheet, "sheet不存在");
    int lastRow = sheet.getLastRowNum();
    for (int i = rowBeginIndex; i <= lastRow; i++) {
      Object[] os = readRow(sheetIndex, i, columnBeginIndex, columnEndIndex, classs);
      interceptor.handle(os, i + 1);
    }
  }

  @Override
  public void readSheet(Integer sheetIndex, int columnBeginIndex, int columnEndIndex, int[] rowIndexs, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    for (int i : rowIndexs) {
      Object[] os = readRow(sheetIndex, i, columnBeginIndex, columnEndIndex, classs);
      interceptor.handle(os, i);
    }
  }

  /**
   * 根据cell 类型返回对应的值
   */
  private Object getObjectValue(Cell cell) {
    Object r = "";
    if (cell != null) {
      CellType type = cell.getCellType();
      switch (type) {
        case NUMERIC:
          double val = cell.getNumericCellValue();
          long temp = Math.round(val);
          if (val == temp) {
            r = temp;
          } else {
            r = val;
          }
          break;
        case BOOLEAN:
          r = cell.getBooleanCellValue();
          break;
        case FORMULA:
          r = getObjectValue(eva.evaluateInCell(cell));
          break;
        default:
          r = cell.getRichStringCellValue().getString();
          break;
      }
    }
    return r;
  }

  @Override
  public Object[] readRow(Integer sheetIndex, int rowIndex, int columnBeginIndex, int columnEndIndex, Class<?>[] classs) throws UnsupportedOperationException {
    if (sheetIndex < 0 || rowIndex < 0 || columnBeginIndex < 0 || columnEndIndex < 0 || columnBeginIndex >= columnEndIndex) {
      throw new ExcelWraperException("行列索引 必须大于等于0且列开始索引必须小于等于列结束索引");
    }
    Sheet sheet = wb.getSheetAt(sheetIndex);
    if (sheet == null) {
      throw new ExcelWraperException("sheet不存在");
    }
    Object[] os = new Object[columnEndIndex - columnBeginIndex + 1];
    Row row = sheet.getRow(rowIndex);
    if (row != null) {
      for (int i = 0; i < os.length; i++) {
        Cell cell = row.getCell(i + columnBeginIndex);
        os[i] = getObjectValue(cell);
        if (os[i] != null && classs != null && classs.length == os.length && classs[i] != null && !classs[i].isAssignableFrom(os[i].getClass())) {
          os[i] = convertUtilsBean.convert(os[i], classs[i]);
        }
      }
    } else {
      os = new Object[columnEndIndex - columnBeginIndex + 1];
      Arrays.fill(os, "");
    }
    return os;
  }

  @Override
  public Workbook getWorkbook() {
    return wb;
  }

}
