package com.bizunited.platform.kuiper.starter.common.excel.reader.impl;

import com.bizunited.platform.kuiper.starter.common.constant.ExcelConstants;
import com.bizunited.platform.kuiper.starter.common.excel.ExcelUtils;
import com.bizunited.platform.kuiper.starter.common.excel.IRecordInterceptor;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelMigrateException;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelWraperException;
import com.bizunited.platform.kuiper.starter.common.excel.reader.IExcelReader;
import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.openxml4j.exceptions.OpenXML4JException;
import org.apache.poi.openxml4j.opc.OPCPackage;
import org.apache.poi.ss.usermodel.BuiltinFormats;
import org.apache.poi.ss.usermodel.DataFormatter;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.xssf.eventusermodel.XSSFReader;
import org.apache.poi.xssf.model.SharedStringsTable;
import org.apache.poi.xssf.model.StylesTable;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFRichTextString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

/**
 * 基于事件读取xlsx格式excel(SAX模式)
 *
 * @author Keller
 * @create 2020/8/24
 */
public class NewExcelEventReader extends DefaultHandler implements IExcelReader {

  private static final Logger LOGGER = LoggerFactory.getLogger(NewExcelEventReader.class);

  public static ConvertUtilsBean convertUtilsBean = new ConvertUtilsBean();

  static {
    convertUtilsBean.register(new Converter() {

      @SuppressWarnings("unchecked")
      @Override
      public <T> T convert(Class<T> type, Object value) {
        if (value != null) {
          return (T) DateUtil.getJavaDate(Double.parseDouble(value.toString()));
        }
        return null;
      }
    }, Date.class);
    convertUtilsBean.register(new Converter() {

      @SuppressWarnings("unchecked")
      @Override
      public <T> T convert(Class<T> type, Object value) {
        if (value != null) {
          String src = value.toString();
          if (src.matches("\\d+(\\.\\d+)?")) {
            BigDecimal bigDecimal = new BigDecimal(value.toString());
            return (T) new Integer(bigDecimal.intValue());
          }
        }
        return null;
      }
    }, Integer.class);
  }

  /**
   * 基础excel cell数据类型
   */
  enum CellDataType {
    BOOL, ERROR, FORMULA, INLINESTR, SSTINDEX, NUMBER, DATE, NULL
  }

  /**
   * excel 操作SharedStringsTable
   */
  private SharedStringsTable sst;

  /**
   * XSSF 格式Workbook reder
   */
  private XSSFReader reader;

  /**
   * 最后列内容
   */
  private String lastContents;

  /**
   * 临时行数据
   */
  private Object[] tempRowdata;

  /**
   * 单行数据处理拦截器
   */
  private IRecordInterceptor<Object[]> recordInterceptor = null;

  /**
   * 最小列数
   */
  private int minColumn;

  /**
   * 行索引
   */
  private int rowIndex = 0;

  /**
   * 列索引
   */
  private int columnIndex = 0;

  /**
   * 行开始索引
   */
  private int rowBeginIndex = 0;

  /**
   * 列开始索引
   */
  private int columnBeginIndex = 0;

  /**
   * 列数据对应类型
   */
  private Class<?>[] columnClasss;

  /**
   * 读取行范围
   */
  private int[] rowIndexs;

  /**
   * xml格式reader
   */
  private XMLReader parser;

  /**
   * 单元格数据类型，默认为字符串类型
   */
  private CellDataType nextDataType = CellDataType.SSTINDEX;

  /**
   * 日期处理formatter
   */
  private final DataFormatter formatter = new DataFormatter();

  /**
   * 单元格日期格式的索引
   */
  private short formatIndex;

  /**
   * 日期格式字符串
   */
  private String formatString;

  /**
   * 单元格
   */
  private StylesTable stylesTable;


  /**
   * 初始化excel时间处理reader
   *
   * @param is
   * @throws Exception
   */
  public NewExcelEventReader(InputStream is, File file) throws IOException, OpenXML4JException, SAXException {
    //流方式处理POI存在BUG，在100w+条数据的时候会出现OOM的情况，使用文件路径替代流的读取方式
    // 不使用该方式读取文件OPCPackage pkg = OPCPackage.open(is)
    OPCPackage pkg = OPCPackage.open(file);

    reader = new XSSFReader(pkg);
    this.sst = reader.getSharedStringsTable();
    this.stylesTable = reader.getStylesTable();
    parser = XMLReaderFactory.createXMLReader("org.apache.xerces.parsers.SAXParser");
    parser.setContentHandler(this);
  }

  /**
   * See org.xml.sax.helpers.DefaultHandler javadocs
   */
  @Override
  public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException {
    lastContents = "";
    if ("row".equals(name)) {
      rowIndex = Integer.parseInt(attributes.getValue("r")) - 1;
    } else if ("c".equals(name)) {
      String r = attributes.getValue("r");
      columnIndex = ExcelUtils.getColumnIndex(r.replaceAll("[0-9]", "")) - 1;
      //设定单元格类型
      this.setNextDataType(attributes);
    }
  }

  @Override
  public void endElement(String uri, String localName, String name) throws SAXException {
    if (ExcelConstants.EXCEL_ROW_FALG.equals(name)) {
      handleNewRow();
    } else if (ExcelConstants.EXCEL_VALUE_FALG.equals(name) || ExcelConstants.EXCEL_ELEMENT_FLAG.equals(name)) {
      Object r = null;
      if (ExcelConstants.EXCEL_VALUE_FALG.equals(name)) {
        r = this.getDataValue(lastContents.trim());
      } else {
        r = lastContents.trim();
      }
      if (rowIndex >= rowBeginIndex && columnIndex >= columnBeginIndex && columnIndex < (columnBeginIndex + minColumn)) {
        // 每一列
        if (columnClasss != null && columnClasss.length == minColumn && columnClasss[columnIndex - columnBeginIndex] != null) {
          if (!columnClasss[columnIndex - columnBeginIndex].isAssignableFrom(r.getClass())) {
            tempRowdata[columnIndex - columnBeginIndex] = ExcelUtils.convert(r, columnClasss[columnIndex - columnBeginIndex]);
          } else {
            tempRowdata[columnIndex - columnBeginIndex] = r;
          }
          tempRowdata[columnIndex - columnBeginIndex] = r;
        } else {
          tempRowdata[columnIndex - columnBeginIndex] = r;
        }
      }
    }
  }

  private void handleNewRow() {
    if (rowIndexs != null) {
      if (ArrayUtils.contains(rowIndexs, rowIndex)) {
        recordInterceptor.handle(tempRowdata, rowIndex + 1);
        rowIndexs = ArrayUtils.removeElement(rowIndexs, rowIndex);
      }
      if (rowIndexs.length == 0) {
        throw new ExcelWraperException("搜索完毕");
      }
    } else {
      if (rowIndex >= rowBeginIndex) {
        recordInterceptor.handle(tempRowdata, rowIndex + 1);
      }
    }
  }

  /**
   * 处理数据类型
   *
   * @param attributes
   */
  public void setNextDataType(Attributes attributes) {
    //cellType为空，则表示该单元格类型为数字 默认
    nextDataType = CellDataType.NUMBER;
    formatIndex = -1;
    formatString = null;
    //单元格类型
    String cellType = attributes.getValue(ExcelConstants.EXCEL_TYPE_FLAG);
    String cellStyleStr = attributes.getValue(ExcelConstants.EXCEL_CELL_TYPE_STR);

    if (ExcelConstants.EXCEL_CELL_TYPE_BOOLEAN.equals(cellType)) {
      //处理布尔值
      nextDataType = CellDataType.BOOL;
    } else if (ExcelConstants.EXCEL_CELL_TYPE_ERROR.equals(cellType)) {
      //处理错误
      nextDataType = CellDataType.ERROR;
    } else if (ExcelConstants.EXCEL_CELL_TYPE_INLINESTR.equals(cellType)) {
      nextDataType = CellDataType.INLINESTR;
    } else if (ExcelConstants.EXCEL_CELL_TYPE_STR.equals(cellType)) {
      //处理字符串
      nextDataType = CellDataType.SSTINDEX;
    } else if (ExcelConstants.EXCEL_CELL_TYPE_FORMULA.equals(cellType)) {
      nextDataType = CellDataType.FORMULA;
    }

    if (cellStyleStr != null) {
      //处理日期
      int styleIndex = Integer.parseInt(cellStyleStr);
      XSSFCellStyle style = stylesTable.getStyleAt(styleIndex);
      formatIndex = style.getDataFormat();
      formatString = style.getDataFormatString();
      if (formatString.contains("m/d/yy") || formatString.contains("m/d/yyyy") || formatString.contains("yyyy/mm/dd") || formatString.contains("yyyy/m/d")) {
        nextDataType = CellDataType.DATE;
        formatString = "yyyy-MM-dd hh:mm:ss";
      }

      if (StringUtils.isBlank(formatString)) {
        nextDataType = CellDataType.NULL;
        formatString = BuiltinFormats.getBuiltinFormat(formatIndex);
      }
    }
  }

  /**
   * 对解析出来的数据进行类型处理
   *
   * @param value 单元格的值，
   *              value代表解析：BOOL的为0或1， ERROR的为内容值，FORMULA的为内容值，INLINESTR的为索引值需转换为内容值，
   *              SSTINDEX的为索引值需转换为内容值， NUMBER为内容值，DATE为内容值
   * @return
   */
  @SuppressWarnings("deprecation")
  public Object getDataValue(String value) {
    Object resutl = null;
    switch (nextDataType) {
      // 这几个的顺序不能随便交换，交换了很可能会导致数据错误
      case BOOL:
        //布尔值
        char first = value.charAt(0);
        resutl = first == '0' ? Boolean.FALSE : Boolean.TRUE;
        break;
      case ERROR:
        //错误
        resutl = "\"ERROR:" + value + '"';
        break;
      case FORMULA:
        //公式
        resutl = value;
        break;
      case INLINESTR:
        XSSFRichTextString rtsi = new XSSFRichTextString(value);
        resutl = rtsi.toString();
        break;
      case SSTINDEX:
        //字符串
        String sstIndex = value;
        try {
          int idx = Integer.parseInt(sstIndex);
          //根据idx索引值获取内容值
          XSSFRichTextString rtss = new XSSFRichTextString(sst.getEntryAt(idx));
          resutl = rtss.toString();
          //有些字符串是文本格式的，但内容却是日期
        } catch (NumberFormatException ex) {
          resutl = value;
        }
        break;
      case NUMBER:
        //数字处理
        String numStr = "";
        if (formatString != null) {
          numStr = formatter.formatRawCellContents(Double.parseDouble(value), formatIndex, formatString).trim();
        } else {
          numStr = value;
        }
        resutl = numStr.replace("_", "").trim();
        break;
      case DATE:
        //日期处理
        String dateTimeStr = "";
        dateTimeStr = formatter.formatRawCellContents(Double.parseDouble(value), formatIndex, formatString);
        //对日期字符串作特殊处理，去掉T
        resutl = dateTimeStr.replace("T", " ");
        break;
      default:
        resutl = " ";
        break;
    }
    return resutl;
  }

  @Override
  public void characters(char[] ch, int start, int length) throws SAXException {
    lastContents += new String(ch, start, length);
  }

  @Override
  public void readSheet(Integer sheetIndex, int rowBeginIndex, int columnBeginIndex, int columnEndIndex, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    reset(sheetIndex, rowBeginIndex, columnBeginIndex, columnEndIndex, null, classs, interceptor);
  }

  @Override
  public void readSheet(Integer sheetIndex, int columnBeginIndex, int columnEndIndex, int[] rowIndexs, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    reset(sheetIndex, 0, columnBeginIndex, columnEndIndex, rowIndexs, classs, interceptor);
  }

  @Override
  public Object[] readRow(Integer sheetIndex, int rowIndex, int columnBeginIndex, int columnEndIndex, Class<?>[] classs) throws UnsupportedOperationException {
    final List<Object[]> list = new ArrayList<Object[]>();
    reset(sheetIndex, 0, columnBeginIndex, columnEndIndex, new int[]{rowIndex}, classs, new IRecordInterceptor<Object[]>() {

      @Override
      public void handle(Object[] t, int rowIndex) {
        list.add(t);
      }

    });
    return list.isEmpty() ? null : list.get(0);
  }

  /**
   * 初始化全局的参数，开始读取操作
   *
   * @param sheetIndex
   * @param rowBeginIndex
   * @param columnBeginIndex
   * @param columnEndIndex
   * @param rowIndexs
   * @param classs
   * @param interceptor
   */
  private void reset(Integer sheetIndex, int rowBeginIndex, int columnBeginIndex, int columnEndIndex, int[] rowIndexs, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    Validate.notNull(interceptor, "excel数据处理拦截器为空,请检查！");
    this.rowBeginIndex = rowBeginIndex;
    this.columnBeginIndex = columnBeginIndex;
    this.recordInterceptor = interceptor;
    this.columnClasss = classs;
    this.recordInterceptor = interceptor;
    this.minColumn = columnEndIndex - columnBeginIndex + 1;
    this.rowIndexs = rowIndexs;
    this.rowIndex = 0;
    this.columnIndex = 0;
    tempRowdata = new Object[minColumn];
    Iterator<InputStream> sheets = null;
    try {
      sheets = reader.getSheetsData();
    } catch (IOException | InvalidFormatException e) {
      throw new ExcelWraperException("excel文件读取失败", e);
    }
    int i = 0;
    while (sheets.hasNext()) {
      InputStream sheet = sheets.next();
      if (sheetIndex != null && sheetIndex >= 0 && !sheetIndex.equals(i)) {
        i++;
        continue;
      }
      InputSource sheetSource = new InputSource(sheet);
      try {
        parser.parse(sheetSource);
        sheet.close();
      } catch (ExcelWraperException e) {
        if (!"搜索完毕".equals(e.getMessage())) {
          LOGGER.error(e.getMessage(), e);
          throw new ExcelMigrateException("excel 封装错误");
        }
      } catch (SAXException e) {
        throw new ExcelWraperException("excel文件读取失败", e);
      } catch (IOException e) {
        throw new ExcelWraperException("excel文件读取失败", e);
      }
      i++;
    }
  }
}
