package com.bizunited.platform.kuiper.starter.common.excel.reader.impl;

import com.bizunited.platform.kuiper.starter.common.excel.ExcelUtils;
import com.bizunited.platform.kuiper.starter.common.excel.IRecordInterceptor;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelWraperException;
import com.bizunited.platform.kuiper.starter.common.excel.reader.IExcelReader;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.apache.poi.hssf.eventusermodel.EventWorkbookBuilder;
import org.apache.poi.hssf.eventusermodel.FormatTrackingHSSFListener;
import org.apache.poi.hssf.eventusermodel.HSSFEventFactory;
import org.apache.poi.hssf.eventusermodel.HSSFListener;
import org.apache.poi.hssf.eventusermodel.HSSFRequest;
import org.apache.poi.hssf.eventusermodel.MissingRecordAwareHSSFListener;
import org.apache.poi.hssf.eventusermodel.dummyrecord.LastCellOfRowDummyRecord;
import org.apache.poi.hssf.eventusermodel.dummyrecord.MissingCellDummyRecord;
import org.apache.poi.hssf.record.BOFRecord;
import org.apache.poi.hssf.record.BlankRecord;
import org.apache.poi.hssf.record.BoolErrRecord;
import org.apache.poi.hssf.record.BoundSheetRecord;
import org.apache.poi.hssf.record.CellValueRecordInterface;
import org.apache.poi.hssf.record.FormulaRecord;
import org.apache.poi.hssf.record.LabelRecord;
import org.apache.poi.hssf.record.LabelSSTRecord;
import org.apache.poi.hssf.record.NumberRecord;
import org.apache.poi.hssf.record.Record;
import org.apache.poi.hssf.record.RowRecord;
import org.apache.poi.hssf.record.SSTRecord;
import org.apache.poi.hssf.record.StringRecord;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 基于事件读取xls格式excel（大数据量文件读取使用）
 *
 * @author Keller
 * @create 2020/8/24
 */
public class OldExcelEventReader implements HSSFListener, IExcelReader {

  private static final Logger LOGGER = LoggerFactory.getLogger(OldExcelEventReader.class);

  /**
   * poi文件操作
   */
  private POIFSFileSystem fs;

  /**
   * Should we output the formula, or the value it has?
   */
  private boolean outputFormulaValues = true;

  /*** excel中的字符串集合 */
  private SSTRecord sstRecord;
  private FormatTrackingHSSFListener formatListener;

  /**
   * So we known which sheet we're on
   */
  private List<BoundSheetRecord> boundSheetRecords = new ArrayList<BoundSheetRecord>();

  private boolean outputNextStringRecord;

  /**
   * 单行数据处理拦截器
   */
  private IRecordInterceptor<Object[]> recordInterceptor = null;

  /**
   * 临时行数据
   */
  private List<Object> tempRowdata = new ArrayList<Object>();

  /**
   * 行索引
   */
  private int rowIndex = 0;

  /**
   * 列索引
   */
  private int columnIndex = 0;

  /**
   * 行开始索引
   */
  private int rowBeginIndex = 0;

  /**
   * 列开始索引
   */
  private int columnBeginIndex = 0;

  /**
   * 最小列数
   */
  private int minColumn = 0;

  /**
   * 行范围数组
   */
  private int[] rowIndexs;

  /**
   * 列值对应java class类型
   */
  private Class<?>[] columnClasss;

  /**
   * 当前处理sheet缩影
   */
  private Integer sheetIndex;

  /**
   * HSSF时间处理工程类
   */
  private HSSFEventFactory factory = new HSSFEventFactory();

  /**
   * HSSF请求
   */
  private HSSFRequest request;

  private byte[] data;

  /**
   * @param is The POIFSFileSystem to process
   * @throws IOException
   */
  public OldExcelEventReader(InputStream is) throws IOException {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    byte[] buffer = new byte[4096];
    int read = 0;
    while ((read = is.read(buffer)) != -1) {
      baos.write(buffer, 0, read);
    }
    data = baos.toByteArray();
    formatListener = new FormatTrackingHSSFListener(new MissingRecordAwareHSSFListener(this));
    request = new HSSFRequest();
    if (outputFormulaValues) {
      request.addListenerForAllRecords(formatListener);
    } else {
      EventWorkbookBuilder.SheetRecordCollectingListener workbookBuildingListener = new EventWorkbookBuilder.SheetRecordCollectingListener(formatListener);
      request.addListenerForAllRecords(workbookBuildingListener);
    }
  }

  /**
   * Main HSSFListener method, processes events, and outputs the CSV as the file is processed.
   */
  @Override
  public void processRecord(Record record) {
    /*
    1.获取单元格类型
    2.针对单元格类型返回值
    3.公式、日期、数字等单元格进行值格式处理
     */
    String thisStr = null;
    switch (record.getSid()) {
      //新的sheet
      case BoundSheetRecord.sid:
        boundSheetRecords.add((BoundSheetRecord) record);
        break;
      case BOFRecord.sid:
        break;
      //新的一行
      case RowRecord.sid:
        break;
      case SSTRecord.sid:
        sstRecord = (SSTRecord) record;
        break;
      case BlankRecord.sid:
        thisStr = "";
        break;
      case BoolErrRecord.sid:
        thisStr = "";
        break;
      //公式单元格处理
      case FormulaRecord.sid:
        FormulaRecord frec = (FormulaRecord) record;
        if (outputFormulaValues) {
          if (Double.isNaN(frec.getValue())) {
            // Formula result is a string
            // This is stored in the next record
            outputNextStringRecord = true;
          } else {
            thisStr = formatListener.formatNumberDateCell(frec);
          }
        } else {
          thisStr = "function";
        }
        break;
      //字符类型单元格处理
      case StringRecord.sid:
        if (outputNextStringRecord) {
          // String for formula
          StringRecord srec = (StringRecord) record;
          thisStr = srec.getString();
          outputNextStringRecord = false;
        }
        break;
      case LabelRecord.sid:
        LabelRecord lrec = (LabelRecord) record;
        thisStr = lrec.getValue();
        break;
      case LabelSSTRecord.sid:
        LabelSSTRecord lsrec = (LabelSSTRecord) record;
        if (sstRecord == null) {
          thisStr = "";
        } else {
          thisStr = sstRecord.getString(lsrec.getSSTIndex()).toString();
        }
        break;
      //数字类型单元格处理
      case NumberRecord.sid:
        NumberRecord numrec = (NumberRecord) record;
        thisStr = formatListener.formatNumberDateCell(numrec);
        break;
      default:
        break;
    }
    if (sheetIndex != null && sheetIndex > 0 && !sheetIndex.equals(boundSheetRecords.size() - 1)) {
      return;
    }
    if (record instanceof CellValueRecordInterface) {
      CellValueRecordInterface rcRecord = (CellValueRecordInterface) record;
      rowIndex = rcRecord.getRow();
      columnIndex = rcRecord.getColumn();
    }
    if (record instanceof MissingCellDummyRecord) {
      thisStr = "";
    }
    if (columnIndex >= columnBeginIndex) {
      // 每一列
      if (thisStr != null && columnClasss != null && columnClasss.length == minColumn && columnClasss[columnIndex] != null) {
        try {
          if (!columnClasss[columnIndex].isAssignableFrom(thisStr.getClass())) {
            tempRowdata.add(ExcelUtils.convert(thisStr, columnClasss[columnIndex]));
          } else {
            tempRowdata.add(thisStr);
          }
        } catch (Exception e) {
          LOGGER.error(e.getMessage(), e);
          tempRowdata.add(thisStr);
        }
      } else {
        tempRowdata.add(thisStr);
      }
    }
    if (record instanceof LastCellOfRowDummyRecord) {
      // 行结束
      Object[] r = tempRowdata.toArray();
      if (r.length != minColumn) {
        r = Arrays.copyOf(r, minColumn);
      }
      if (rowIndexs != null) {
        if (ArrayUtils.contains(rowIndexs, rowIndex)) {
          recordInterceptor.handle(r, rowIndex + 1);
          rowIndexs = ArrayUtils.removeElement(rowIndexs, rowIndex);
        }
        if (rowIndexs.length == 0) {
          throw new ExcelWraperException("搜索完毕");
        }
      } else {
        if (rowIndex >= rowBeginIndex) {
          recordInterceptor.handle(r, rowIndex + 1);
        }
      }
      tempRowdata.clear();
    }
  }

  @Override
  public void readSheet(Integer sheetIndex, int rowBeginIndex, int columnBeginIndex, int columnEndIndex, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    reset(sheetIndex, rowBeginIndex, columnBeginIndex, columnEndIndex, null, classs, interceptor);
  }

  @Override
  public void readSheet(Integer sheetIndex, int columnBeginIndex, int columnEndIndex, int[] rowIndexs, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    reset(sheetIndex, 0, columnBeginIndex, columnEndIndex, rowIndexs, classs, interceptor);
  }

  @Override
  public Object[] readRow(Integer sheetIndex, int rowIndex, int columnBeginIndex, int columnEndIndex, Class<?>[] classs) throws UnsupportedOperationException {
    final List<Object[]> list = new ArrayList<>();
    reset(sheetIndex, 0, columnBeginIndex, columnEndIndex, new int[]{rowIndex}, classs, new IRecordInterceptor<Object[]>() {

      @Override
      public void handle(Object[] t, int rowIndex) {
        list.add(t);
      }

    });
    return list.isEmpty() ? null : list.get(0);
  }

  /**
   * 初始化excel读取参数
   *
   * @param sheetIndex
   * @param rowBeginIndex
   * @param columnBeginIndex
   * @param columnEndIndex
   * @param rowIndexs
   * @param classs
   * @param interceptor
   */
  private void reset(Integer sheetIndex, int rowBeginIndex, int columnBeginIndex, int columnEndIndex, int[] rowIndexs, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    Validate.notNull(interceptor, "excel处理拦截器为空,请检查！");

    this.rowBeginIndex = rowBeginIndex;
    this.columnBeginIndex = columnBeginIndex;
    this.recordInterceptor = interceptor;
    this.columnClasss = classs;
    this.recordInterceptor = interceptor;
    this.minColumn = columnEndIndex - columnBeginIndex + 1;
    this.sheetIndex = sheetIndex;
    this.rowIndexs = rowIndexs;
    this.rowIndex = 0;
    this.columnIndex = 0;
    try {
      this.fs = new POIFSFileSystem(new ByteArrayInputStream(data));
    } catch (IOException e) {
      throw new ExcelWraperException("excel文件读取错误", e);
    }
    try {
      factory.processWorkbookEvents(request, fs);
    } catch (ExcelWraperException e) {
      if (!"搜索完毕".equals(e.getMessage())) {
        LOGGER.error(e.getMessage(), e);
      }
    } catch (IOException e) {
      throw new ExcelWraperException("excel文件读取错误", e);
    }

  }

}
