package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.service.StaticTemplateService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;

@Api(value = "StaticTemplateController")
@RestController
@RequestMapping("/v1/kuiper/staticTemplates")
public class StaticTemplateController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(StaticTemplateController.class);
  
  @Autowired
  private StaticTemplateService staticTemplateService;
  
  @ApiOperation(value = "该方法用于基于saturn扫描组件中已扫描确定的某一个静态模型（完整的静态模型类名），完成一个静态表单模板的初始化过程"
      + "这个过程一般发生在操作人员通过页面上的“创建静态表单模板”功能选定一个静态模型，并点击“创建”按钮时发生。 通过这个方法，系统将返回这个静态模型的初始化信息。初始化信息包括静态模型本身的描述，还包括静态模型关联的一般属性、关联属性，以及关联属性中OneToMany性质关联的下一级属性等信息"
      + "以及根据静态模型关联情况初始化完成的“静态服务源”信息——当然也包括了直接关联的服务源和OneToMany关系中关联的服务源。" 
      , notes="请注意，这里并不是返回PersistentClass对象的json结构，而是一个满足技术白皮书中静态模型结构定义的JSON结构。详细情况可参见《表单引擎技术白皮书V1.1》白皮书文档中的描述")
  @PostMapping("/initStaticTemplate")
  @CacheEvict(cacheNames="template" , allEntries=true)
  public ResponseModel initStaticTemplate(@RequestBody @ApiParam(name = "template", value = "新的表单模板信息，至少说明版本信息、模板类型、已经使用的静态类全名，其它不需要指定了") TemplateEntity template ) {
    try {
      Principal principal = this.getPrincipal();
      TemplateEntity result = this.staticTemplateService.initStaticTemplate(template , principal , true);
      return this.buildHttpResultW(result, "properties", "relations","creator",
              "groupRelations","groupRelations.properties","groupRelations.relations","groupRelations.itemRelations",
              "groupRelations.itemRelations.properties","groupRelations.itemRelations.relations",
              "itemRelations","itemRelations.properties","itemRelations.relations",
              "event","visibility","visibility.attributes","visibility.buttons");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "对指定的表单模板进行升级处理，通过这个接口服务系统将自动保留templateId对应的模板的全部主要设定（包括模型、数据源、事件、可见性）" , 
      notes="在只更改模板基本信息，版本信息的情况下，拷贝出一个新的模板")
  @PutMapping("")
  @CacheEvict(cacheNames="template" , allEntries=true)
  public ResponseModel upgrade(@ApiParam(name = "templateId", value = "指定的表单模板升级时使用的参考模板") @RequestParam(name="templateId") String templateId , 
                               @ApiParam(name = "newVersion", value = "新的版本号") @RequestParam(name="newVersion") String newVersion,
                               @ApiParam(value = "更新模版实例使用当前模版版本") @RequestParam(name="updateInstance", required = false, defaultValue = "0") Boolean updateInstance) {
    try {
      Principal principal = this.getPrincipal();
      TemplateEntity result = this.staticTemplateService.upgradeStaticTemplate(templateId, newVersion, updateInstance, principal);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * @param templateId
   */
  @ApiOperation(value = "删除指定的模板信息，真删除——包括模板的可见性、模板的事件、模板的数据源等信息。" , 
      notes="已经存在实例在运行的模板信息，是不允许进行删除的")
  @DeleteMapping("")
  @CacheEvict(cacheNames="template" , allEntries=true)
  public ResponseModel deleteById(@ApiParam(name = "id", value = "表单模板的数据库编号") String id) {
    try {
      this.staticTemplateService.deleteById(id);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
