package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.TemplateStyleEntity;
import com.bizunited.platform.kuiper.starter.service.TemplateStyleService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.io.IOException;
import java.io.OutputStream;
import java.util.List;

@Api(value = "TemplateStyleController")
@RestController
@RequestMapping("/v1/kuiper/templateStyles")
public class TemplateStyleController extends BaseController {
  private static final Logger LOGGER = LoggerFactory.getLogger(TemplateStyleController.class);
  @Autowired
  private TemplateStyleService templateStyleService;
  
  @ApiOperation(value = "查询目前系统中所有定义的样式信息，无论样式信息处于什么状态。并按照创建时间进行倒序排列")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      List<TemplateStyleEntity> results = templateStyleService.findAll();
      return this.buildHttpResultW(results);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "按照指定的状态条件，查询符合状态的样式信息")
  @RequestMapping(value = "/findByTstatus", method = RequestMethod.GET)
  public ResponseModel findByTstatus(@ApiParam(name = "tstatus", value = "1：状态正确；0：状态异常") @RequestParam("tstatus") Integer tstatus) {
    try {
      List<TemplateStyleEntity> results = templateStyleService.findByTstatus(tstatus);
      return this.buildHttpResultW(results);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "查询目前系统中，指定样式数据编号的样式基本信息")
  @RequestMapping(value = "/findDetailsById", method = RequestMethod.GET)
  public ResponseModel findDetailsById(@ApiParam(name = "id", value = "指定的样式编号信息") @RequestParam("id") String id) {
    try {
      TemplateStyleEntity results = templateStyleService.findDetailsById(id);
      return this.buildHttpResultW(results);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "查询目前系统中，指定样式业务编号（code）的样式基本信息")
  @RequestMapping(value = "/findDetailsByCode", method = RequestMethod.GET)
  public ResponseModel findDetailsByCode(@ApiParam(name = "code", value = "指定的样式业务编号信息") @RequestParam("code") String code) {
    try {
      TemplateStyleEntity results = templateStyleService.findDetailsByCode(code);
      return this.buildHttpResultW(results);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "按照指定的表单样式业务编号（code），查询指定的样式文件内容信息" , notes="指定的文件内容，如果没有查询到任何文件内容，则返回一个空内容")
  @RequestMapping(value = "/findContentByCode", method = RequestMethod.GET)
  public void findContentByCode(HttpServletResponse response , 
                                @ApiParam(name = "code", value = "指定的样式业务编号信息") @RequestParam("code") String code) {
    try {
      byte[] cssBytes = templateStyleService.findContentByCode(code);
      if (cssBytes == null) {
        return;
      }
      // 显示到页面上
      this.writeResponseGif(response, cssBytes);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
    }
  }
  /**
   * 向response中写入css内容
   * @param response HttpServletResponse
   * @param result 写入的byte信息
   */
  private void writeResponseGif(HttpServletResponse response, byte[] result) {
    response.setContentType("text/css");
    OutputStream out = null;
    try {
      out = response.getOutputStream();
      out.write(result);
    } catch (Exception e) {
      throw new IllegalArgumentException(e.getMessage() , e);
    }
  }
  @ApiOperation(value = "创建一个模板样式信息" , notes="创建时，需要传入code、name、systemStyle(true：系统样式；false：非系统样式)和file（样式文件）信息")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(HttpServletRequest request , @ApiParam(required=true , name="file" , value="只能上传一个样式文件信息，且只能是后缀css、less") @RequestParam("file") MultipartFile file) {
    try {
      String code = request.getParameter("code");
      String name = request.getParameter("name");
      String systemStyle = request.getParameter("systemStyle");
      Boolean currentSystemStyle = Boolean.valueOf(systemStyle);
      TemplateStyleEntity templateStyle = new TemplateStyleEntity();
      templateStyle.setCode(code);
      templateStyle.setName(name);
      // 文件内容信息
      Validate.notNull(file , "创建样式信息时，必须上传文件信息!!");
      String originalFilename = file.getOriginalFilename();
      String prefix = null;
      int prefixIndex = originalFilename.lastIndexOf('.');
      if (prefixIndex != -1) {
        prefix = originalFilename.substring(prefixIndex + 1);
        prefix = prefix.toLowerCase();
      }
      TemplateStyleEntity result = templateStyleService.create(templateStyle, file.getBytes(), currentSystemStyle, prefix);
      return this.buildHttpResultW(result);
    } catch (RuntimeException | IOException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "修改指定的模板样式信息" , notes="只有一部分信息可以修改（只有样式中文说明，文件内容信息），除了file可以指定以外，还可以指定name参数，另外id参数是必须传入的")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(HttpServletRequest request , 
                              @ApiParam(required=false , name="file" , value="只能上传一个样式文件信息，且只能是后缀css、less") @RequestParam(name="file" , required=false) MultipartFile file) {
    try {
      TemplateStyleEntity templateStyle = new TemplateStyleEntity();
      String id = request.getParameter("id");
      String name = request.getParameter("name");
      templateStyle.setId(id);
      templateStyle.setName(name);
      // 可能的文件内容信息
      byte[] fileContent = null;
      String prefix = null;
      if(file != null) {
        String originalFilename = file.getOriginalFilename();
        int prefixIndex = originalFilename.lastIndexOf('.');
        if (prefixIndex != -1) {
          prefix = originalFilename.substring(prefixIndex + 1);
          prefix = prefix.toLowerCase();
        }
        fileContent = file.getBytes();
      }
      TemplateStyleEntity result = templateStyleService.update(templateStyle, fileContent, prefix);
      return this.buildHttpResultW(result);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "按照指定的样式数据层编号，对指定样式信息的状态进行修改")
  @RequestMapping(value = "/updateStatusById", method = RequestMethod.PATCH)
  public ResponseModel updateStatusById(@ApiParam(required=true , name="id" , value="要进行修改的样式编号信息") @RequestParam("id") String id ,
                               @ApiParam(required=true , name="tstatus" , value="改变后的状态信息，true：启用；false：禁用") @RequestParam("tstatus") boolean tstatus) {
    try {
      templateStyleService.updateStatusById(id, tstatus);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}