package com.bizunited.platform.kuiper.starter.repository;

import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.starter.repository.internal.TemplateRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.Date;
import java.util.List;

/**
 * 和表单模板有关的数据层操作都在这里定义
 * @author yinwenjie
 */
@Repository("TemplateRepository")
public interface TemplateRepository
    extends
      JpaRepository<TemplateEntity, String>,
      JpaSpecificationExecutor<TemplateEntity>,
      TemplateRepositoryCustom {
  
  /**
   * 修改指定的templateCode和cversion的DefaultVersion属性为true
   * @param templateCode 指定的表单模板业务编号
   * @param cversion 指定的表单模板版本号
   */
  @Modifying
  @Query(value="update engine_form_template set default_version = 1, modify_time = :modifyTime where code = :templateCode and cversion = :cversion" , nativeQuery=true)
  public void updateDefaultVersion(@Param("templateCode") String templateCode ,@Param("cversion") String cversion,  @Param("modifyTime") Date modifyTime);
  
  /**
   * 修改指定的templateCode的DefaultVersion属性为false
   * @param templateCode
   */
  @Modifying
  @Query(value="update engine_form_template set default_version = 0 where code = :templateCode" , nativeQuery=true)
  public void updateDefaultVersion(@Param("templateCode") String templateCode);
  
  /**
   * 按照模板编号，删除指定的模板基础信息
   * @param templateId 执行的模板基础信息
   */
  @Modifying
  @Query(value="delete from engine_form_template where engine_form_template.id = :templateId" , nativeQuery=true)
  public void deleteByTemplateId(@Param("templateId") String templateId);

  /**
   * 根据编码和版本号查询
   * @param code
   * @param cversion
   * @return
   */
  public TemplateEntity findByCodeAndCversion(String code , String cversion);
  
  /**
   * 查询指定模板的默认版本信息（只有基本信息，没有关联信息）
   * @param code 指定的模板业务编号
   * @return
   */
  @Query("from TemplateEntity te where te.code = :code and te.defaultVersion = true order by te.createTime desc ")
  public TemplateEntity findByDefaultVersion(String code);
  
  /**
   * 查询code一致的模板，并按照时间进行倒序，主要是找到最新的code模板信息
   * @param code 模板业务编号
   * @return
   */
  @Query("from TemplateEntity te where te.code = :code order by te.createTime desc ")
  public List<TemplateEntity> findByCode(@Param("code") String code);
  /**
   * 该接口查询指定模板下的属性结构信息。注意，只包括属性结构，也就是主模型上的一般属性、关联属性。
   * 如果要获取模板下的事件、布局等信息，请调用其它相关接口
   * @param id 指定的表单模板的数据层编号
   */
  @Query(value="select distinct template from TemplateEntity template "
      + " left join fetch template.properties p "
      + " left join fetch template.relations r "
      + " where template.id = :id ")
  public TemplateEntity findDetailsById(@Param("id") String id);
  /**
   * 按照指定的模板样式（样式code），查询使用其的多个模板信息。
   * @param formStyle 指定的模板样式（样式code）
   * 并且项目名为空
   * @return 
   */
  @Query("select t from TemplateEntity t where t.formStyle = :formStyle and t.projectName = '' or  t.projectName is null")
  public List<TemplateEntity> findByFormStyleAndBlankProjectName(@Param("formStyle") String formStyle);
  /**
   * 查询一批指定的id编号的表单模板信息在系统中真实存在的数量
   * @param ids
   * @return
   */
  @Query(value="select count(*) from engine_form_template where id in (:ids) " , nativeQuery = true)
  public int countByIds(@Param("ids") String[] ids);

  /**
   * 根据ID集合查询详情模版
   * @param ids
   * @return
   */
  @Query(value="select distinct template from TemplateEntity template "
      + " left join fetch template.properties p "
      + " left join fetch template.relations r "
      + " where template.id in :ids ")
  List<TemplateEntity> findDetailsByIds(@Param("ids") String[] ids);

  /**
   * 根据编码和版本号统计数量
   * @param code
   * @param cversion
   * @return
   */
  @Query("select count(*) from TemplateEntity t where t.code = :code and t.cversion = :cversion")
  long countByCodeAndVersion(@Param("code") String code, @Param("cversion") String cversion);

  /**
   * 按照指定的模板样式（样式code），查询使用其的多个模板信息。
   * @param formStyle 指定的模板样式（样式code）
   * @return
   */
  public List<TemplateEntity> findByFormStyleAndProjectName(String formStyle, String projectName);
}