package com.bizunited.platform.kuiper.starter.repository.table;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import org.springframework.util.CollectionUtils;

/**
 * TODO 目前该实现方法仅支持MySQL 5.6、5.7数据库
 * @author yinwenjie
 *
 */
@Repository("TableOperateRepositoryImpl")
public class TableOperateRepositoryImpl implements TableOperateRepositoryCustom {
  
  /**
   * 数据表名、字段名需要满足的命名规则
   */
  private static final String TABLE_AND_FIELD_PATTERN = "^[A-Za-z_]{1}[\\w]+$";
  private static final String TABLE_ERROR = "错误的数据表名，数据表必须以字母开头，大于两位，且只包括字母、数字和下划线；并且不能含有数据库保留关键字";
  private static final String FIELD_ERROR = "错误的数据字段名，数据字段必须以字母开头，大于两位，且只包括字母、数字和下划线；并且不能含有数据库保留关键字";
  private static final String TAG_SEPARATOR = "line.separator";
  private static final String MESS_REF_MANYTOONE = "ManyToOne";
  private static final String MESS_REF_MANYTOMANY = "ManyToMany";
  private static final String MESS_REF_VARCHAR = "varchar";
  private static final String MESS_REF_ENGINE = ") ENGINE=InnoDB DEFAULT CHARSET=utf8;";
  private static final String OP_CREATE_TABLE = "CREATE TABLE `";
  
  /**
   * 保留字段名，这些字段名不能出现在动态模型中。
   * 要进行验证报错
   */
  private static final String[] RETAINFIELDNAMES = new String[] {"add" , "all" , "alter" , "analyze" , "and" , "as" ,
      "asc" , "asensitive" , "before" , "between" , "bigint" , "binary" , "blob" , "both" , "by" , "call" , "cascade" , "case" , "create" , 
      "change" , "char" , "character" , "check" , "collate" , "column" , "condition" , "connection" , "constraint" , "continue" , "convert" , 
      "cross" , "current_date" , "current_time" , "current_timestamp" , "current_user" , "cursor" , "database" , "databases" , "day_hour" , 
      "day_microsecond" , "day_minute" , "day_second" , "dec" , "decimal" , "declare" , "default" , "delayed" , "delete" , "desc" , 
      "describe" , "deterministic" , "distinct" , "distinctrow" , "div" , "double" , "drop" , "dual" , "each" , "else" , "elseif" , "enclosed" , 
      "escaped" , "exists" , "exit" , "explain" , "false" , "fetch" , "float" , "float4" , "float8" , "for" , "force" , "foreign" , "from" , 
      "fulltext" , "goto" , "grant" , "group" , "having" , "high_priority" , "hour_microsecond" , "hour_minute" , "hour_second" , "if" , 
      "ignore" , "in" , "index" , "infile" , "inner" , "inout" , "insensitive" , "insert" , "int" , "int1" , "int2" , "int3" , "int4" , "int8" , 
      "integer" , "interval" , "into" , "is" , "iterate" , "join" , "key" , "keys" , "kill" , "label" , "leading" , "leave" , "left" , "like" , 
      "limit" , "linear" , "lines" , "load" , "localtime" , "localtimestamp" , "lock" , "long" , "longblob" , "longtext" , "loop" , "low_priority" , 
      "match" , "mediumblob" , "mediumint" , "mediumtext" , "middleint" , "minute_microsecond" , "minute_second" , "mod" , "modifies" , 
      "natural" , "not" , "no_write_to_binlog" , "null" , "numeric" , "on" , "optimize" , "option" , "optionally" , "or" , "order" , "out" , 
      "outer" , "outfile" , "precision" , "primary" , "procedure" , "purge" , "raid0" , "range" , "read" , "reads" , "real" , "references" , 
      "regexp" , "release" , "rename" , "repeat" , "replace" , "require" , "restrict" , "return" , "revoke" , "right" , "rlike" , "schema" , 
      "second_microsecond" , "select" , "sensitive" , "separator" , "set" , "show" , "smallint" , "spatial" , "specific" , "sql" , 
      "sqlexception" , "sqlstate" , "sqlwarning" , "sql_big_result" , "sql_calc_found_rows" , "sql_small_result" , "ssl" , "starting" , 
      "straight_join" , "table" , "terminated" , "then" , "tinyblob" , "tinyint" , "tinytext" , "to" , "trailing" , "trigger" , "true" , 
      "undo" , "union" , "unique" , "unlock" , "unsigned" , "update" , "usage" , "use" , "using" , "utc_date" , "utc_time" , "utc_timestamp" , 
      "values" , "varbinary" , MESS_REF_VARCHAR , "varcharacter" , "varying" , "when" , "where" , "while" , "with" , "write" , "x509" , "xor" , "year_month" , "zerofill"};
  
  /**
   * 目前动态数据表只支持的数据类型
   */
  private static Map<String, String> classTypeMapping = new HashMap<>();
  
  static {
    classTypeMapping.put("java.lang.String", MESS_REF_VARCHAR);
    classTypeMapping.put("java.lang.Integer", "int");
    classTypeMapping.put("java.lang.Long", "bigint");
    classTypeMapping.put("java.lang.Float", "float");
    classTypeMapping.put("java.math.BigDecimal", "double");
    classTypeMapping.put("java.lang.Double", "double");
    classTypeMapping.put("java.lang.Boolean", "bit");
    classTypeMapping.put("java.util.Date", "timestamp");
  }
  
  @Autowired
  @PersistenceContext
  private EntityManager entityManager;
  
  /**
   * 在正式开始业务表创建、修改动作时，检查主数据表设定的合规性
   * @param template
   */
  private void validateTemplate(TemplateEntity template) {
    Validate.notNull(template , "动态模板信息必须传入");
    String resouceTableName = template.getTableName();
    Validate.matchesPattern(resouceTableName, TABLE_AND_FIELD_PATTERN, TABLE_ERROR);
    Validate.isTrue(!StringUtils.equalsAnyIgnoreCase(resouceTableName, RETAINFIELDNAMES) , TABLE_ERROR);
    // 主表下的字段
    Set<TemplatePropertyEntity> properties =  template.getProperties();
    this.validateProperties(properties);
    // 然后检测可能的分组
    Set<TemplateGroupEntity> groups = template.getGroupRelations();
    this.validateGroups(groups);
    // 最后是检测可能的明细
    Set<TemplateItemEntity> items = template.getItemRelations();
    this.validateItems(items);
  }
  
  /**
   * 在正式开始业务表创建、修改动作时，检查分组数据表设定的合规性
   * @param template
   */
  private void validateGroups(Set<TemplateGroupEntity> groups) {
    if(groups == null) {
      return;
    }
    
    for (TemplateGroupEntity templateGroupItem : groups) {
      String tableName = templateGroupItem.getTableName();
      Validate.matchesPattern(tableName, TABLE_AND_FIELD_PATTERN, TABLE_ERROR);
      Validate.isTrue(!StringUtils.equalsAnyIgnoreCase(tableName, RETAINFIELDNAMES) , TABLE_ERROR);
      // 分组下的字段
      Set<TemplatePropertyEntity> properties = templateGroupItem.getProperties();
      this.validateProperties(properties);
      // 分组下的明细
      Set<TemplateItemEntity> items = templateGroupItem.getItemRelations();
      this.validateItems(items);
    }
  }
  
  /**
   * 在正式开始业务表创建、修改动作时，检查明细数据表设定的合规性
   * @param template
   */
  private void validateItems(Set<TemplateItemEntity> items) {
    if(items == null) {
      return;
    }
    for (TemplateItemEntity templateItem : items) {
      String tableName = templateItem.getTableName();
      Validate.matchesPattern(tableName, TABLE_AND_FIELD_PATTERN, TABLE_ERROR);
      Validate.isTrue(!StringUtils.equalsAnyIgnoreCase(tableName, RETAINFIELDNAMES) , TABLE_ERROR);
      // 明细下的字段信息
      Set<TemplatePropertyEntity> properties = templateItem.getProperties();
      this.validateProperties(properties);
    }
  }
  
  /**
   * 在正式开始业务表创建、修改动作时，检查数据表中各字段设定的合规性
   * @param template
   */
  private void validateProperties(Set<TemplatePropertyEntity> properties) {
    if(properties == null) {
      return;
    }
    for (TemplatePropertyEntity templateProperty : properties) {
      String propertyDbName = templateProperty.getPropertyDbName();
      Validate.matchesPattern(propertyDbName, TABLE_AND_FIELD_PATTERN, FIELD_ERROR);
      Validate.isTrue(!StringUtils.equalsAnyIgnoreCase(propertyDbName, RETAINFIELDNAMES) , FIELD_ERROR);
    }
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.table.TableOperateRepositoryCustom#createTable(com.bizunited.platform.kuiper.entity.TemplateEntity)
   */
  @Override
  @Transactional
  public void createTable(TemplateEntity template) {
    /*
     * 处理过程如下：
     * 注意，验证过程目前可以忽略，因为外层方法已经都做了（如果有必要，后续版本再增加）
     * 2、首先处理主数据表中的一般属性、ManyToOne性质的关联属性、ManyToMany性质的关联属性
     *  a、一般属性的处理最简单，这里不再赘述，只需要关注字段名、字段类型、可能的字段长度、可能的字段是否可以为null的设定即可
     *  接着以元数据表为基准，依据设定的“唯一性”字段，建立唯一键约束。（注意，这里并不创建外键关联）
     *  另外，注意，应该自行添加主键、表单实例信息
     *  b、然后处理ManyToOne性质的关联属性，需要关注字段名、是否可以为null的设定，因为这种关联字段类型肯定为string，长度肯定为255；
     *  接着以元数据表和目标数据表为基准，增加建立外键关联的脚本段。
     *  c、然后处理ManyToMany性质的关联属性，主要关注字段名、是否可以为null
     *  接着以元数据表、目标数据表为基准，使用元数据表_目标数据表_mapping的表明，建立映射表，并且建立这个映射表的外键关联
     *  注意：ManyToMany的中间映射表的建表脚本，将在本方法最后执行
     * 
     * 
     * 注意：以上相关操作点，需要做成私有方法，因为后续的过程，还会使用这些方法
     * 另外注意：操作中，数据表名的检测放在本方法的最开始位置进行
     * 
     * 3、接着处理主数据表的OneToMany明细表
     *  a、首先是明细表中的一般字段，很好处理，直接调用2.a方法
     *  注意：不同的是，需要自行建立PK主键，以及和主表的关联字段
     *  b、然后处理明细表中的ManyToOne性质的关联属性，直接调用2.b方法
     *  c、最后使用明细主标名_id的方式，将这个明细表和主表的主键建立外键关系
     *  d、然后处理明细表中的ManyToMany性质的关联属性，直接使用2.c方法
     * 
     * 4、接着处理主数据表的OneToOne分组表
     *  a、分组表中的一般字段很好处理，直接调用2.a方法，这里就不再赘述
     *  b、然后处理分组表中的ManyToOne性质的关联属性，直接调用2.b方法
     *  注意：使用分组名_id的字段，将这个分组表和主表的主键建立外键关联
     *  c、然后处理分组表中的ManyToMany性质的关联属性，直接调用2.c方法
     *  d、接着处理分组表中的OneToMany性质的关联属性，直接使用3 的私有方法
     *  
     * 5、开始执行各数据库脚本
     *   a、首先执行主表建表脚本
     *   b、然后执行各OneToOne的建表脚本
     *   c、然后执行各OneToMany的建表脚本
     *   d、最后是各个ManyToMany的建表脚本
     * */
    // 首先检测各种数据表名、字段名是否符合要求
    this.validateTemplate(template);
    // 2、=======
    String resouceTableName = template.getTableName();
    Set<TemplatePropertyEntity> properties = template.getProperties();
    StringBuilder sqlDdl = new StringBuilder();
    sqlDdl.append(OP_CREATE_TABLE + resouceTableName + "` (").append(System.getProperty(TAG_SEPARATOR));
    
    // 2.a、======
    sqlDdl.append(this.createTableProperties(properties)).append(System.getProperty(TAG_SEPARATOR));
    // 2.b、======
    Set<TemplateRelationEntity> relations = template.getRelations();
    if(relations != null && !relations.isEmpty()) {
      Set<TemplateRelationEntity> manyToOneRelations = relations.stream().filter(item -> item.getRelationType().equals(MESS_REF_MANYTOONE)).collect(Collectors.toSet());
      if(manyToOneRelations != null && !manyToOneRelations.isEmpty()) {
        sqlDdl.append(this.createManyToOneRelations(manyToOneRelations)).append(System.getProperty(TAG_SEPARATOR));
      }
    }
    // 确认PK和表单实例forminstanceid的索引
    sqlDdl.append("  PRIMARY KEY (`id`) ").append(System.getProperty(TAG_SEPARATOR));
    sqlDdl.append(MESS_REF_ENGINE).append(System.getProperty(TAG_SEPARATOR));
    // 2.c、======创建多对多映射信息
    List<String> manyToManyMappingSqls = new LinkedList<>();
    if(relations != null && !relations.isEmpty()) {
      Set<TemplateRelationEntity> manyToManyRelations = relations.stream().filter(item -> item.getRelationType().equals(MESS_REF_MANYTOMANY)).collect(Collectors.toSet());
      if(manyToManyRelations != null && !manyToManyRelations.isEmpty()) {
        this.createManyToManyRelations(resouceTableName, manyToManyRelations , manyToManyMappingSqls);
      }
    }
    
    // 3、======
    Set<TemplateItemEntity> oneToManyRelations = template.getItemRelations();
    List<String> oneToManySqls = new LinkedList<>();
    this.createItems(oneToManyRelations, manyToManyMappingSqls , oneToManySqls);
    
    // 4、======
    Set<TemplateGroupEntity> oneToOneRelations = template.getGroupRelations();
    List<String> oneToOneSqls = new LinkedList<>();
    this.createGroups(oneToOneRelations, manyToManyMappingSqls, oneToOneSqls, oneToManySqls);
    
    // 5、=====
    // TODO 注意，如果是强制更新数据表，那么就需要先做反向删除
    // 5.a 首先是主表
    Query sqlQuery = entityManager.createNativeQuery(sqlDdl.toString());
    sqlQuery.executeUpdate();
    // 5.b 然后是可能的OneToOne分组下的表
    for (String oneToOneSql : oneToOneSqls) {
      sqlQuery = entityManager.createNativeQuery(oneToOneSql);
      sqlQuery.executeUpdate();
    }
    // 5.c 然后是可能的OneToMany明细编辑表
    for (String oneToManySql : oneToManySqls) {
      sqlQuery = entityManager.createNativeQuery(oneToManySql);
      sqlQuery.executeUpdate();
    }
    // 5.d 最后是可能的ManyToMany映射表
    for (String manyToManyMappingSql : manyToManyMappingSqls) {
      sqlQuery = entityManager.createNativeQuery(manyToManyMappingSql);
      sqlQuery.executeUpdate();
    }
  }
  
  /**
   * 用于根据一般属性的描述集合，生成创建数据表时需要的一般字段脚本
   * @param properties 属性集合
   * @return 
   */
  private String createTableProperties(Set<TemplatePropertyEntity> properties) {
    Validate.isTrue(properties != null && !properties.isEmpty() , "在构建一般属性创建语句时，不允许传入空集合描述");
    StringBuilder sqlDdl = new StringBuilder();
    for (TemplatePropertyEntity templateProperty : properties) {
      Integer maxLen = templateProperty.getMaxLen();
      // 默认长度就是255
      if(maxLen == null) {
        maxLen = 255;
      }
      String propertyDbName = templateProperty.getPropertyDbName();
      String propertyClassName = templateProperty.getPropertyClassName();
      Boolean nullable = templateProperty.getNullable();
      Boolean unique = templateProperty.getUnique();
      Boolean primaryKey = templateProperty.getPrimaryKey();
      
      sqlDdl.append(this.buildSqlPropertyLine(propertyDbName, propertyClassName, maxLen, nullable)).append(System.getProperty(TAG_SEPARATOR));
      // 如果岗前字段有唯一性限制，还需要为其加载唯一键索引
      if(Boolean.TRUE.equals(unique) && !Boolean.TRUE.equals(primaryKey)) {
        sqlDdl.append(this.buildSqlUkLine(propertyDbName)).append(System.getProperty(TAG_SEPARATOR));
      }
    }
    
    return sqlDdl.toString();
  }
  
  /**
   * 该方法用于根据ManyToOne性质的关联属性，生成创建数据表时需要的关联字段脚本
   * @param relations ManyToOne性质的关联属性
   * @return
   */
  private String createManyToOneRelations(Set<TemplateRelationEntity> relations) {
    Validate.isTrue(relations != null && !relations.isEmpty() , "在构建ManyToOne关联属性创建语句时，不允许传入空集合描述");
    StringBuilder sqlDdl = new StringBuilder();
    for (TemplateRelationEntity templateRelationItem : relations) {
      String relationType = templateRelationItem.getRelationType();
      Validate.isTrue(StringUtils.equals(relationType, MESS_REF_MANYTOONE) , "发现关联类型错误的属性，请检查");
      
      String propertyDbName = templateRelationItem.getPropertyDbName();
      String propertyClassName = templateRelationItem.getPropertyClassName();
      String targetTableName = templateRelationItem.getTargetTableName();
      Validate.notBlank(targetTableName , "在处理关联关系时，发现`%s`（ManyToOne）没有填写目标数据表，请检查!!" , propertyDbName);
      
      if(StringUtils.isBlank(propertyClassName)) {
        propertyClassName = "java.lang.String";
      }
      Integer maxLen = 255;
      Boolean nullable = templateRelationItem.getNullable();
      // 目前默认目标数据表中的id就是它的主键
      String targetPkName = "id";
      
      // 首先建立这个字段的创建脚本，然后是为这个脚本添加索引
      sqlDdl.append(this.buildSqlPropertyLine(propertyDbName, propertyClassName, maxLen, nullable)).append(System.getProperty(TAG_SEPARATOR));
      sqlDdl.append(this.buildSqlFkLine(propertyDbName, targetTableName, targetPkName)).append(System.getProperty(TAG_SEPARATOR));
    }
    return sqlDdl.toString();
  }
  
  /**
   * 该方法用于根据ManyToMany性质的关联属性，生成创建数据表时所需的关联字段脚本，包括中间表的生成脚本。
   * @param resouceTableName
   * @param relations
   * @return
   */
  private void createManyToManyRelations(String resouceTableName , Set<TemplateRelationEntity> relations , List<String> mappingTableSqlDdls) {
    Validate.isTrue(relations != null , "在构建ManyToMany关联属性创建语句时，不允许传入空集合描述");
    for (TemplateRelationEntity templateRelationItem : relations) {
      StringBuilder sqlDdl = new StringBuilder();
      String relationType = templateRelationItem.getRelationType();
      Validate.isTrue(StringUtils.equals(relationType, MESS_REF_MANYTOMANY) , "发现关联类型错误的属性，请检查");
      
      String targetTableName = templateRelationItem.getTargetTableName();
      Validate.notBlank(targetTableName , "确定的ManyToMany关系，其目标数据表名必须填写，请检查数据表【%s】的设定!!" , resouceTableName);
      String targetPkName = "id";
      String resoucePkName = "id";
      // 为了保证一张表和另一张表的映射表不会出现 两个，
      // 基于resouceTableName和targetTableName创建的映射表名，需要进行排序
      List<String> sortedNames = Arrays.asList(resouceTableName , targetTableName).stream().sorted().collect(Collectors.toList());
      String mappingTable = String.format("%s_%s_mapping", sortedNames.get(0) , sortedNames.get(1));
      String mappingResourceKey = String.format("%s_%s", resouceTableName , resoucePkName);
      String mappingTargetKey = String.format("%s_%s", targetTableName , targetPkName);
      // FK有两个，命名分别为 FK_随机字符串
      String mappingResourceFk = UUID.randomUUID().toString();
      mappingResourceFk = StringUtils.join("FK_",mappingResourceFk).toUpperCase().replaceAll("\\-", "");
      String mappingTargetFk = UUID.randomUUID().toString();
      mappingTargetFk = StringUtils.join("FK_",mappingTargetFk).toUpperCase().replaceAll("\\-", "");
      
      // 根据源数据表和目前表数据表，建立映射表的脚本信息（包括关联信息）
      sqlDdl.append(OP_CREATE_TABLE + mappingTable + "` (").append(System.getProperty(TAG_SEPARATOR));
      sqlDdl.append("  `" + mappingResourceKey + "` varchar(255) NOT NULL,").append(System.getProperty(TAG_SEPARATOR));
      sqlDdl.append("  `" + mappingTargetKey + "` varchar(255) NOT NULL,").append(System.getProperty(TAG_SEPARATOR));
      sqlDdl.append("  PRIMARY KEY (`" + mappingResourceKey + "`,`" + mappingTargetKey + "`),").append(System.getProperty(TAG_SEPARATOR));
      sqlDdl.append("  CONSTRAINT `" + mappingResourceFk + "` FOREIGN KEY (`" + mappingResourceKey + "`) REFERENCES `" + resouceTableName + "` (`" + resoucePkName + "`),").append(System.getProperty(TAG_SEPARATOR));
      sqlDdl.append("  CONSTRAINT `" + mappingTargetFk + "` FOREIGN KEY (`" + mappingTargetKey + "`) REFERENCES `" + targetTableName + "` (`" + targetPkName + "`) ").append(System.getProperty(TAG_SEPARATOR));
      sqlDdl.append(MESS_REF_ENGINE).append(System.getProperty(TAG_SEPARATOR));
      mappingTableSqlDdls.add(sqlDdl.toString());
    }
  }
  
  /**
   * 该私有方法用于处理基于上级数据表parentTableName的OneToMany关联信息
   * @param parentTableName
   * @param oneToManyRelations
   * @param manyToManyMappingSqls
   * @return
   */
  private void createItems(Set<TemplateItemEntity> oneToManyRelations , List<String> manyToManyMappingSqls , List<String> oneToManySqls) {
    if(oneToManyRelations == null || oneToManyRelations.isEmpty()) {
      return;
    }
    for (TemplateItemEntity oneToManyItem : oneToManyRelations) {
      StringBuilder oneToManySqlDdl = new StringBuilder();
      String currentTableName = oneToManyItem.getTableName();
      Validate.notBlank(currentTableName , "在构建明细编辑项时，未发现明细编辑项对应的数据表名，请检查!!");
      oneToManySqlDdl.append(OP_CREATE_TABLE + currentTableName + "` (").append(System.getProperty(TAG_SEPARATOR));
      
      // 3.a、===========
      // 注意，这里要自行构建PK
      Set<TemplatePropertyEntity> oneToManyProperties = oneToManyItem.getProperties();
      oneToManySqlDdl.append(this.createTableProperties(oneToManyProperties)).append(System.getProperty(TAG_SEPARATOR));
      
      // 3.b和3.c======
      // 注意，构建ManyToOne关系时，要自行构建和主表的外键关联关系(并且要设定回溯关联)
      Set<TemplateRelationEntity> omRlations = oneToManyItem.getRelations();
      if(omRlations == null) {
        omRlations = new LinkedHashSet<>();
      }
      if(omRlations != null && !omRlations.isEmpty()) {
        Set<TemplateRelationEntity> manyToOneRelations = omRlations.stream().filter(item -> item.getRelationType().equals(MESS_REF_MANYTOONE)).collect(Collectors.toSet());
        if(manyToOneRelations != null && !manyToOneRelations.isEmpty()) {
          oneToManySqlDdl.append(this.createManyToOneRelations(manyToOneRelations)).append(System.getProperty(TAG_SEPARATOR));
        }
      }
      // 为主表关联建立外键，建立主键信息
      oneToManySqlDdl.append("  PRIMARY KEY (`id`) ").append(System.getProperty(TAG_SEPARATOR));
      oneToManySqlDdl.append(MESS_REF_ENGINE).append(System.getProperty(TAG_SEPARATOR));
      oneToManySqls.add(oneToManySqlDdl.toString());
      
      // 3.d===========
      if(omRlations != null && !omRlations.isEmpty()) {
        Set<TemplateRelationEntity> manyToManyRelations = omRlations.stream().filter(item -> item.getRelationType().equals(MESS_REF_MANYTOMANY)).collect(Collectors.toSet());
        if(manyToManyRelations != null && !manyToManyRelations.isEmpty()) {
          this.createManyToManyRelations(currentTableName, manyToManyRelations , manyToManyMappingSqls);
        }
      }
    }
  }
  
  /**
   * 该方法基于上层的resouceTableName数据表，完成其下的oneToOne关联的数据表构建
   * @param resouceTableName
   * @param oneToOneRelations
   * @param manyToManyMappingSqls
   * @return
   */
  private void createGroups(Set<TemplateGroupEntity> oneToOneRelations , List<String> manyToManyMappingSqls , List<String> oneToOneSqls , List<String> oneToManySqls) {
    if(oneToOneRelations != null && !oneToOneRelations.isEmpty()) {
      for (TemplateGroupEntity oneToOneGroup : oneToOneRelations) {
        StringBuilder oneToOneSqlDdl = new StringBuilder();
        String currentTableName = oneToOneGroup.getTableName();
        String parentTableName = oneToOneGroup.getParentTableName();
        Validate.notBlank(currentTableName , "在构建分组项时，未发现分组项对应的数据表名，请检查!!");
        Validate.notBlank(parentTableName , "在构建分组项时，未发现分组项对应的上层数据表名，请检查!!");
        oneToOneSqlDdl.append(OP_CREATE_TABLE + currentTableName + "` (").append(System.getProperty(TAG_SEPARATOR));
        
        // 4.a、===========
        Set<TemplatePropertyEntity> oneToOneProperties = oneToOneGroup.getProperties();
        if(oneToOneProperties != null && !oneToOneProperties.isEmpty()) {
          oneToOneSqlDdl.append(this.createTableProperties(oneToOneProperties)).append(System.getProperty(TAG_SEPARATOR));
        }
        
        // 4.b、=======
        Set<TemplateRelationEntity> ooRlations = oneToOneGroup.getRelations();
        if(ooRlations != null && !ooRlations.isEmpty()) {
          Set<TemplateRelationEntity> manyToOneRelations = ooRlations.stream().filter(item -> item.getRelationType().equals(MESS_REF_MANYTOONE)).collect(Collectors.toSet());
          if(manyToOneRelations != null && !manyToOneRelations.isEmpty()) {
            oneToOneSqlDdl.append(this.createManyToOneRelations(manyToOneRelations)).append(System.getProperty(TAG_SEPARATOR));
          }
        }
        // 建立主键信息
        oneToOneSqlDdl.append("  PRIMARY KEY (`id`) ").append(System.getProperty(TAG_SEPARATOR));
        oneToOneSqlDdl.append(MESS_REF_ENGINE).append(System.getProperty(TAG_SEPARATOR));
        oneToOneSqls.add(oneToOneSqlDdl.toString());
        
        // 4.c、=========
        if(ooRlations != null && !ooRlations.isEmpty()) {
          Set<TemplateRelationEntity> manyToManyRelations = ooRlations.stream().filter(item -> item.getRelationType().equals(MESS_REF_MANYTOMANY)).collect(Collectors.toSet());
          if(manyToManyRelations != null && !manyToManyRelations.isEmpty()) {
            this.createManyToManyRelations(currentTableName, manyToManyRelations , manyToManyMappingSqls);
          }
        }
        
        // 4.d、=========
        Set<TemplateItemEntity> currentOneToManyRelations = oneToOneGroup.getItemRelations();
        this.createItems(currentOneToManyRelations, manyToManyMappingSqls , oneToManySqls);
      }
    }
  }
  
  private String buildSqlFkLine(String propertyDbName , String targetTableName , String targetPropertyDbName) {
    String fkLine = null;
    // 外键名采用格式为：完全随机名
    String currentUuid = UUID.randomUUID().toString();
    String fkName = StringUtils.join("FK_",currentUuid).toUpperCase().replaceAll("\\-", "");
    fkLine = String.format("  CONSTRAINT `%s` FOREIGN KEY (`%s`) REFERENCES `%s` (`%s`) , ", fkName , propertyDbName , targetTableName , targetPropertyDbName);
    return fkLine;
  }


  /**
   * 该属性为当前指定数据表中的指定字段，创建UK唯一键索引
   * @param tableName 指定的数据表(当前数据表)
   * @param propertyDbName 指定的数据库字段
   * @return
   */
  private String buildSqlUkLine(String propertyDbName) {
    // UK唯一键命名的方式为：完全随机名
    String currentUuid = UUID.randomUUID().toString();
    String ukName = StringUtils.join("UK_",currentUuid).toUpperCase().replaceAll("\\-", "");
    return String.format("  UNIQUE KEY `%s` (`%s`),", ukName , propertyDbName);
  }


  private String buildSqlPropertyLine(String propertyDbName , String propertyClassName , Integer maxlen , boolean nullable) {
    String field;
    String dbType = classTypeMapping.get(propertyClassName);
    if(StringUtils.equals(dbType, MESS_REF_VARCHAR)) {
      field = String.format(" `%s` %s (%d) ", propertyDbName , dbType , maxlen);
    } else {
      field = String.format(" `%s` %s ", propertyDbName , dbType);
    }
    
    // 是否为null的判定
    if(nullable) {
      field = StringUtils.join(field, " NULL , " );
    } else {
      field = StringUtils.join(field, " NOT NULL , " );
    }

    return field;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.table.TableOperateRepositoryCustom#existTableName(java.lang.String)
   */
  @Override
  public Boolean existTableName(String tableName) {
    /*
     * 直接按照数据表名称查询即可
     * 如果没有抛异常，说明已经存在这个数据表
     * */
    Validate.notBlank(tableName , "数据表的名字必须传入!!");
    String querySql = " show tables like ?1 ;";
    Query query = this.entityManager.createNativeQuery(querySql);
    query.setParameter(1, tableName);
    List<?> result = query.getResultList();
    return !CollectionUtils.isEmpty(result);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.table.TableOperateRepositoryCustom#upgradeTable(com.bizunited.platform.kuiper.entity.TemplateEntity)
   */
  @Override
  public void upgradeTable(TemplateEntity sourceTemplate , TemplateEntity targetTemplate) {
    /* 
     * 操作的基本原则是：
     * 1、没有id的字段描述就是新增的字段（无论是一般性字段还是关联性字段）
     * 2、需要逐个检查字符串字段的长度、必填字段的“必填性指定”是否进行了修改
     * 
     * 那么操作步骤如下：
     * 1、比较主表中的基本属性、关联属性，并形成针对主表的数据表修改语句
     * 2、比较各个分组表中的基本属性、关联属性，并形成针对分主表的修改语句
     * 3、比较个分组表中的明细表 其中的基本属性、关联属性，并形成针对明细表的修改语句
     * 4、比较各个明细表，其中的基本属性、关联属性，并形成针对明细表的修改语句
     * 5、开始执行这些SQL语句
     *   5.1、首先从主表开始执行
     *   5.2、然后是分组表
     *   5.3、然后是明细表
     * */
    // 1、======
    // 主表的修改语句存储在这里
    List<String> mainTableUpgradeSqls = new LinkedList<>();
    // 各映射表的修改语句存储在这里
    List<String> mappingTableUpgradeSqls = new LinkedList<>();
    Set<TemplatePropertyEntity> sourceProperties = sourceTemplate.getProperties();
    Set<TemplateRelationEntity> sourceRelations = sourceTemplate.getRelations();
    if(sourceRelations == null) {
      sourceRelations = Sets.newHashSet();
    }
    Set<TemplatePropertyEntity> targetProperties = targetTemplate.getProperties();
    Set<TemplateRelationEntity> targetRelations = targetTemplate.getRelations();
    if(targetRelations == null) {
      targetRelations = Sets.newHashSet();
    }
    String resouceTableName = targetTemplate.getTableName();
    Validate.notBlank(resouceTableName , "错误的数据表信息，请检查!!");
    this.compareProperties(resouceTableName, sourceProperties, targetProperties, mainTableUpgradeSqls);
    this.compareRelations(resouceTableName, sourceRelations, targetRelations, mainTableUpgradeSqls, mappingTableUpgradeSqls);
    
    // 2和3、======
    // 分组表自身的修改信息放在这里
    List<String> groupAlterTableSqls = new LinkedList<>();
    // 分组信息的新增、修改sql放在这里
    List<String> oneToOneSqls = new LinkedList<>();
    // 分钟明细表的新增、修改sql放在这里
    List<String> oneToManySqls = new LinkedList<>();
    Set<TemplateGroupEntity> sourceGroups = sourceTemplate.getGroupRelations();
    if(sourceGroups == null) {
      sourceGroups = Sets.newHashSet();
    }
    Set<TemplateGroupEntity> targetGroups = targetTemplate.getGroupRelations();
    Set<String> sourceGroupTableNames = sourceGroups.stream().map(TemplateGroupEntity::getTableName).collect(Collectors.toSet());
    Set<String> targetGroupTableNames = targetGroups.stream().map(TemplateGroupEntity::getTableName).collect(Collectors.toSet());
    // 验证targetGroupTableNames中的所有分组表信息和sourceGroupTableNames中的元素是包含关系
    Validate.isTrue(Sets.difference(sourceGroupTableNames, targetGroupTableNames).isEmpty() , "不允许进行分组信息的删除，请检查!!");
    // 转成Mapping，后续好调用
    Map<String, TemplateGroupEntity> sourceMappingGroups = sourceGroups.stream().collect(Collectors.toMap(TemplateGroupEntity::getTableName , item-> item));
    if(targetGroups != null && !targetGroups.isEmpty()) {
      for (TemplateGroupEntity targetGroup : targetGroups) {
        String groupTableName = targetGroup.getTableName();
        TemplateGroupEntity sourceGroup = sourceMappingGroups.get(groupTableName);
        this.compareGroups(groupTableName, sourceGroup, targetGroup, groupAlterTableSqls, mappingTableUpgradeSqls, oneToOneSqls, oneToManySqls);
      }
    }
    
    // 4、======
    // 明细表自身的修改信息放在这里
    List<String> itemAlterTableSqls = new LinkedList<>();
    Set<TemplateItemEntity> sourceTemplateItems = sourceTemplate.getItemRelations();
    if(sourceTemplateItems == null) {
      sourceTemplateItems = Sets.newHashSet();
    }
    Set<TemplateItemEntity> targetTemplateItems = targetTemplate.getItemRelations();
    Set<String> sourceItemsTableNames = sourceTemplateItems.stream().map(TemplateItemEntity::getTableName).collect(Collectors.toSet());
    Set<String> targetItemsTableNames = targetTemplateItems.stream().map(TemplateItemEntity::getTableName).collect(Collectors.toSet());
    // 验证targetItemsTableNames明细表明和sourceItemsTableNames中的元素是包含关系
    Validate.isTrue(Sets.difference(sourceItemsTableNames, targetItemsTableNames).isEmpty() , "不允许进行明细编辑项信息的删除，请检查!!");
    // 转成mapping，后面好使用
    Map<String, TemplateItemEntity> sourceMappingItems = sourceTemplateItems.stream().collect(Collectors.toMap(TemplateItemEntity::getTableName, item -> item));
    if(targetTemplateItems != null && !targetTemplateItems.isEmpty()) {
      for (TemplateItemEntity targetItem : targetTemplateItems) {
        String tableName = targetItem.getTableName();
        TemplateItemEntity sourceItem = sourceMappingItems.get(tableName);
        this.compareItems(tableName, sourceItem, targetItem, itemAlterTableSqls, mappingTableUpgradeSqls, oneToManySqls);
      }
    }
    
    // 5、=====开始执行
    // mainTableUpgradeSqls
    for (String mainTableUpgradeSql : mainTableUpgradeSqls) {
      Query sqlQuery = entityManager.createNativeQuery(mainTableUpgradeSql);
      sqlQuery.executeUpdate();
    }
    // itemAlterTableSqls
    for (String itemAlterTableSql : itemAlterTableSqls) {
      Query sqlQuery = entityManager.createNativeQuery(itemAlterTableSql);
      sqlQuery.executeUpdate();
    }
    // groupAlterTableSqls
    for (String groupAlterTableSql : groupAlterTableSqls) {
      Query sqlQuery = entityManager.createNativeQuery(groupAlterTableSql);
      sqlQuery.executeUpdate();
    }
    // oneToOneSqls
    for (String oneToOneSql : oneToOneSqls) {
      Query sqlQuery = entityManager.createNativeQuery(oneToOneSql);
      sqlQuery.executeUpdate();
    }
    // oneToManySqls
    for (String oneToManySql : oneToManySqls) {
      Query sqlQuery = entityManager.createNativeQuery(oneToManySql);
      sqlQuery.executeUpdate();
    }
    // mappingTableUpgradeSqls
    for (String mappingTableUpgradeSql : mappingTableUpgradeSqls) {
      Query sqlQuery = entityManager.createNativeQuery(mappingTableUpgradeSql);
      sqlQuery.executeUpdate();
    }
  }

  @Override
  @Transactional
  public void executeNativeSQL(String sql) {
    Query sqlQuery = entityManager.createNativeQuery(sql);
    sqlQuery.executeUpdate();


  }

  @Override
  public List<?> executeQuerySql(String querySql) {
    Query query = entityManager.createNativeQuery(querySql);
    return query.getResultList();
  }


  /**
   * 比较两个“数据表名”相同的分组信息，并按照比较的情况，生成基于当前分组表的更新SQL，这有可能就包括了整个分组的新增SQL
   * @param resouceTableName 上级数据表（实际上就是主表）
   * @param sourceGroup 源分组数据表
   * @param targetGroup 目标分组数据表
   * @param alterTablesSqls
   */
  private void compareGroups(String resouceTableName , TemplateGroupEntity sourceGroup , TemplateGroupEntity targetGroup , List<String> groupAlterTableSqls , List<String> manyToManyMappingSqls , List<String> oneToOneSqls , List<String> oneToManySqls) {
    /*
     * 一、可能的情况包括没有原始分组（sourceGroup），这种情况下，这个targetGroup就是一个新增的分组
     * 那么其下的的一般属性、关联属性、以及可能的分组下明细 都需要新增
     * 这个分组也要新增
     * 
     * 二、如果有原始分组、那么这个分组下的一般属性、关联属性就要进行可能的修改、也要进行分组下可能的明细表的各属性修改
     * 1、首先验证一般属性和关联属性的正确性
     * 2、为当前分组表下的一般属性和关联属性确认字段变更的SQL脚本
     * 3、处理当前分组表下可能存在的明细编辑表
     */
    Validate.notNull(targetGroup , "不支持对一个已有分组信息进行删除，请检查主表【%s】下的分组设定" , resouceTableName);
    // 如果条件成立，说明是第一种情况
    if(sourceGroup == null) {
      this.createGroups(Sets.newHashSet(targetGroup), manyToManyMappingSqls, oneToOneSqls, oneToManySqls);
      return;
    }
    // 剩下的情况就是第二种情况
    // 1、=======
    Set<TemplatePropertyEntity> sourceProperties = sourceGroup.getProperties();
    Set<TemplateRelationEntity> sourceRelations = sourceGroup.getRelations();
    if(sourceRelations == null) {
      sourceRelations = Sets.newHashSet();
    }
    Set<TemplatePropertyEntity> targetProperties = targetGroup.getProperties();
    Set<TemplateRelationEntity> targetRelations = targetGroup.getRelations();
    if(targetRelations == null) {
      targetRelations = Sets.newHashSet();
    }
    // 2、=======分组表下的一般属性和关联属性
    this.compareProperties(resouceTableName, sourceProperties, targetProperties, groupAlterTableSqls);
    this.compareRelations(resouceTableName, sourceRelations, targetRelations, groupAlterTableSqls, manyToManyMappingSqls);
    
    // 3、=======然后是分组表下可能的明细属性
    Set<TemplateItemEntity> sourceItems = sourceGroup.getItemRelations();
    Set<TemplateItemEntity> targetItems = targetGroup.getItemRelations();
    if((sourceItems == null || sourceItems.isEmpty())
        && (targetItems == null || targetItems.isEmpty())) {
      return;
    }
    if(sourceItems == null) {
      sourceItems = Sets.newHashSet();
    }
    // 验证targetItems集合中的元素包含sourceItems集合中的所有元素
    Set<String> sourceTableNameItems = sourceItems.stream().map(TemplateItemEntity::getTableName).collect(Collectors.toSet());
    Set<String> targetTableNameItems = targetItems.stream().map(TemplateItemEntity::getTableName).collect(Collectors.toSet());
    Validate.isTrue(Sets.difference(sourceTableNameItems, targetTableNameItems).isEmpty() , "不允许对分组下的明细编辑项进行删除操作，请检查数据表【%s】下的明细编辑项设定" , resouceTableName);
    // 做成mapping后续方便调用
    Map<String , TemplateItemEntity> sourceMappingItems = sourceItems.stream().collect(Collectors.toMap(TemplateItemEntity::getTableName, item -> item));
    for (TemplateItemEntity targetItem : targetItems) {
      String tableName = targetItem.getTableName();
      TemplateItemEntity sourceItem = sourceMappingItems.get(tableName);
      this.compareItems(tableName, sourceItem, targetItem, groupAlterTableSqls, manyToManyMappingSqls, oneToManySqls);
    }
  } 
  
  /**
   * 比较两个“数据表名”相同的明细信息，并按照比较的情况，生成基于当前明细表的更新SQL，这有可能就包括了整个明细的新增SQL
   * @param resouceTableName
   * @param sourceItem
   * @param targetItem
   * @param alterTablesSqls
   * @param manyToManyMappingSqls
   * @param oneToManySqls
   */
  private void compareItems(String resouceTableName , TemplateItemEntity sourceItem , TemplateItemEntity targetItem , List<String> alterTablesSqls , List<String> manyToManyMappingSqls , List<String> oneToManySqls) {
    /*
     * 可能的情况包括：
     * 1、没有当前明细表对应的sourceItem，这种情况就是新增当前的targetItem
     * 包括其下的一般属性和关联属性
     * 2、存在sourceItem和同一明细表名称的targetItem，这种情况下就是修改信息
     * 包括修改一般属性和关联属性
     * */
    // 1、======
    // 如果条件成立，则是情况1
    if(sourceItem == null) {
      this.createItems(Sets.newHashSet(targetItem), manyToManyMappingSqls, oneToManySqls);
      return;
    }
    // 剩下的只可能是情况2了
    Set<TemplatePropertyEntity> sourceProperties = sourceItem.getProperties();
    Set<TemplateRelationEntity> sourceRelations = sourceItem.getRelations();
    if(sourceRelations == null) {
      sourceRelations = Sets.newHashSet();
    }
    Set<TemplatePropertyEntity> targetProperties = targetItem.getProperties();
    Set<TemplateRelationEntity> targetRelations = targetItem.getRelations();
    if(targetRelations == null) {
      targetRelations = Sets.newHashSet();
    }
    this.compareProperties(resouceTableName, sourceProperties, targetProperties, alterTablesSqls);
    this.compareRelations(resouceTableName, sourceRelations, targetRelations, alterTablesSqls, manyToManyMappingSqls);
  }
  
  /**
   * 比较两个“一般属性集合”的差异，并按照比较的情况，生成基于resouceTableName数据源字段更新的SQL语句（目前仅支持MySQL）
   * @param resouceTableName 依据的数据表
   * @param sourceProperties 源一般字段
   * @param targetProperties 目标一般字段
   */
  private void compareProperties(String resouceTableName , Set<TemplatePropertyEntity> sourceProperties , Set<TemplatePropertyEntity> targetProperties , List<String> alterTablesSqls) {
    /*
     * 需要比较的信息，包括：
     * 1、依据sourceProperties中存在的String类型的字段，验证其长度是否变化
     * 如果变化，则生成相关长度变化的脚本（只能增加长度，不能减少长度）
     * 2、依据sourceProperties中存在的，not null的字段，验证其是否改变，
     * 如果变化，则生成相关“可为null”的变化脚本（只能由not null变为 null，不能反过来）
     * 3、依据targetProperties中和sourceProperties中的差集，生成新增数据字段的脚本
     * （为了保证之前的业务表单实例可用，新增的字段，都可以为null）
     * */
    Validate.notBlank(resouceTableName , "错误的数据表名信息，请检查!!");
    Set<String> targetPropertyDbNames = targetProperties.stream().map(TemplatePropertyEntity::getPropertyDbName).collect(Collectors.toSet());
    Set<String> sourcePropertyDbNames = sourceProperties.stream().map(TemplatePropertyEntity::getPropertyDbName).collect(Collectors.toSet());
    // 验证targetProperties中的元素和sourceProperties中的元素存在包含关系
    Validate.isTrue(Sets.difference(sourcePropertyDbNames, targetPropertyDbNames).isEmpty() , "在调整分组表【%s】时，不允许删除已有的一般属性,请检查!!" , resouceTableName);
    // 换成有序的K-V结构，利于后面使用
    Map<String, TemplatePropertyEntity> targetPropertiesMapping = targetProperties.stream().collect(Collectors.toMap(TemplatePropertyEntity::getPropertyDbName, item -> item));
    
    // 1、===== 
    List<TemplatePropertyEntity> sourceStringProperties = sourceProperties.stream().filter(item -> StringUtils.equals(item.getPropertyClassName(), "java.lang.String")).collect(Collectors.toList());
    for (TemplatePropertyEntity sourceStringProperty : sourceStringProperties) {
      Integer sourceMaxLen = sourceStringProperty.getMaxLen();
      if(sourceMaxLen == null) {
        sourceMaxLen = 255;
      }
      String propertyDbName = sourceStringProperty.getPropertyDbName();
      TemplatePropertyEntity targetTemplateProperty = targetPropertiesMapping.get(propertyDbName);
      Integer targetMaxLen = targetTemplateProperty.getMaxLen();
      if(targetMaxLen == null) {
        targetMaxLen = 255;
      }
      
      // 如果验证不成立，说明长度变小了，则抛出错误
      Validate.isTrue(sourceMaxLen <= targetMaxLen , "在验证%s[%s]属性时，发现错误的长度变化，请检查" , propertyDbName , resouceTableName);
      // 如果条件成立，则说明长度变大了，需要生成SQL
      if(targetMaxLen > sourceMaxLen) {
        String alterSql = String.format("alter table %s modify column %s varchar(%d);", resouceTableName , propertyDbName , targetMaxLen);
        alterTablesSqls.add(alterSql);
      }
    }
    
    // 2、====
    List<TemplatePropertyEntity> sourceNotNullProperties = sourceProperties.stream().filter(item -> !item.getNullable()).collect(Collectors.toList());
    for(TemplatePropertyEntity sourceNotNullProperty : sourceNotNullProperties) {
      String propertyDbName = sourceNotNullProperty.getPropertyDbName();
      boolean sourceNullable = sourceNotNullProperty.getNullable();
      TemplatePropertyEntity targetTemplateProperty = targetPropertiesMapping.get(propertyDbName);
      boolean targetNullable = targetTemplateProperty.getNullable();
      String sourcePropertyClassName = sourceNotNullProperty.getPropertyClassName();
      String targetPropertyClassName = targetTemplateProperty.getPropertyClassName();
      Validate.isTrue(StringUtils.equals(sourcePropertyClassName, targetPropertyClassName) , "数据表的字段类型不允许修改，请检查%s【%s】" , propertyDbName , resouceTableName);
      Integer maxlen = targetTemplateProperty.getMaxLen();
      if(maxlen == null) {
        maxlen = 255;
      }
      String dbType = classTypeMapping.get(sourcePropertyClassName);
      
      // 如果验证成立，则说明可为null的变成了 not null的，报出错误
      Validate.isTrue(!(sourceNullable && !targetNullable) , "为null的字段，不能修改为not null，请检查%s[%s]" , propertyDbName , resouceTableName);
      // 如果条件成立，说明需要进行not null 到 null的更新操作
      if(!sourceNullable && targetNullable) {
        String alterSql = null;
        if(StringUtils.equals(dbType, MESS_REF_VARCHAR)) {
          alterSql = String.format("alter table %s modify column %s %s (%d) null ;", resouceTableName , propertyDbName , dbType , maxlen);
        } else {
          alterSql = String.format("alter table %s modify column %s %s null ;", resouceTableName , propertyDbName , dbType);
        }
        alterTablesSqls.add(alterSql);
      }
    }
    
    // 3、======
    // 这个集合里面，就是新增的一般属性字段
    SetView<String> diffTargetPropertyDbNames = Sets.difference(targetPropertyDbNames, sourcePropertyDbNames);
    for (String diffTargetPropertyDbName : diffTargetPropertyDbNames) {
      TemplatePropertyEntity targetProperty = targetPropertiesMapping.get(diffTargetPropertyDbName);
      String targetPropertyClassName = targetProperty.getPropertyClassName();
      String propertyDbName = targetProperty.getPropertyDbName();
      Integer maxlen = targetProperty.getMaxLen();
      if(maxlen == null) {
        maxlen = 255;
      }
      
      String dbType = classTypeMapping.get(targetPropertyClassName);
      String alterSql = null;
      if(StringUtils.equals(dbType, MESS_REF_VARCHAR)) {
        alterSql = String.format("alter table %s add %s %s (%d) null ;", resouceTableName , propertyDbName , dbType , maxlen);
      } else {
        alterSql = String.format("alter table %s add %s %s null ;", resouceTableName , propertyDbName , dbType);
      }
      alterTablesSqls.add(alterSql);
    }
  }
  
  /**
   * 比较两个“关联属性集合”的差异，并按照比较的情况，生成基于resouceTableName数据源字段更新的SQL语句（目前仅支持MySQL）
   * @param resouceTableName 指定的数据表
   * @param sourceRelations 源——关联字段
   * @param targetRelations 目标——关联字段
   * @param alterTablesSqls
   */
  private void compareRelations(String resouceTableName , Set<TemplateRelationEntity> sourceRelations , Set<TemplateRelationEntity> targetRelations , List<String> alterTablesSqls , List<String> mappingTableSqls) {
    /*
     * 需要比较的信息：
     * 1、判定当前字段的差异，取得这些差异
     * 如果不选在任何差异，就无需处理了，因为关联性字段的比较中，已有的关联信息是不允许变化的
     * 2、如果当前ManyToOne关系存在不同，则为当前数据表生成新字段和新外键
     * 3、如果挡墙ManyToMany关系存在不同，则为当前数据表生成Mapping映射表和外键，注意需要检查这个映射表的存在性，
     * */
    // 1、======
    if(sourceRelations == null || sourceRelations.isEmpty()) {
      return;
    }
    if(targetRelations == null || targetRelations.isEmpty()) {
      return;
    }
    if(!sourceRelations.isEmpty() && targetRelations.isEmpty()) {
      throw new IllegalArgumentException(String.format("不支持对关联属性进行删除，请检查数据表【%s】中的关联属性设定!!" , resouceTableName));
    }
    Set<String> targetRelationNames = targetRelations.stream().map(TemplateRelationEntity::getPropertyName).collect(Collectors.toSet());
    Set<String> sourceRelationNames = sourceRelations.stream().map(TemplateRelationEntity::getPropertyName).collect(Collectors.toSet());
    SetView<String> targetDiddRelations = Sets.difference(targetRelationNames, sourceRelationNames);
    SetView<String> sourceDiddRelations = Sets.difference(sourceRelationNames, targetRelationNames);
    Validate.isTrue(sourceDiddRelations.isEmpty() , "进行关联属性修改时，只能新增关联信息，不能删除或更改已有的关联，请检查数据表【%s】" , resouceTableName);
    if(targetDiddRelations.isEmpty()) {
      return;
    }
    
    // 2、======
    List<TemplateRelationEntity> diffManyToOneRelations = targetRelations.stream().filter(item -> targetDiddRelations.contains(item.getPropertyName()) && StringUtils.equals(item.getRelationType(), MESS_REF_MANYTOONE)).collect(Collectors.toList());
    if(diffManyToOneRelations != null) {
      for (TemplateRelationEntity diffRelation : diffManyToOneRelations) {
        String propertyClassName = diffRelation.getPropertyClassName();
        if(StringUtils.isBlank(propertyClassName)) {
          propertyClassName = "java.lang.String";
        }
        String dbType = classTypeMapping.get(propertyClassName);
        String propertyDbName = diffRelation.getPropertyDbName();
        Integer maxlen = 255;
        String targetTableName = diffRelation.getTargetTableName();
        Validate.notBlank(targetTableName , "字段%s【%s】的目标数据表不能为空，请检查" , propertyDbName , resouceTableName);
        
        // 先建立属性更新的语句
        String propertyAlterPropertySql = null;
        if(StringUtils.equals(propertyClassName, "java.lang.String")) {
          propertyAlterPropertySql = String.format(" alter table %s add %s %s (%d) null;", resouceTableName , propertyDbName , dbType , maxlen);
        } else {
          propertyAlterPropertySql = String.format(" alter table %s add %s %s null;", resouceTableName , propertyDbName , dbType);
        }
        alterTablesSqls.add(propertyAlterPropertySql);
        // 然后建立关联外键的语句
        String fkName = String.format("FK_%s_%s", resouceTableName , propertyDbName);
        String propertyAlterUkSql = String.format("alter table %s add constraint `%s` foreign key (`%s`) references %s (`id`);", resouceTableName , fkName , propertyDbName , targetTableName);
        alterTablesSqls.add(propertyAlterUkSql);
      }
    }
    
    // 3、====
    Set<TemplateRelationEntity> diffManyToManyRelations = targetRelations.stream().filter(item -> targetDiddRelations.contains(item.getPropertyName()) && StringUtils.equals(item.getRelationType(), MESS_REF_MANYTOMANY)).collect(Collectors.toSet());
    if(diffManyToManyRelations != null && !diffManyToManyRelations.isEmpty()) {
      // 重用方法，建立关联数据表
      this.createManyToManyRelations(resouceTableName, diffManyToManyRelations, mappingTableSqls);
    }
  }
}