package com.bizunited.platform.kuiper.starter.service;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.kuiper.entity.ListTemplateEntity;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.security.Principal;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 列表模板管理业务模型的服务层接口定义
 * @author yinwenjie
 */
public interface ListTemplateService { 
  /**
   * 初始化一个新的列表模板模型对象，初始化的列表模板将只包括基本信息，不包括列表模型JSON/XML形式的内容描述
   */
  public ListTemplateEntity init(ListTemplateEntity listTemplate , Principal principal);
  /**
   * 更新一个已有的列表模板模型对象，修改的列表模板将只包括基本信息，不包括列表模型JSON/XML形式的内容描述
   */
  public ListTemplateEntity update(ListTemplateEntity listTemplate);
  /**
   * 按照列表模板的数据层编号，更新这个列表模板的布局内容（JSON描述结构）
   * @param id 指定的列表模板编号
   * @param content 需要在文件中保存的JSON形式
   */
  public ListTemplateEntity updateContentById(String id , JSONObject content);
  /**
   * 按照指定的列表模板数据层编号，查询这个列表模板的布局内容（JSON描述结构）
   * @param id 指定的列表模板编号
   * @return 返回的json描述内容
   */
  public JSONObject findContentById(String id);
  /**
   * 按照主键进行详情查询（包括关联信息）
   * @param id 主键
   */
  public ListTemplateEntity findDetailsById(String id);
  /**
   * 按照列表模板的编码进行查询，并按照创建时间顺序进行反序排列
   * @param code 指定的编码
   * @return
   */
  public Set<ListTemplateEntity> findByCode(String code);
  /**
   * 按照ListTemplateEntity的主键编号，查询指定的数据信息（不包括任何关联信息）
   * @param id 主键
   * */
  public ListTemplateEntity findById(String id);
  /**
   * 这里是针对列表模板的数据库层分页查询定义
   * @param pageable 分页信息
   * @param conditions 查询条件现在包括编号、版本、中文名信息
   */
  public Page<ListTemplateEntity> findByConditions(Pageable pageable , Map<String, Object> conditions);
  /**
   * 查询一批指定的id编号的列表模板信息在系统中真实存在的数量
   * @param ids
   * @return
   */
  public int countByIds(String[] ids);
  /**
   *  按照主键进行信息的真删除 
   * @param id 主键 
   */
  public void deleteById(String id);

  /**
   * 按照列表模板的编码和版本号进行查询
   * @param code 列表编码
   * @param cversion 列表编码的版本号
   * @return
   */
  JSONObject findContentByCodeAndCversion(String code, String cversion);

  /**
   * 按照列表模板的编码，更新这个列表模板的布局内容（JSON描述结构）
   * @param code 列表编码
   * @param cversion 列表编码的版本号
   * @param content 需要在文件中保存的JSON形式
   * @return
   */
  ListTemplateEntity updateContentByCodeAndCversion(String code, String cversion, JSONObject content);


  /**
   * 按照列表模板的编码和版本号查询默认版本,返回这个列表模板的布局内容（JSON描述结构）
   * @param code 列表编码
   * @return
   */
  JSONObject findDefaultContentByCode(String code);

  /**
   * 根据表编码和列表编码的版本号更新列表模板版本号
   * @param code 列表编码
   * @param cversion 列表编码的版本号
   */
  void updateDefaultVersion(String code, String cversion);

  /**
   * 按照列表模板的编码和版本号查询默认版本
   * @param code 列表编码
   * @return
   */
  ListTemplateEntity findDefaultByCode(String code);

  /**
   * 根据条件查询所有模版
   * @param condition
   * @return
   */
  List<ListTemplateEntity> findAllByConditions(ListTemplateEntity condition);

  /**
   * 根据ID查询数据
   * @param ids
   * @return
   */
  List<ListTemplateEntity> findByIds(String[] ids);

  /**
   * 导入列表模版数据
   * @param importModel
   */
  void importData(MigrateImportModel importModel);
}