package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateEventEntity;
import com.bizunited.platform.kuiper.entity.TemplateLayoutEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.starter.repository.TemplateRepository;
import com.bizunited.platform.kuiper.starter.service.TemplateEventService;
import com.bizunited.platform.kuiper.starter.service.TemplateLayoutService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService;
import com.bizunited.platform.saturn.model.PersistentProperty;
import com.bizunited.platform.saturn.model.PersistentRelation;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.security.Principal;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * 这个抽象类，负责对静态模板和动态模板的处理过程中相同的处理步骤进行统一描述
 * 以便减少两个类中的重复编码块
 * @author yinwenjie
 */
public abstract class AbstractTemplateService {
  @Autowired
  private UserService userService;
  @Autowired
  private TemplateRepository templateRepository;
  @Autowired
  private TemplateLayoutService templateLayoutService;
  @Autowired
  private TemplateEventService templateEventService;
  @Autowired
  private TemplateVisibilityService templateVisibilityService;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 保存模板基本信息——只包括基本信息
   * @param template
   * @param principal
   */
  protected void saveTemplate(TemplateEntity template  , Principal principal) {
    Date currentTime = new Date();
    template.setCreateTime(currentTime);
    template.setModifyTime(currentTime);
    template.setDefaultVersion(false);
    template.setDomain(template.getDomain() == null?"":template.getDomain());
    template.setProjectName(platformContext.getAppName());
    template.setProjectName(template.getProjectName() == null?"":template.getProjectName());
    String account = principal.getName();
    UserVo creator = userService.findByAccount(account);
    Validate.notNull(creator , "没有找到任何创建者信息，请检查!!");
    template.setCreator(account);
    this.templateRepository.save(template);
    this.templateRepository.flush();
  }
  
  /**
   * 该上层方法用于对模板结构进行升级操作
   * @param templateId
   * @param newTemplateId
   * @param newTemplate
   */
  protected void upgradeTemplateLayout(String templateId , String newTemplateId , TemplateEntity newTemplate) {
    try {
      // 可能的PC端布局
      JSONObject templateLayout = this.templateLayoutService.findDetailsByTemplateId(templateId , 1);
      if(templateLayout != null) {
        TemplateLayoutEntity newLayout = this.templateLayoutService.save(newTemplateId , 1 , templateLayout);
        newTemplate.setTemplateLayout(newLayout);
      }
      // 可能的移动端布局
      templateLayout = this.templateLayoutService.findDetailsByTemplateId(templateId , 2);
      if(templateLayout != null) {
        TemplateLayoutEntity newLayout = this.templateLayoutService.save(newTemplateId , 2 , templateLayout);
        newTemplate.setMobileTemplateLayout(newLayout);
      }
      // 可能的打印端布局
      templateLayout = this.templateLayoutService.findDetailsByTemplateId(templateId , 3);
      if(templateLayout != null) {
        TemplateLayoutEntity newLayout = this.templateLayoutService.save(newTemplateId , 3 , templateLayout);
        newTemplate.setPrintTemplateLayout(newLayout);
      }
    } catch (IOException e) {
      throw new IllegalArgumentException(e);
    }
  }
  
  /**
   * 该上层方法用于对模板事件进行升级操作
   * @param templateId
   * @param newTemplateId
   * @param newTemplate
   */
  protected void upgradeTemplateEvent(String templateId , String newTemplateId , TemplateEntity newTemplate) {
    List<TemplateEventEntity> templateEvents = this.templateEventService.findByTemplateId(templateId);
    if(templateEvents != null && !templateEvents.isEmpty()) {
      Set<TemplateEventEntity> newTemplateEvents = this.templateEventService.upgrade(newTemplateId, templateEvents);
      newTemplate.setEvent(newTemplateEvents);
    }
  }
  
  /**
   * 该上层方法用于对模板可见性进行升级操作
   * @param templateId
   * @param newTemplateId
   * @param newTemplate
   */
  protected void upgradeTemplateVisibility(String templateId , String newTemplateId , TemplateEntity newTemplate) {
    Set<TemplateVisibilityEntity> templateVisibilities = this.templateVisibilityService.findDetailsByTemplateId(templateId);
    if(templateVisibilities != null && !templateVisibilities.isEmpty()) {
      Set<TemplateVisibilityEntity> newTemplateVisibilities = this.templateVisibilityService.upgrade(newTemplateId, templateVisibilities);
      newTemplate.setVisibility(newTemplateVisibilities);
    }
  }

  /**
   * 初始化模版依赖实体信息
   * @param persistentRelationItem
   * @param parentClassName
   * @return
   */
  protected TemplateRelationEntity initTemplateRelationEntity(PersistentRelation persistentRelationItem, String parentClassName) {
    TemplateRelationEntity relationModel = new TemplateRelationEntity();
    relationModel.setCanInsert(persistentRelationItem.getCanInsert());
    relationModel.setCanUpdate(persistentRelationItem.getCanUpdate());
    relationModel.setIndex(persistentRelationItem.getIndex());
    relationModel.setNullable(persistentRelationItem.getNullable());
    relationModel.setPropertyClassName(persistentRelationItem.getPropertyClass());
    String propertyDbName = persistentRelationItem.getPropertyDbName() == null?"":persistentRelationItem.getPropertyDbName();
    Validate.isTrue(propertyDbName.length() <= 64 , "如填写字段的数据库属性[%s]，其长度应该小于64个字符，请检查!!" , propertyDbName);
    relationModel.setPropertyDbName(propertyDbName);
    relationModel.setPropertyDesc(persistentRelationItem.getPropertyDesc());
    String propertyName = persistentRelationItem.getPropertyName();
    Validate.notBlank(propertyName , "明细项属性不能为空，请检查!!");
    Validate.isTrue(propertyName.length() <= 64 , "属性名[%s]的长度需要小于64个字符，请检查!!" , propertyName);
    relationModel.setPropertyName(propertyName);
    relationModel.setRelationType(persistentRelationItem.getRelationType().toString());
    // 确实是否是回溯模型 TODO 维度还需要再深入
    relationModel.setBackProperty(StringUtils.equals(parentClassName, persistentRelationItem.getPropertyClass()));
    relationModel.setCurrentView(null);
    relationModel.setTargetTableName("");

    return relationModel;
  }

  /**
   * 初始化模版的基本信息实体内容
   * @param persistentProperty
   * @return
   */
  protected TemplatePropertyEntity initTemplatePropertyEntity(PersistentProperty persistentProperty) {
    TemplatePropertyEntity propertyEntity = new TemplatePropertyEntity();
    propertyEntity.setCanInsert(persistentProperty.getCanInsert());
    propertyEntity.setCanUpdate(persistentProperty.getCanUpdate());
    propertyEntity.setUnique(persistentProperty.getUnique());
    propertyEntity.setIndex(persistentProperty.getIndex());
    propertyEntity.setNullable(persistentProperty.getNullable());
    propertyEntity.setPrimaryKey(persistentProperty.getPrimaryKey());
    propertyEntity.setPropertyClassName(persistentProperty.getPropertyClass());
    String propertyDbName = persistentProperty.getPropertyDbName() == null?"":persistentProperty.getPropertyDbName();
    Validate.isTrue(propertyDbName.length() <= 64 , "如填写字段的数据库属性[%s]，其长度应该小于64个字符，请检查!!" , propertyDbName);
    propertyEntity.setPropertyDbName(propertyDbName);
    propertyEntity.setPropertyDesc(persistentProperty.getPropertyDesc());
    String propertyName = persistentProperty.getPropertyName();
    Validate.notBlank(propertyName , "明细项属性不能为空，请检查!!");
    Validate.isTrue(propertyName.length() <= 64 , "属性名[%s]的长度需要小于64个字符，请检查!!" , propertyName);
    propertyEntity.setPropertyName(propertyName);
    propertyEntity.setDefaultKeys(persistentProperty.getDefaultKeys());
    propertyEntity.setDefaultValues(persistentProperty.getDefaultValues());
    propertyEntity.setDefaultController(persistentProperty.getDefaultType() != null?persistentProperty.getDefaultType():"");
    propertyEntity.setMaxLen(persistentProperty.getMaxLen());
    return propertyEntity;
  }

}
