package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.kuiper.entity.FormDetailsExportBoxEntity;
import com.bizunited.platform.kuiper.entity.ListTemplateEntity;
import com.bizunited.platform.kuiper.starter.common.constant.ExcelConstants;
import com.bizunited.platform.kuiper.starter.common.excel.ExcelExportWrapper;
import com.bizunited.platform.kuiper.starter.common.excel.ExcelExportWrapperBuilder;
import com.bizunited.platform.kuiper.starter.common.excel.NebulaExcelColumn;
import com.bizunited.platform.kuiper.starter.common.excel.NebulaExcelExport;
import com.bizunited.platform.kuiper.starter.common.excel.exception.ExcelMigrateException;
import com.bizunited.platform.kuiper.starter.repository.FormDetailsExportBoxRepository;
import com.bizunited.platform.kuiper.starter.service.FormDetailsExportBoxTaskService;
import com.bizunited.platform.kuiper.starter.service.ListTemplateService;
import com.bizunited.platform.kuiper.starter.service.instances.imports.FormDetailsExportBoxProcess;
import com.bizunited.platform.venus.common.service.file.FileRelativeTemplate;
import com.google.common.collect.Maps;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.poi.ss.usermodel.Sheet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.AnnotatedType;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author Keller
 * @create 2020/8/25
 */
@Service("FormDetailsExportBoxTaskServiceImpl")
public class FormDetailsExportBoxTaskServiceImpl implements FormDetailsExportBoxTaskService {

  private static final Logger LOGGER = LoggerFactory.getLogger(FormDetailsExportBoxTaskServiceImpl.class);

  private static final String FORM_DETAILS = "formdetails";

  @Autowired
  private FileRelativeTemplate fileRelativeTemplate;

  @Autowired
  private FormDetailsExportBoxRepository formDetailsExportBoxRepository;

  @Autowired
  private ListTemplateService listTemplateService;

  @Value("${venus.file.fileRoot}")
  private String fileRoot;

  @Override
  @Async("fromDetailsMigrateExecutor")
  public void exportProcess(FormDetailsExportBoxEntity entity, FormDetailsExportBoxProcess<?> process, String account, Map<String, Object> params) {
    /*
        1、创建导入记录信息并封装导出字段与查询参数
        2、导出字段信息与查询参数信息参入业务处理类(分页方式获取数据)
        2、导出数据写入本地临时文件
        3、上传文件至venus
        4、更新导入记录信息importProcess
     */
    Validate.notBlank(entity.getListTemplateCode(), "列表模板编码为空，请检查");

    //初始化分页参数
    Class<?> excelBean = getInterfaceGeneric(process, 0);

    NebulaExcelExport nebulaExcelExport = null;
    if (excelBean != null) {
      nebulaExcelExport = excelBean.getAnnotation(NebulaExcelExport.class);
    }
    Validate.notNull(nebulaExcelExport, "excel导出bean未配置NebulaExcelExport注解");

    //完成导出excel表头信息封装 (未考虑实现复杂表头，合并样式等)
    Map<Integer, String> titleMap = Maps.newTreeMap();
    Map<Integer, Field> fieldMap = Maps.newTreeMap();

    Field[] fields = excelBean.getDeclaredFields();
    int rowId = 0;
    for (Field field : fields) {
      //检查是否存在@ExcelExport注解 可以支持不存在@ExcelExport情况
      if (nebulaExcelExport == null) {
        titleMap.put(rowId, field.getName());
        rowId++;
      } else {
        //检查是否存在@ExcelColumn注解
        NebulaExcelColumn nebulaExcelColumn = field.getAnnotation(NebulaExcelColumn.class);
        if (nebulaExcelColumn == null) {
          continue;
        }
        if (StringUtils.isNoneBlank(nebulaExcelColumn.title())) {
          titleMap.put(nebulaExcelColumn.order(), nebulaExcelColumn.title());
        } else {
          titleMap.put(nebulaExcelColumn.order(), field.getName());
        }
        fieldMap.put(nebulaExcelColumn.order(), field);
      }
    }

    String[] titles = new String[titleMap.values().size()];
    titles = titleMap.values().toArray(titles);
    //创建导出文件
    String relativePath = fileRelativeTemplate.generatRelativePath(FORM_DETAILS, "");
    File exportFile = createLocalExcelFile("".getBytes(), relativePath);

    //初始化excel参数
    AtomicInteger rowIndex = new AtomicInteger(0);
    AtomicInteger columnIndex = new AtomicInteger(0);

    JSONArray exportField = null;
    if (StringUtils.isNoneBlank(entity.getExportField())) {
      exportField = JSONArray.parseArray(entity.getExportField());
    }

    JSONArray exportParam = null;
    if (StringUtils.isNoneBlank(entity.getExportParam())) {
      exportParam = JSONArray.parseArray(entity.getExportParam());
    }
    Optional<JSONArray> opExportParam = Optional.ofNullable(exportParam);

    try (OutputStream os = new FileOutputStream(exportFile)) {
      ExcelExportWrapper excelExportWrapper = ExcelExportWrapperBuilder.buildWithSXSSFWorkbook();
      //创建导出数据sheet 设置sheet名字
      Sheet sheet = sheetBuild(excelExportWrapper, nebulaExcelExport);

      //完成表头写入
      writeHeaderData(excelExportWrapper, sheet, titles, rowIndex, columnIndex);

      //调用业务接口获取数据信息
      Pageable pageable = PageRequest.of(0, nebulaExcelExport.pageSize());

      //判断列表模板版本，为空则使用默认模板
      String listTemplateVersion = entity.getListTemplateVersion();
      if (StringUtils.isNoneBlank(entity.getListTemplateCode()) && StringUtils.isBlank(listTemplateVersion)) {
        ListTemplateEntity listTemplateEntity = listTemplateService.findDefaultByCode(entity.getListTemplateCode());
        Validate.notNull(listTemplateEntity, "列表模板数据错误，请检查！");
        listTemplateVersion = listTemplateEntity.getCversion();
      }

      Optional<String> dataViewCode = Optional.ofNullable(entity.getDataViewCode());

      Page<?> page = process.exportProcess(pageable, dataViewCode, exportField, opExportParam, params);
      //写入第一批数据
      //数据封装
      writePageData(excelExportWrapper, sheet, page, fieldMap, rowIndex, columnIndex);

      //后续批次数据处理
      while (page.hasNext()) {
        if (!checkStatus(entity.getId())) {
          os.close();
          if (!exportFile.delete()) {
            LOGGER.warn("导出临时文件删除错误");
          }
          throw new ExcelMigrateException("导出数据取消");
        }
        pageable = page.nextPageable();
        page = process.exportProcess(pageable, dataViewCode, exportField, opExportParam, params);
        writePageData(excelExportWrapper, sheet, page, fieldMap, rowIndex, columnIndex);
      }
      excelExportWrapper.export(os);
      LOGGER.info("导出数据写入临时文件处理完成");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      this.complateRecord(entity);
      throw new IllegalArgumentException("导出文件失败");
    }

    entity.setFileName(exportFile.getName());
    entity.setRelativeLocal(relativePath);
    if (StringUtils.isNoneBlank(nebulaExcelExport.excelName())) {
      if (nebulaExcelExport.excelName().endsWith(ExcelConstants.EXCEL_XLS_END) || nebulaExcelExport.excelName().endsWith(ExcelConstants.EXCEL_XLSX_END)) {
        entity.setDownloadFileName(nebulaExcelExport.excelName());
      } else {
        entity.setDownloadFileName(StringUtils.join(nebulaExcelExport.excelName(), ExcelConstants.EXCEL_XLSX_END));
      }
    } else {
      entity.setDownloadFileName(entity.getFileName());
    }
    this.complateRecord(entity);
    LOGGER.info("更新导出数据记录");
  }

  private boolean checkStatus(String id) {
    Optional<FormDetailsExportBoxEntity> op = this.formDetailsExportBoxRepository.findById(id);
    FormDetailsExportBoxEntity entity = op.orElse(null);
    Validate.notNull(entity, "导出数据记录为空");
    Integer status = entity.getExecuteResult();
    if (status != 2) {
      return false;
    } else {
      return true;
    }
  }

  private Sheet sheetBuild(ExcelExportWrapper excelExportWrapper, NebulaExcelExport nebulaExcelExport) {
    Sheet sheet = null;
    if (nebulaExcelExport != null) {
      sheet = excelExportWrapper.buildSheet(nebulaExcelExport.sheetName());
    } else {
      sheet = excelExportWrapper.buildSheet("数据");
    }
    return sheet;
  }

  private void writeHeaderData(ExcelExportWrapper excelExportWrapper, Sheet sheet, String[] titles, AtomicInteger rowIndex, AtomicInteger columnIndex) {
    excelExportWrapper.paint(sheet, rowIndex.intValue(), columnIndex.intValue(), titles);
    rowIndex.getAndIncrement();
  }

  private void writePageData(ExcelExportWrapper excelExportWrapper, Sheet sheet, Page<?> page, Map<Integer, Field> fieldMap, AtomicInteger rowIndex, AtomicInteger columnIndex) {
    Iterator<?> iter = page.iterator();
    while (iter.hasNext()) {
      Object data = iter.next();
      //处理数据
      Object[] values = getValues(data, fieldMap);
      excelExportWrapper.paint(sheet, rowIndex.intValue(), columnIndex.intValue(), values);
      rowIndex.getAndIncrement();
    }
  }

  private void complateRecord(FormDetailsExportBoxEntity entity) {
    Validate.notNull(entity, "导出记录不能为空");
    entity.setExecuteEndTime(new Date());
    entity.setExecuteResult(1);
    this.formDetailsExportBoxRepository.saveAndFlush(entity);
  }

  private File createLocalExcelFile(byte[] bytes, String relativePath) {
    String renameImage = UUID.randomUUID().toString();
    String fileRename = StringUtils.join(renameImage, ".", "xlsx");
    File folder = new File(StringUtils.join(fileRoot, relativePath));
    if (!folder.exists()) {
      folder.mkdirs();
    }
    //保存错误文件至指定文件夹
    File file = new File(folder, fileRename);
    try {
      Validate.isTrue(file.createNewFile(), "创建本地文件失败：%s", file.getAbsolutePath());
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("创建本地文件失败：%s", e.getMessage()));
    }
    if (bytes != null) {
      ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
      try (FileOutputStream fos = new FileOutputStream(file)) {
        int maxLen = 4096;
        int realLen;
        byte[] contents = new byte[maxLen];
        while ((realLen = bais.read(contents, 0, maxLen)) != -1) {
          fos.write(contents, 0, realLen);
        }
      } catch (FileNotFoundException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(String.format("未找到本地文件：%s", file.getAbsolutePath()));
      } catch (IOException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(String.format("读取本地文件失败：%s", e.getMessage()));
      }
    }
    return file;
  }

  private Object[] getValues(Object o, Map<Integer, Field> fieldMap) {
    List<Object> values = Lists.newArrayList();
    fieldMap.values().stream().forEach(field -> {
      //允许字段访问
      field.setAccessible(true);
      NebulaExcelColumn nebulaExcelColumn = field.getAnnotation(NebulaExcelColumn.class);
      Class<?> fieldType = field.getType();
      Method fieldGetMethod = null;
      try {
        fieldGetMethod = o.getClass().getDeclaredMethod("get" + StringUtils.capitalize(field.getName()));
        Object value = fieldGetMethod.invoke(o);
        if (value == null) {
          values.add(null);
          return;
        }
        //获取值处理
        if (String.class.isAssignableFrom(fieldType)) {
          values.add(String.valueOf(value));
        } else if (Date.class.isAssignableFrom(fieldType)) {
          String dateFormat = null;
          if (nebulaExcelColumn != null) {
            dateFormat = nebulaExcelColumn.format();
          }
          Date date = (Date) value;
          SimpleDateFormat sdf = null;
          if (StringUtils.isNoneBlank(dateFormat)) {
            sdf = new SimpleDateFormat(dateFormat);
          } else {
            sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
          }
          values.add(sdf.format(date));
        } else if (Integer.class.isAssignableFrom(fieldType) || int.class.isAssignableFrom(fieldType)) {
          Integer intval = (Integer) value;
          values.add(intval);
        } else if (Long.class.isAssignableFrom(fieldType) || long.class.isAssignableFrom(fieldType)) {
          Long temp = (Long) value;
          values.add(temp);
        } else if (Double.class.isAssignableFrom(fieldType) || double.class.isAssignableFrom(fieldType)) {
          Double temp = (Double) value;
          values.add(temp);
        } else if (Boolean.class.isAssignableFrom(fieldType)) {
          Boolean temp = (Boolean) value;
          values.add(temp);
        } else if (Float.class.isAssignableFrom(fieldType) || float.class.isAssignableFrom(fieldType)) {
          Float temp = (Float) value;
          values.add(temp);
        } else {
          LOGGER.error(String.format("not supper type: %s ", fieldType));
          throw new ExcelMigrateException(nebulaExcelColumn, "字段类型无效 字段:" + field.getName() + " 类型:" + fieldType);
        }
      } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
        LOGGER.error(e.getMessage(), e);
        throw new ExcelMigrateException(nebulaExcelColumn, "字段属性get方法失败，请检查:" + field.getName(), e);
      }
    });
    return values.toArray(new Object[values.size()]);
  }

  /**
   * 获取实现泛型类型
   *
   * @param obj
   * @param index
   * @return
   */
  private Class<?> getInterfaceGeneric(Object obj, Integer index) {
    Class<?> clazz = null;
    if (AopUtils.isCglibProxy(obj) || AopUtils.isAopProxy(obj)) {
      //针对代理类的处理
      clazz = AopUtils.getTargetClass(obj);
    } else {
      clazz = obj.getClass();
    }
    AnnotatedType[] annotatedTypes = clazz.getAnnotatedInterfaces();
    if (annotatedTypes == null) {
      return null;
    }
    Type type = annotatedTypes[0].getType();
    if (!(type instanceof ParameterizedType)) {
      return null;
    }
    Type[] params = ((ParameterizedType) type).getActualTypeArguments();
    if (params == null || params.length < index) {
      return null;
    }
    try {
      Class<?> cc = ((Class<?>) params[index]);
      ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
      return Class.forName(cc.getName(), false, classLoader);
    } catch (ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
    }
    return null;
  }
}
