package com.bizunited.platform.kuiper.starter.service.internal;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.Principal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.UUID;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.bizunited.platform.kuiper.entity.FormInstanceUuidEntity;
import com.bizunited.platform.kuiper.entity.FromDetailsImportEntity;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.service.InstanceService;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.repository.FromDetailsImportRepository;
import com.bizunited.platform.kuiper.starter.service.FromDetailsImportService;
import com.bizunited.platform.kuiper.starter.service.instances.imports.FromDetailsImportProcess;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;

/**
 * 基于表单引擎，列表引擎，对业务数据通过xls文件的方式进行导入，并关联相关表单实例的服务的默认实现
 * @author yinwenjie
 */
@Component("FromDetailsImportServiceImpl")
public class FromDetailsImportServiceImpl implements FromDetailsImportService {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(FromDetailsImportServiceImpl.class);
  
  @Autowired
  private TemplateService templateService;
  
  @Autowired
  private InstanceService instanceService;
  
  @Autowired
  private UserService userService;
  
  @Autowired
  private ApplicationContext applicationContext;
  
  @Autowired
  private VenusFileService venusFileService;
  
  @Autowired
  private FromDetailsImportRepository fromDetailsImportRepository;
  
  /**
   * 当前已经被扫描的，进程中包含的所有业务导入处理器的类名信息
   */
  private static Set<String> processClassNamesMapping = new HashSet<>();
  
  @SuppressWarnings("unchecked")
  @Override
  @Transactional
  public void imports(byte[] fileBytes, Map<String, Object> params, Principal principal , String templateCode, String templateVersion , String processClassName) {
    Validate.notNull(fileBytes , "本次导出操作所基于的xls/xlsx文件内容不能为null");
    Map<String, Object> currentParams = params;
    if(currentParams == null) {
      currentParams = new HashMap<>();
    }
    // 检查操作者信息
    Validate.notNull(principal , "当前操作者信息必须传入");
    String account = principal.getName();
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser , "错误的操作者信息，请检查!!");
    // 检验模板信息
    TemplateEntity currentTemplate = null;
    // 如果条件成立，则使用默认版本
    if(StringUtils.isBlank(templateVersion)) {
      List<TemplateEntity> templetes = this.templateService.findByCode(templateCode);
      Validate.notNull(templetes , "未找到指定的表单模板信息[%s]，请检查!!" , templateCode);
      Optional<TemplateEntity> templateOp = templetes.stream().filter(item->item.getDefaultVersion()).findFirst();
      currentTemplate = templateOp.orElse(null);
      Validate.notNull(currentTemplate , "未找到指定的表单模板的默认版本信息[%s]，请检查!!" , templateCode);
    } else {
      currentTemplate = this.templateService.findByCodeAndCversion(templateCode, templateVersion);
      Validate.notNull(currentTemplate , "未找到指定的表单模板信息[%s : %s]，请检查!!" , templateCode , templateVersion);
    }
    // 检验处理器信息
    Validate.notBlank(processClassName , "指定的process Class name必须传入，请检查!!");
    Class<?> processClass = null;
    try {
      processClass = applicationContext.getClassLoader().loadClass(processClassName);
    } catch (ClassNotFoundException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(String.format("在当前进程的spring ioc容器中，未发现指定的process Class[%s]，请检查!!" , processClassName));
    }
    Validate.isTrue(FromDetailsImportProcess.class.isAssignableFrom(processClass), "当前给定的processClassName并没有正确实现FromDetailsImportProcess接口，请检查!!");
    FromDetailsImportProcess<FormInstanceUuidEntity> process = (FromDetailsImportProcess<FormInstanceUuidEntity>)applicationContext.getBean(processClass);
    
    /*
     * 在完成边界校验后，处理过程如下所示：
     * 1、首先调用processClassName所对应处理器的验证方法，完成验证过程
     * 2、根据第一步得到的集合信息，创建表单实例，并且为这些表单实例完成表单实例的创建过程，并为模型绑定表单实例信息到对应的对象上
     * 注意：这个操作步骤不会处理业务数据，只会基于业务数据创建表单实例和活动信息
     * 3、调用processClassName所对应处理器的execute方法，完成业务数据的导入和业务逻辑的处理过程
     * */
    
    // 1、========
    Set<FormInstanceUuidEntity> instanceUuidEntities = null;
    try (InputStream is = new ByteArrayInputStream(fileBytes)) {
      XSSFWorkbook workbook = new XSSFWorkbook(is);
      instanceUuidEntities = process.validate(workbook, currentParams, templateCode, templateVersion);
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException("目前业务数据的批量导入和表单生成功能，只支持基于xls,xlsx文件的导入，请检查文件内容和文件格式!!");
    }
    Validate.isTrue(!CollectionUtils.isEmpty(instanceUuidEntities) , "本次导入操作，未发现任何需要导入的业务数据，请检查业务数据导入前的校验逻辑!!");
    
    
    // 2、========
    for (FormInstanceUuidEntity instanceEntity : instanceUuidEntities) {
      InstanceEntity instance = this.instanceService.createByTemplateCode(templateCode, templateVersion, account);
      instanceEntity.setFormInstanceId(instance.getId());
    }
    
    // 3、=======
    process.execute(instanceUuidEntities, currentParams);
  }

  @Override
  public Set<String> findProcessClassNames() {
    return processClassNamesMapping;
  }

  @Override
  public void putProcessClassNames(String processClassNames) {
    if(StringUtils.isBlank(processClassNames)) {
      return;
    }
    processClassNamesMapping.add(processClassNames);
  }

  @Override
  @Transactional
  public FromDetailsImportEntity importTemplate(byte[] fileBytes, String listTemplateCode, String listTemplateVersion, String buttonCode, String originalFilename) {
    // 检查是否是xls、xlsx文件信息
    Validate.notNull(fileBytes , "模板文件内容必须传入");
    try (InputStream is = new ByteArrayInputStream(fileBytes);
        XSSFWorkbook workbook = new XSSFWorkbook(is)) {
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException("业务数据的导入模板文件，只支持基于xls,xlsx文件，请检查文件内容和文件格式!!");
    }
    // 检查必须传入的值
    Validate.notBlank(listTemplateCode , "数据导入模板上传时，列表模板编码信息必须传入!!");
    Validate.notBlank(listTemplateVersion , "数据导入模板上传时，列表模板版本信息必须传入!!");
    Validate.notBlank(buttonCode , "数据导入模板上传时，功能按钮编码信息必须传入!!");
    Validate.notBlank(originalFilename , "数据导入模板上传时，原始文件名必须传入!!");
    // 确认后缀
    String filenameSuffix = StringUtils.substringAfterLast(originalFilename, ".");
    Validate.notBlank(filenameSuffix , "未发现上传的模板文件名的后缀信息，请检查!!");
    filenameSuffix = StringUtils.lowerCase(filenameSuffix);
    Validate.isTrue(StringUtils.containsAny(filenameSuffix, "xls","xlsx") , "目前模板文件的扩展名仅支持xls和xlsx两类，请检查!!");
    
    /*
     * 在对入参边界进行验证后，处理过程如下所示：
     * 1、首先确定这个模板文件是否之前就已经存在对应的按钮信息，如果存在就是修改操作
     * 否则就是添加操作
     * 2、使用fileService，进行文件的内容写入工作，并确认得到文件的相对路径、文件重命名信息
     * 3、根据第一步的场景确认的信息，进行文件的保存操作，并进行返回
     * */
    
    // 1、======
    FromDetailsImportEntity currentFromDetailsImport = this.fromDetailsImportRepository.findByCodeAndVersionAndButtonCode(listTemplateCode, listTemplateVersion, buttonCode);
    if(currentFromDetailsImport == null) {
      currentFromDetailsImport = new FromDetailsImportEntity();
      currentFromDetailsImport.setButtonCode(buttonCode);
      currentFromDetailsImport.setListTempleteCode(listTemplateCode);
      currentFromDetailsImport.setListTempleteVersion(listTemplateVersion);
    }
    
    // 2、======
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String uuid = UUID.randomUUID().toString();
    String systemName = "fromDetailsImport";
    String fileRename = uuid + "." + filenameSuffix;
    String relativePath = StringUtils.join("/", systemName, "/" , folderName ,  "/", (new Random().nextInt(100) % 10));
    this.venusFileService.saveFile(relativePath, originalFilename, fileRename, fileBytes);
    
    // 3、保存模板基本信息，并进行处理结果返回
    currentFromDetailsImport.setFileName(fileRename);
    currentFromDetailsImport.setRelativePath(relativePath);
    fromDetailsImportRepository.save(currentFromDetailsImport);
    return currentFromDetailsImport;
  }

  /**
   * 根据文件名称和相对路径查询
   * @param fileName
   * @param relativeLocal
   * @return
   */
  @Override
  public FromDetailsImportEntity findByFileNameAndRelativeLocal(String fileName, String relativeLocal) {
    if(StringUtils.isBlank(fileName) || StringUtils.isBlank(relativeLocal)){
      return null;
    }
    FromDetailsImportEntity fromDetailsImport = fromDetailsImportRepository.findByFileNameAndRelativePath(fileName, relativeLocal);
    if(fromDetailsImport == null) {
      return null;
    }
    return fromDetailsImport;
  }
}