package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.kuiper.entity.InstanceActivityEntity;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.starter.repository.InstanceActivityRepository;
import com.bizunited.platform.kuiper.starter.repository.InstanceRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateVisibilityRepository;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import static com.bizunited.platform.kuiper.starter.common.constant.Constants.VISIBILITY_CREATE;

/**
 * @author yinwenjie
 */
@Service("InstanceActivityServiceImpl")
public class InstanceActivityServiceImpl implements InstanceActivityService {

  @Autowired
  private InstanceActivityRepository instanceActivityRepository;
  @Autowired
  private InstanceRepository instanceRepository;
  @Autowired
  private TemplateVisibilityRepository templateVisibilityRepository;
  
  @Override
  @Transactional
  public InstanceActivityEntity create(InstanceActivityEntity activity, String account) {
    /*
     * 1、验证传入的activity对象，主要是其中的关键信息
     * 2、按照当前模板信息查询当前模板下目前最后一个活动（作为当前新建活动的上一个活动）
     * 3、补充其它对象，并且进行插入
     * */
    // 1、=======
    Validate.notNull(activity , "创建活动时，必须传入活动信息!!");
    Validate.isTrue(StringUtils.isBlank(activity.getId()) , "创建活动时，活动编号不能传入!!");
    activity.setId(null);
    // 表单实例检测
    InstanceEntity currentInstance = activity.getInstance();
    Validate.notNull(currentInstance , "创建活动时，必须有相关的实例信息，请检查!!");
    String instanceId = currentInstance.getId();
    Validate.notBlank(instanceId , "创建活动时，未发现相关实例的数据编号，请检查!!");
    Optional<InstanceEntity> op = this.instanceRepository.findById(instanceId);
    InstanceEntity dbInstance = op.orElse(null);
    Validate.notNull(dbInstance , "未找到指定的表单实例信息，请检查!!");

    // 模板检测
    TemplateEntity dbTemplate = dbInstance.getTemplate();
    Validate.notNull(dbTemplate , "创建活动时，未在数据库发现指定的模板信息!!");
    // 模板可见性检测
    TemplateVisibilityEntity templateVisibility = activity.getTemplateVisibility();
    Validate.notNull(templateVisibility , "创建活动时，指定的模板可见性信息必须传入!!");
    String templateVisibilityId = templateVisibility.getId();
    Validate.notBlank(templateVisibilityId , "创建活动时，指定的模板可见性的编号信息必须传入!!");
    Optional<TemplateVisibilityEntity> opVisibility = this.templateVisibilityRepository.findById(templateVisibilityId);
    TemplateVisibilityEntity dbTemplateVisibility = opVisibility.orElse(null);
    Validate.notNull(dbTemplateVisibility , "创建活动时，指定的模板可见性未在数据库中发现!!");
    TemplateVisibilityEntity currentTemplateVisibility = this.templateVisibilityRepository.findByTemplateIdAndVisibilityName(dbTemplate.getId(), dbTemplateVisibility.getVisibilityName());
    Validate.notNull(currentTemplateVisibility , "创建活动指定的活动不属于指定的表单模板，请检测!!");
    // 操作用户信息检测
    Validate.notBlank(account , "当前操作者信息必须传入");
    // 活动名称
    String visibilityName = dbTemplateVisibility.getVisibilityName();
    Validate.notBlank(visibilityName , "指定的活动可见性的名称必须有值!!");
    
    // 2、=======
    InstanceActivityEntity previousActivity = null;
    if(activity.getPreviousActivity() == null) {
      Object[][] results = this.instanceActivityRepository.findByInstanceIdAndMaxCreateTime(instanceId);
      if(results != null && results.length > 0) {
        String maxActivityId = results[0][0].toString();
        Optional<InstanceActivityEntity> opActivity = this.instanceActivityRepository.findById(maxActivityId);
        previousActivity = opActivity.orElse(null);
      }
    } else {
      String previousInstanceActivityId = activity.getPreviousActivity().getId();
      Validate.notBlank(previousInstanceActivityId , "指定的前序活动必须指定活动编号信息");
      Optional<InstanceActivityEntity> optional = this.instanceActivityRepository.findById(previousInstanceActivityId);
      Validate.isTrue(optional.isPresent() , "未发现指定的前序活动信息，请检查!!");
    }
    // 注意如果当前实例存在前驱活动，则当前被创建的活动的名字不能为create
    if(previousActivity != null) {
      Validate.isTrue(!StringUtils.equals(visibilityName, VISIBILITY_CREATE) , "当前活动存在前驱活动，不能使用create可见性进行活动创建!!");
    }
    
    // 3、=======
    Date time = new Date();
    activity.setCreateTime(time);
    activity.setCreator(account);
    activity.setModifyer(account);
    activity.setModifyTime(time);
    // 注意，由于数据层JPA使用多级关联返回previousActivity信息
    // 但是这里只返回其中一级，所以要进行特别处理
    InstanceActivityEntity currentActivity = null;
    if(previousActivity != null) {
      currentActivity = new InstanceActivityEntity();
      currentActivity.setCreateTime(previousActivity.getCreateTime());
      currentActivity.setCreator(previousActivity.getCreator());
      currentActivity.setId(previousActivity.getId());
      currentActivity.setModifyer(previousActivity.getModifyer());
      currentActivity.setModifyTime(previousActivity.getModifyTime());
    }
    activity.setPreviousActivity(currentActivity);
    activity.setTemplate(dbTemplate);
    activity.setTemplateVisibility(dbTemplateVisibility);
    if(StringUtils.isBlank(activity.getTaskCode())) {
      activity.setTaskCode(UUID.randomUUID().toString());
    }

    activity.setProjectName(ApplicationContextUtils.getProjectName());
    this.instanceActivityRepository.save(activity);
    return activity;
  }

  @Override
  @Transactional
  public InstanceActivityEntity create(String taskCode, String instanceId, String visibilityName, String account) {
    /*
     * 在输入值验证后的处理过程为：
     * 1、如果可见性是create，则特殊处理，根据实例ID和taskCode查询活动，如果活动不存则继续往下执行
     * 2、如果taskCode能够找到活动明细，则直接返回
     * 3、如果没有找到活动明细，则通过instanceId对应的模板和visibilityName进行活动创建
     * （可直接调用另一个create方法）
     * */
    // 1、=======
    if(VISIBILITY_CREATE.equals(visibilityName)) {
      if(StringUtils.isBlank(instanceId)) {
        return null;
      }
      Set<InstanceActivityEntity> activities = instanceActivityRepository.findDetailsByInstanceIdAndVisibility(instanceId, visibilityName);
      if(CollectionUtils.isEmpty(activities)) {
        return null;
      }
      return activities.iterator().next();
    }
    // 2、=======
    Validate.notBlank(taskCode , "第三方系统的唯一活动识别号必须传入!!");
    InstanceActivityEntity instanceActivity = this.instanceActivityRepository.findDetailsByTaskCode(taskCode);
    if(instanceActivity != null) {
      return instanceActivity;
    }
    
    // 3、=======
    Validate.notBlank(instanceId , "在创建活动时，实例编号名必须传入!!");
    Validate.notBlank(visibilityName , "在创建活动时，可见性名必须传入!!");
    Optional<InstanceEntity> op = this.instanceRepository.findById(instanceId);
    InstanceEntity instance = op.orElse(null);
    Validate.notNull(instance , "在创建活动时，未找到指定的表单实例信息!!");

    TemplateEntity currentTemplate = instance.getTemplate();
    // 查询模板中指定的可见性
    Validate.isTrue(!StringUtils.equals(visibilityName, VISIBILITY_CREATE) , "名为create的可见性不允许独立创建为一个活动实例");
    TemplateVisibilityEntity currentTemplateVisibility = this.templateVisibilityRepository.findByTemplateIdAndVisibilityName(currentTemplate.getId(), visibilityName);
    Validate.notNull(currentTemplateVisibility , "在指定的表单实例下未找到指定的可见性信息,请检查");
    // 开始创建
    InstanceActivityEntity currentActivity = new InstanceActivityEntity();
    currentActivity.setInstance(instance);
    currentActivity.setTemplateVisibility(currentTemplateVisibility);
    currentActivity.setTaskCode(taskCode);
    return this.create(currentActivity, account);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.formengine.starter.service.InstanceActivityService#updateModifyer(java.lang.String, java.security.Principal)
   */
  @Override
  @Transactional
  public InstanceActivityEntity updateModifyer(String activityId, String account) {
    Validate.notBlank(activityId , "修改活动时，活动编号必须传入!!");
    Optional<InstanceActivityEntity> op = this.instanceActivityRepository.findById(activityId);
    InstanceActivityEntity activity = op.orElse(null);
    Validate.notNull(activity , "未找到指定的活动信息!!");

    // 操作用户信息检测
    Validate.notBlank(account , "当前操作者用户信息必须传入");
    // 写入修改者
    activity.setModifyer(account);
    activity.setModifyTime(new Date());
    this.instanceActivityRepository.save(activity);
    return activity;
  }

  @Override
  @Transactional
  public void updateTemplate(List<InstanceActivityEntity> activities, TemplateEntity template) {
    Validate.notNull(activities, "活动信息不能为空");
    Validate.notNull(template, "模板对象不能为空！！！");
    Validate.notBlank(template.getCode(), "模板编号不能为空！！！");
    Validate.notBlank(template.getId(), "模版ID不能为空！！！");
    for (InstanceActivityEntity activity : activities) {
      TemplateVisibilityEntity visibility = activity.getTemplateVisibility();
      TemplateVisibilityEntity newVisibility = templateVisibilityRepository.findByTemplateIdAndVisibilityName(template.getId(), visibility.getVisibilityName());
      Validate.notNull(newVisibility, "未找到模版[%s]可见性：%s", template.getCode(), visibility.getVisibilityName());
      activity.setTemplate(template);
      activity.setTemplateVisibility(newVisibility);
      instanceActivityRepository.save(activity);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.formengine.starter.service.InstanceActivityService#findByInstanceId(java.lang.String)
   */
  @Override
  public Set<InstanceActivityEntity> findDetailsByInstanceId(String instanceId) {
    if(StringUtils.isBlank(instanceId)) {
      return Sets.newHashSet();
    }
    
    // 注意，由于数据层JPA使用多级关联返回previousActivity信息
    // 但是这里只返回其中一级，所以要进行特别处理
    Set<InstanceActivityEntity> instanceActivities = this.instanceActivityRepository.findDetailsByInstanceId(instanceId);
    if(instanceActivities != null && !instanceActivities.isEmpty()) {
      instanceActivities.forEach(instanceActivityItem -> {
        InstanceActivityEntity previous = instanceActivityItem.getPreviousActivity();
        if(previous != null) {
          InstanceActivityEntity currentActivity = new InstanceActivityEntity();
          currentActivity.setCreateTime(previous.getCreateTime());
          currentActivity.setCreator(previous.getCreator());
          currentActivity.setId(previous.getId());
          currentActivity.setModifyer(previous.getModifyer());
          currentActivity.setModifyTime(previous.getModifyTime());
          instanceActivityItem.setPreviousActivity(currentActivity);
        } else {
          instanceActivityItem.setPreviousActivity(null);
        }
      });
    }
    
    return instanceActivities;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.formengine.starter.service.InstanceActivityService#findByInstanceId(java.lang.String)
   */
  @Override
  public Set<InstanceActivityEntity> findByInstanceId(String instanceId) {
    if(StringUtils.isBlank(instanceId)) {
      return Sets.newHashSet();
    }
    
    return this.instanceActivityRepository.findByInstanceId(instanceId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.formengine.starter.service.InstanceActivityService#findById(java.lang.String)
   */
  @Override
  public InstanceActivityEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.instanceActivityRepository.findDetailsById(id);
  }

  /**
   * 根据活动id，查询活动基本信息
   */
  @Override
  public InstanceActivityEntity findById(String id) {
    Optional<InstanceActivityEntity> op = this.instanceActivityRepository.findById(id);
    return  op.orElse(null);
  }

  @Override
  public InstanceActivityEntity findByTaskCode(String taskCode) {
    if(StringUtils.isBlank(taskCode)) {
      return null;
    }
    return this.instanceActivityRepository.findByTaskCode(taskCode);
  }

  @Override
  public InstanceActivityEntity findByInstanceIdAndTaskCode(String instanceId, String taskCode){
    if(StringUtils.isBlank(instanceId)) return null;
    if(StringUtils.isBlank(taskCode)) return null;
    return instanceActivityRepository.findByInstanceIdAndTaskCode(instanceId, taskCode);
  }

  @Override
  public Set<InstanceActivityEntity> findByInstanceIdAndVisibility(String instanceId, String visibility) {
    if(StringUtils.isBlank(instanceId) || StringUtils.isBlank(visibility)) {
      return Sets.newHashSet();
    }
    return instanceActivityRepository.findByInstanceIdAndVisibility(instanceId, visibility);
  }

  @Override
  public String findIdByMaxCreateTime(String instanceId) {
    if(StringUtils.isBlank(instanceId)) {
      return null;
    }
    Object[][] results = this.instanceActivityRepository.findByInstanceIdAndMaxCreateTime(instanceId);
    if(results == null || results.length == 0) {
      return null;
    }
    return results[0][0].toString();
  }

}