package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.kuiper.entity.PageEntity;
import com.bizunited.platform.kuiper.entity.PageEventEntity;
import com.bizunited.platform.kuiper.starter.repository.PageEventRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.PageEventService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import static com.bizunited.platform.kuiper.starter.common.constant.Constants.JS_FILE_SUFFIX;

/**
 * 页面事件服务实现
 * @Author: Paul Chan
 * @Date: 2019-12-31 16:44
 */
@Service("PageEventServiceImpl")
public class PageEventServiceImpl implements PageEventService {

  @Autowired
  private PageEventRepository pageEventRepository;
  @Autowired
  private VenusFileService kuiperFileService;
  @Autowired
  private KuiperToolkitService kuiperToolkitService;


  /**
   * @param page
   * @param events
   * @return
   */
  @Override
  @Transactional
  public Set<PageEventEntity> save(PageEntity page, Set<PageEventEntity> events) {
    this.saveValidation(page, events);
    Set<PageEventEntity> oldEvents = pageEventRepository.findByPageId(page.getId());
    if(CollectionUtils.isEmpty(events)) {
      this.deleteByPageId(page.getId());
      return Sets.newHashSet();
    }
    if(CollectionUtils.isEmpty(oldEvents)) {
      return this.create(page, events);
    }
    Set<PageEventEntity> createEvents = new HashSet<>();
    Set<PageEventEntity> updateEvents = new HashSet<>();
    Set<PageEventEntity> deleteEvents = new HashSet<>();
    kuiperToolkitService.collectionDiscrepancy(events, oldEvents, PageEventEntity::getEventId, deleteEvents, updateEvents, createEvents);
    this.create(page, createEvents);
    this.update(updateEvents, events);
    this.delete(deleteEvents);
    createEvents.addAll(updateEvents);
    return createEvents;
  }

  @Override
  public Set<PageEventEntity> findByPageId(String pageId) {
    if(StringUtils.isBlank(pageId)) {
      return Sets.newHashSet();
    }
    return pageEventRepository.findByPageId(pageId);
  }

  @Override
  public Set<PageEventEntity> findDetailsByPageId(String pageId) {
    if(StringUtils.isBlank(pageId)) {
      return Sets.newHashSet();
    }
    Set<PageEventEntity> events = pageEventRepository.findByPageId(pageId);
    this.loadEventContent(events);
    return events;
  }

  @Override
  public PageEventEntity findDetailById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    PageEventEntity event = pageEventRepository.findById(id).orElse(null);
    if(event != null) {
      this.loadEventContent(event);
    }
    return event;
  }

  @Override
  public PageEventEntity findDetailByPageFlowCodeAndPageCodeAndEventId(String pageFlowCode, String pageCode, String eventId) {
    if(StringUtils.isAnyBlank(pageFlowCode, pageCode, eventId)) {
      return null;
    }
    PageEventEntity event = pageEventRepository.findDetailByPageFlowCodeAndPageCodeAndEventId(pageFlowCode, pageCode, eventId);
    if(event != null) {
      this.loadEventContent(event);
    }
    return event;
  }

  @Override
  @Transactional
  public void deleteByPageId(String pageId) {
    Validate.notBlank(pageId, "事件ID不能为空！");
    Set<PageEventEntity> events = pageEventRepository.findByPageId(pageId);
    this.delete(events);
  }

  /**
   * 删除事件和文件内容
   * @param events
   */
  private void delete(Set<PageEventEntity> events) {
    if (!CollectionUtils.isEmpty(events)){
      for (PageEventEntity event : events) {
        kuiperFileService.deleteFile(event.getRelativePath(), event.getFileName(), event.getFileName());
        pageEventRepository.delete(event);
      }
    }
  }

  /**
   * 更新页面事件
   * 更新文件内容前会对传入的内容和已有内容进行对比，如果内容不一致才会更新文件内容
   * @param events
   * @param updateEvents
   * @return
   */
  private Set<PageEventEntity> update(Set<PageEventEntity> events, Set<PageEventEntity> updateEvents) {
    if(CollectionUtils.isEmpty(events)) {
      return Sets.newHashSet();
    }
    Map<String, PageEventEntity> updateEventMap = updateEvents.stream().collect(Collectors.toMap(PageEventEntity::getEventId, e -> e));
    for (PageEventEntity event : events) {
      PageEventEntity updateEvent = updateEventMap.get(event.getEventId());
      this.loadEventContent(event);
      if(!updateEvent.getEventContent().equals(event.getEventContent())) {
        event.setEventContent(updateEvent.getEventContent());
        this.saveFile(event);
      }
    }
    return events;
  }

  /**
   * 加载事件内容
   * @param events
   * @return
   */
  private Set<PageEventEntity> loadEventContent(Set<PageEventEntity> events) {
    if(CollectionUtils.isEmpty(events)) {
      return events;
    }
    for (PageEventEntity event : events) {
      this.loadEventContent(event);
    }
    return events;
  }

  /**
   * 加载事件内容
   * @return
   */
  private PageEventEntity loadEventContent(PageEventEntity event) {
    byte[] bytes = kuiperFileService.readFileContent(event.getRelativePath(), event.getFileName());
    if(bytes != null) {
      String content = new String(bytes, StandardCharsets.UTF_8);
      event.setEventContent(content);
    }
    return event;
  }

  /**
   * 批量创建事件
   * @param page
   * @param events
   * @return
   */
  private Set<PageEventEntity> create(PageEntity page, Set<PageEventEntity> events) {
    if(CollectionUtils.isEmpty(events)) {
      return Sets.newHashSet();
    }
    for (PageEventEntity event : events) {
      event.setPage(page);
      this.saveFile(event);
      event.setProjectName(ApplicationContextUtils.getProjectName());
      pageEventRepository.save(event);
    }
    return events;
  }

  /**
   * 保存事件内容到文件系统中
   * @param event
   * @return
   */
  private PageEventEntity saveFile(PageEventEntity event) {
    if(StringUtils.isBlank(event.getId())) {
      String folderName = new SimpleDateFormat("yyyyMMdd").format(new Date());
      String rename = UUID.randomUUID().toString();
      String fileRename = StringUtils.join(rename, ".", JS_FILE_SUFFIX);
      String relativePath = StringUtils.join("/page/event/", folderName , "/", new Random().nextInt(10));
      event.setRelativePath(relativePath);
      event.setFileName(fileRename);
    }
    byte[] content = event.getEventContent().getBytes(StandardCharsets.UTF_8);
    kuiperFileService.saveFile(event.getRelativePath(), event.getFileName(), event.getFileName(), content);
    return event;
  }

  /**
   * 保存前数据验证
   * @param page
   * @param events
   */
  private void saveValidation(PageEntity page, Set<PageEventEntity> events) {
    Validate.notNull(page, "页面对象不能为空");
    Validate.notBlank(page.getId(), "页面对象的ID不能为空");
    if(CollectionUtils.isEmpty(events)) {
      return;
    }
    Set<String> eventIds = new HashSet<>();
    for (PageEventEntity event : events) {
      Validate.notBlank(event.getEventId(), "事件ID不能为空");
      Validate.isTrue(eventIds.add(event.getEventId()), "事件ID重复，请检查");
      Validate.notNull(event.getEventContent(), "事件内容不能为空");
    }
  }

}
