package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateEventEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemExcelEntity;
import com.bizunited.platform.kuiper.entity.TemplateLayoutEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.service.DynamicTemplateService;
import com.bizunited.platform.kuiper.service.StaticTemplateService;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.repository.TemplateGroupRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateItemRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.TemplateEventService;
import com.bizunited.platform.kuiper.starter.service.TemplateItemExcelService;
import com.bizunited.platform.kuiper.starter.service.TemplateLayoutService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_DB_FORM_INSTANCE_ID_NAME;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_DB_PK_NAME;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGE_SIZE;
import static com.bizunited.platform.common.constant.Constants.PROJECT_NAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.FORM_TEMPLATE_FILENAME;
import static com.bizunited.platform.kuiper.starter.common.enums.FormTemplateTypeEnum.DYNAMIC;
import static com.bizunited.platform.kuiper.starter.common.enums.FormTemplateTypeEnum.STATIC;
import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * 表单模版的服务实现
 * @author yinwenjie
 */
@Service("TemplateServiceImpl")
public class TemplateServiceImpl implements TemplateService {

  private static final Logger LOGGER = LoggerFactory.getLogger(TemplateServiceImpl.class);

  private static final String NULLABLE = "nullable";
  private static final String FIELD_NAME = "fieldName";
  private static final String TYPE = "type";
  private static final String DESC = "desc";
  private static final String OBJECT = "Object";
  private static final String OBJECT_ARRAY = "ObjectArray";
  private static final String KEY = "key";
  private static final String FORMAT_S = "%s-%s";
  private static final String VALIDATE_MESSAGE = "未找到指定的模板信息，请检查!!";

  @Autowired
  private TemplateRepository templateRepository;
  @Autowired
  private TemplateGroupRepository templateGroupRepository;
  @Autowired
  private TemplateItemRepository templateItemRepository;
  @Autowired
  private TemplateEventService templateEventService;
  @Autowired
  private TemplateVisibilityService templateVisibilityService;
  @Autowired
  private TemplateLayoutService templateLayoutService;
  @Autowired
  @Qualifier("KuiperToolkitService")
  private KuiperToolkitService kuiperToolkitService;
  @Autowired
  private StaticTemplateService staticTemplateService;
  @Autowired
  private DynamicTemplateService dynamicTemplateService;
  @Autowired
  private TemplateItemExcelService templateItemExcelService;
  @Autowired
  private PlatformContext platformContext;


  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateService#findById(java.lang.String)
   */
  @Override
  @Cacheable(cacheNames="template" , key="'findById_' + #id" , sync=true)
  public TemplateEntity findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    Optional<TemplateEntity> op = this.templateRepository.findById(id);
    TemplateEntity currentTemplate = op.orElse(null);

    if(currentTemplate == null) {
      return null;
    }

    // 由于要保存到缓存，所以不能有持久化对象
    return this.kuiperToolkitService.copyObjectByWhiteList(currentTemplate, TemplateEntity.class, LinkedHashSet.class, LinkedList.class);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateService#findByCode(java.lang.String)
   */
  @Override
  @Cacheable(cacheNames="template" , key="'findByCode_' + #code" , sync=true)
  public List<TemplateEntity> findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return new ArrayList<>();
    }
    
    List<TemplateEntity> templates = this.templateRepository.findByCode(code);
    if(templates == null || templates.isEmpty()) {
      return new ArrayList<>();
    }
    // 由于要保存到缓存，所以不能有持久化对象
    Collection<TemplateEntity> resutls = this.kuiperToolkitService.copyCollectionByWhiteList(templates, TemplateEntity.class, TemplateEntity.class, LinkedHashSet.class, LinkedList.class);
    return new LinkedList<>(resutls);
  }
  @Override
  public TemplateEntity findByDefaultVersion(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.templateRepository.findByDefaultVersion(code);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateService#findById(java.lang.String)
   */
  @Override
  @Cacheable(cacheNames="template" , key="'findDetailsById_' + #id" , sync=true)
  public TemplateEntity findDetailsById(String id) {
    // 注意：为了性能考虑，不能一次连接10张表，分成三次查询出来
    if(StringUtils.isBlank(id)) {
      return null;
    }

    // 模板的一般属性和关联属性
    TemplateEntity currentTemplate = this.templateRepository.findDetailsById(id);
    if(currentTemplate == null) {
      return null;
    }
    // 排序
    Set<TemplatePropertyEntity> properties = currentTemplate.getProperties();
    currentTemplate.setProperties(this.sortedProperties(properties));
    Set<TemplateRelationEntity> relations = currentTemplate.getRelations();
    currentTemplate.setRelations(this.sortedRelations(relations));
    
    // 分组信息下的一般属性、关联属性和可能的明细编辑属性
    Set<TemplateGroupEntity> templateGroups = this.templateGroupRepository.findDetailsByParentTemplate(id);
    if(templateGroups != null) {
      for (TemplateGroupEntity templateGroup : templateGroups) {
        templateGroup.setProperties(this.sortedProperties(templateGroup.getProperties()));
        templateGroup.setRelations(this.sortedRelations(templateGroup.getRelations()));
        Set<TemplateItemEntity> templateItems = templateGroup.getItemRelations();
        if(templateItems != null) {
          for (TemplateItemEntity templateItem : templateItems) {
            templateItem.setProperties(this.sortedProperties(templateItem.getProperties()));
            templateItem.setRelations(this.sortedRelations(templateItem.getRelations()));
          }
        }
      }
    }
    currentTemplate.setGroupRelations(templateGroups);
    
    // 明细编辑属性下的一般属性和关联属性
    Set<TemplateItemEntity> templateItems = this.templateItemRepository.findDetailsByParentTemplate(id);
    // 再排排序
    if(templateItems != null) {
      for (TemplateItemEntity templateItem : templateItems) {
        templateItem.setProperties(this.sortedProperties(templateItem.getProperties()));
        templateItem.setRelations(this.sortedRelations(templateItem.getRelations()));
      }
    }
    currentTemplate.setItemRelations(templateItems);
    
    // 由于要保存到缓存，所以不能有持久化对象
    return this.kuiperToolkitService.copyObjectByWhiteList(currentTemplate, TemplateEntity.class, LinkedHashSet.class, LinkedList.class, "properties", "relations","creator",
                                                                                                                                                "groupRelations","groupRelations.properties","groupRelations.relations","groupRelations.itemRelations",
                                                                                                                                                "groupRelations.itemRelations.properties","groupRelations.itemRelations.relations",
                                                                                                                                                "itemRelations","itemRelations.properties","itemRelations.relations");
  }
  
  @Override
  public List<TemplateEntity> findByFormStyle(String formStyle) {
    if(StringUtils.isBlank(formStyle)) {
      return Lists.newArrayList();
    }
    String appName = platformContext.getAppName();
    if(StringUtils.isNotBlank(appName)) {
      return this.templateRepository.findByFormStyleAndProjectName(formStyle, appName);
    }
    return this.templateRepository.findByFormStyleAndBlankProjectName(formStyle);
  }
  @Override
  public TemplateEntity findByCodeAndCversion(String templateCode, String cversion) {
    if(StringUtils.isAnyBlank(templateCode, cversion)) {
      return null;
    }
    return templateRepository.findByCodeAndCversion(templateCode, cversion);
  }
  @Override
  public TemplateEntity findDetailsByCodeAndVersion(String code, String version) {
    if(StringUtils.isAnyBlank(code, version)) {
      return null;
    }
    TemplateEntity template = templateRepository.findByCodeAndCversion(code, version);
    if(template == null) {
      return null;
    }
    return this.findDetailsById(template.getId());
  }

  @Override
  public List<TemplateEntity> findAllByConditions(TemplateEntity condition) {
    condition = ObjectUtils.defaultIfNull(condition, new TemplateEntity());
    condition.setProjectName(platformContext.getAppName());
    return templateRepository.findAllByConditions(condition);
  }

  @Override
  public List<TemplateEntity> findDetailsByIds(String[] ids) {
    if(ArrayUtils.isEmpty(ids)) {
      return Lists.newArrayList();
    }
    return templateRepository.findDetailsByIds(ids);
  }

  @Override
  @Transactional(REQUIRES_NEW)
  @SuppressWarnings("unchecked")
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    importModel.appendLine("开始导入数据");
    ZipEntry templateEntry = zipFile.getEntry(FORM_TEMPLATE_FILENAME);
    if(templateEntry == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
      return;
    }
    try (InputStream is = zipFile.getInputStream(templateEntry);
         ObjectInputStream ois = new ObjectInputStream(is)) {
      List<TemplateEntity> templates = (List<TemplateEntity>) ois.readObject();
      Map<String, List<TemplateItemExcelEntity>> templateItemExcel = (Map<String, List<TemplateItemExcelEntity>>) ois.readObject();
      if(CollectionUtils.isEmpty(templates)) {
        importModel.appendLine("导入的数据为空，请检查数据");
        return;
      }
      this.importData(templates, templateItemExcel, importModel);
    } catch (IOException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      importModel.append("读取模版数据失败：").appendLine(e.getMessage());
    }
  }

  /**
   * 批量导入表单模版
   * @param templates
   * @param templateItemExcel
   * @param importModel
   */
  private void importData(List<TemplateEntity> templates, Map<String, List<TemplateItemExcelEntity>> templateItemExcel, MigrateImportModel importModel) {
    importModel.setTotalCount(templates.size());
    for (int i = 0; i < templates.size(); i++) {
      TemplateEntity template = templates.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      String key = String.format("%s|%s", template.getCode(), template.getCversion());
      List<TemplateItemExcelEntity> templateItemExcels = templateItemExcel.get(key);
      this.importData(template, templateItemExcels, importModel);
    }
  }

  /**
   * 处理单个表单模版的导入
   * @param template
   * @param templateItemExcels
   * @param importModel
   */
  private void importData(TemplateEntity template, List<TemplateItemExcelEntity> templateItemExcels, MigrateImportModel importModel) {
    importModel.appendLine(String.format("开始导入数据：编号=%s,版本号=%s", template.getCode(), template.getCversion()));
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    TemplateEntity dbTemplate = templateRepository.findByCodeAndCversion(template.getCode(), template.getCversion());
    if(dbTemplate != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("模版数据已存在，执行跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbTemplate != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      // 更新模版的操作太过复杂，所以暂时不做模版导入更新的操作
      importModel.appendLine("模版数据已存在，执行跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbTemplate != null && ImportExecuteModeEnum.ADD == executeMode) {
      this.handleCreateDataOnExist(template, templateItemExcels, importModel);
      return;
    }
    if(dbTemplate == null) {
      this.handleCreateData(template, templateItemExcels, importModel);
      return;
    }
  }

  /**
   * 处理模版的更新导入
   * @param template
   * @param dbTemplate
   * @param templateItemExcels
   * @param importModel
   */
  private void handleCreateDataOnExist(TemplateEntity template, List<TemplateItemExcelEntity> templateItemExcels, MigrateImportModel importModel) {
    importModel.appendLine("模版数据已存在，执行版本升级导入");
    String newVersion;
    int count = 1;
    long versionCount;
    do {
      newVersion = StringUtils.join(template.getCversion(), "_", count);
      count++;
      versionCount = templateRepository.countByCodeAndVersion(template.getCode(), newVersion);
    } while (versionCount > 0);
    importModel.append("生成新的版本号：").appendLine(newVersion);
    importModel.appendLine("导入模版数据");
    template.setCversion(newVersion);
    this.handleCreateData(template, templateItemExcels, importModel);
  }

  /**
   * 处理表单模版的新增导入
   * @param template
   * @param templateItemExcels
   * @param importModel
   */
  private void handleCreateData(TemplateEntity template, List<TemplateItemExcelEntity> templateItemExcels, MigrateImportModel importModel) {
    importModel.appendLine("导入新增模版数据：编号=%s，版本=%s", template.getCode(), template.getCversion());
    // 将模版信息复制一份，用于保留原始的导入模版信息
    TemplateEntity importTemplate = SerializationUtils.clone(template);
    template.setDefaultVersion(false);
    this.handleTemplateDataId(template);
    Principal principal = SecurityUtils.getPrincipal();
    if(DYNAMIC.getType().equals(template.getType())) {
      importModel.appendLine("初始化动态模版");
      this.handleDynamicTemplate(template);
      TemplateEntity dbTemplate = dynamicTemplateService.initDynamicTemplate(template, principal, true);
      importTemplate.setId(dbTemplate.getId());
    } else if (STATIC.getType().equals(template.getType())) {
      importModel.appendLine("初始化静态模版");
      TemplateEntity dbTemplate = staticTemplateService.initStaticTemplate(template, principal, false);
      importTemplate.setId(dbTemplate.getId());
    }
    // 保存关联的可见性，布局,事件
    importModel.appendLine("保存模版的可见性、布局、事件");
    this.handleEventScript(importTemplate, importModel);
    this.update(importTemplate);
    templateRepository.flush();
    importModel.appendLine("导入模版明细项的excel模版");
    templateItemExcelService.importData(template, templateItemExcels, importModel);
  }

  /**
   * 处理表单模版事件的脚本内容， 从zip压缩包中读取内容放到事件对象中
   * @param template
   * @param importModel
   */
  private void handleEventScript(TemplateEntity template, MigrateImportModel importModel) {
    Set<TemplateEventEntity> events = template.getEvent();
    if(CollectionUtils.isEmpty(events)) {
      return;
    }
    ZipFile zipFile = importModel.getZipFile();
    for (TemplateEventEntity event : events) {
      try {
        byte[] bytes = ZipFileUtils.readZipFile(zipFile, event.getRelativePath(), event.getFileName());
        event.setJavascriptDesc(new String(bytes, StandardCharsets.UTF_8));
      } catch (IOException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(String.format("从压缩包读取文件出错：%s/%s", event.getRelativePath(), event.getFileName()), e);
      }
    }
  }

  /**
   * 处理模版数据中的ID属性
   * @param template
   * @return
   */
  private TemplateEntity handleTemplateDataId(TemplateEntity template) {
    template.setId(null);
    this.handlePropertiesId(template.getProperties());
    this.handleEventId(template.getEvent());
    this.handleGroupId(template.getGroupRelations());
    this.handleRelationId(template.getRelations());
    this.handleVisibilityId(template.getVisibility());
    return template;
  }

  /**
   * 处理动态表单
   * @param template
   * @return
   */
  private TemplateEntity handleDynamicTemplate(TemplateEntity template) {
    this.removeIdProperty(template.getProperties());
    this.removeFormInstanceIdProperty(template.getProperties());
    Set<TemplateItemEntity> items = this.handleDynamicTemplateItem(template.getItemRelations());
    template.setItemRelations(items);
    Set<TemplateGroupEntity> groupRelations = template.getGroupRelations();
    if(!CollectionUtils.isEmpty(groupRelations)) {
      for (TemplateGroupEntity groupRelation : groupRelations) {
        Set<TemplateItemEntity> itemRelations = this.handleDynamicTemplateItem(groupRelation.getItemRelations());
        groupRelation.setItemRelations(itemRelations);
        this.removeIdProperty(groupRelation.getProperties());
        this.removeDynamicTemplateItemParentRelation(groupRelation.getParentTableName(), groupRelation.getRelations());
      }
    }
    return template;
  }

  /**
   * 移除明细项中的id属性
   * @param itemRelations
   * @return
   */
  private Set<TemplateItemEntity> handleDynamicTemplateItem(Set<TemplateItemEntity> itemRelations) {
    if(CollectionUtils.isEmpty(itemRelations)) {
      return itemRelations;
    }
    for (TemplateItemEntity itemRelation : itemRelations) {
      Set<TemplatePropertyEntity> properties = itemRelation.getProperties();
      this.removeIdProperty(properties);
      this.removeDynamicTemplateItemParentRelation(itemRelation.getParentTableName(), itemRelation.getRelations());
    }
    return itemRelations;
  }

  /**
   * 移除分组中关联的父模型的关联信息
   * @param item
   * @return
   */
  private Set<TemplateRelationEntity> removeDynamicTemplateItemParentRelation(String parentTableName, Set<TemplateRelationEntity> relations) {
    if(CollectionUtils.isEmpty(relations)) {
      return relations;
    }
    Iterator<TemplateRelationEntity> iterator = relations.iterator();
    while (iterator.hasNext()) {
      TemplateRelationEntity relation = iterator.next();
      if(relation.getTargetTableName().equals(parentTableName)) {
        iterator.remove();
      }
    }
    return relations;
  }

  /**
   * 移除ID属性
   * @param properties
   * @return
   */
  private Set<TemplatePropertyEntity> removeIdProperty(Set<TemplatePropertyEntity> properties) {
    if(CollectionUtils.isEmpty(properties)) {
      return properties;
    }
    Iterator<TemplatePropertyEntity> iterator = properties.iterator();
    while (iterator.hasNext()) {
      TemplatePropertyEntity property = iterator.next();
      if(StringUtils.equals(property.getPropertyDbName(), DEFAULT_DB_PK_NAME)) {
        iterator.remove();
        continue;
      }
    }
    return properties;
  }

  /**
   * 移除属性中的表单实例ID属性
   * @param properties
   * @return
   */
  private Set<TemplatePropertyEntity> removeFormInstanceIdProperty(Set<TemplatePropertyEntity> properties) {
    if(CollectionUtils.isEmpty(properties)) {
      return properties;
    }
    Iterator<TemplatePropertyEntity> iterator = properties.iterator();
    while (iterator.hasNext()) {
      TemplatePropertyEntity property = iterator.next();
      if(StringUtils.equals(property.getPropertyDbName(), DEFAULT_DB_FORM_INSTANCE_ID_NAME)) {
        iterator.remove();
        continue;
      }
    }
    return properties;
  }

  /**
   * 处理基本属性的ID为null
   * @param properties
   */
  private void handlePropertiesId(Set<TemplatePropertyEntity> properties) {
    if(CollectionUtils.isEmpty(properties)) {
      return;
    }
    properties.forEach(p -> p.setId(null));
  }

  /**
   * 处理可见性的ID为null
   * @param visibilities
   */
  private void handleVisibilityId(Set<TemplateVisibilityEntity> visibilities) {
    if(CollectionUtils.isEmpty(visibilities)) {
      return;
    }
    for (TemplateVisibilityEntity visibility : visibilities) {
      visibility.setId(null);
      if(!CollectionUtils.isEmpty(visibility.getAttributes())) {
        visibility.getAttributes().forEach(a -> a.setId(null));
      }
      if(!CollectionUtils.isEmpty(visibility.getButtons())) {
        visibility.getButtons().forEach(b -> b.setId(null));
      }
    }
  }

  /**
   * 处理关联属性的ID
   * @param relations
   */
  private void handleRelationId(Set<TemplateRelationEntity> relations) {
    if(CollectionUtils.isEmpty(relations)) {
      return;
    }
    for (TemplateRelationEntity relation : relations) {
      relation.setId(null);
    }
  }

  /**
   * 处理分组ID
   * @param groupRelations
   */
  private void handleGroupId(Set<TemplateGroupEntity> groupRelations) {
    if(CollectionUtils.isEmpty(groupRelations)) {
      return;
    }
    for (TemplateGroupEntity groupRelation : groupRelations) {
      groupRelation.setId(null);
      this.handlePropertiesId(groupRelation.getProperties());
      this.handleRelationId(groupRelation.getRelations());
      this.handleItemId(groupRelation.getItemRelations());
    }
  }

  /**
   * 处理明细项的ID
   * @param itemRelations
   */
  private void handleItemId(Set<TemplateItemEntity> itemRelations) {
    if(CollectionUtils.isEmpty(itemRelations)) {
      return;
    }
    for (TemplateItemEntity itemRelation : itemRelations) {
      itemRelation.setId(null);
      this.handlePropertiesId(itemRelation.getProperties());
      this.handleRelationId(itemRelation.getRelations());
    }
  }

  /**
   * 处理事件的ID为null
   * @param events
   */
  private void handleEventId(Set<TemplateEventEntity> events) {
    if(CollectionUtils.isEmpty(events)) {
      return;
    }
    events.forEach(e -> e.setId(null));
  }

  /**
   * 该私有方法用于完成一般属性的排序工作
   * @param properties
   * @return
   */
  private Set<TemplatePropertyEntity> sortedProperties(Set<TemplatePropertyEntity> properties) {
    Set<TemplatePropertyEntity> currentProperties = null;
    if(properties != null) {
      currentProperties = new LinkedHashSet<>();
      currentProperties.addAll(properties.stream().sorted(Comparator.comparing(TemplatePropertyEntity::getIndex)).collect(Collectors.toList()));
    }
    return currentProperties;
  }
  /**
   * 该私有方法用于完成关联属性的排序工作
   * @param relations
   * @return 
   */
  private Set<TemplateRelationEntity> sortedRelations(Set<TemplateRelationEntity> relations) {
    Set<TemplateRelationEntity> currentRelations = null;
    if(relations != null) {
      currentRelations = new LinkedHashSet<>();
      currentRelations.addAll(relations.stream().sorted(Comparator.comparing(TemplateRelationEntity::getIndex)).collect(Collectors.toList()));
    }
    return currentRelations;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateService#findByConditions(java.lang.String, java.lang.String, java.lang.String, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<TemplateEntity> findByConditions(Map<String, Object> conditions, Pageable pageable) {
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if(pageable == null) {
      pageable = PageRequest.of(0, DEFAULT_PAGE_SIZE);
    }
    if(conditions == null) {
      conditions = new HashMap<>();
    }
    conditions.put(PROJECT_NAME, platformContext.getAppName());
    return this.templateRepository.findByConditions(pageable, conditions);
  }
  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return 0;
    }
    return this.templateRepository.countByIds(ids);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateService#update(com.bizunited.platform.kuiper.entity.TemplateEntity)
   */
  @Override
  @Transactional
  @CacheEvict(cacheNames="template" , allEntries=true)
  public TemplateEntity update(TemplateEntity template) {
    /*
     * 处理过程为：
     * 1、首先表单模板的基本信息是不允许修改的（只有部分允许修改），但是必须要保证表单模板的id是否有效
     * 2、调用相关接口对表单模板的布局信息进行修改
     * 3、调用相关接口对表单事件信息进行修改
     * 4、调用相关接口对表单可见性进行修改
     * */
    Validate.notNull(template , "需要保存的表单模板信息必须传入!!");
    String templateId = template.getId();
    Validate.notBlank(templateId , "表单模板编号必须传入!!");
    Optional<TemplateEntity> op = this.templateRepository.findById(templateId);
    TemplateEntity currentTemplate = op.orElse(null);
    Validate.notNull(currentTemplate , "未找到当前指定编号对应的模板信息!!");

    // 1、=======
    currentTemplate.setCode(template.getCode());
    currentTemplate.setCversion(template.getCversion());
    currentTemplate.setDomain(template.getDomain());
    currentTemplate.setFormStyle(template.getFormStyle());
    currentTemplate.setName(template.getName());
    currentTemplate.setType(template.getType());
    currentTemplate.setModifyTime(new Date());
    // 2、======
    try {
      Object templateLayout = template.getTemplateLayout();
      Validate.notNull(templateLayout , "错误的表单模板布局信息，请检查!!");
      TemplateLayoutEntity currentLayout = this.templateLayoutService.save(templateId , 1 , templateLayout);
      Validate.notNull(currentLayout , "错误的表单模板布局保存过程，请检查!!");
      // 试图对移动端和打印端的布局进行处理
      Object mobileTemplateLayout = template.getMobileTemplateLayout();
      if(mobileTemplateLayout != null) {
        currentLayout = this.templateLayoutService.save(templateId , 2 , mobileTemplateLayout);
        Validate.notNull(currentLayout , "错误的表单模板布局保存过程（移动端），请检查!!");
      }
      Object printTemplateLayout = template.getPrintTemplateLayout();
      if(printTemplateLayout != null) {
        currentLayout = this.templateLayoutService.save(templateId , 3 , printTemplateLayout);
        Validate.notNull(currentLayout , "错误的表单模板布局保存过程（打印端），请检查!!");
      }
    } catch (IOException e) {
      throw new IllegalArgumentException(e);
    }
    // 3、======
    Set<TemplateEventEntity> templateEvents = template.getEvent();
    List<TemplateEventEntity> events;
    if(CollectionUtils.isEmpty(templateEvents)) {
      events = new ArrayList<>();
    }else {
      events = new ArrayList<>(templateEvents);
    }
    this.templateEventService.save(templateId, events);
    // 4、======
    Set<TemplateVisibilityEntity> templateVisibilities = template.getVisibility();
    if(templateVisibilities != null && !templateVisibilities.isEmpty()) {
      this.templateVisibilityService.save(currentTemplate, templateVisibilities);
    }
    
    // 保存完毕后进行基本信息的返回
    return currentTemplate;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateService#updateDefaultVersion(java.lang.String)
   */
  @Override
  @Transactional
  @CacheEvict(cacheNames="template" , allEntries=true)
  public void updateDefaultVersion(String templateId) {
    Validate.notBlank(templateId , "指定默认版本信息时,原始指定的templateId必须传入");
    Optional<TemplateEntity> op = this.templateRepository.findById(templateId);
    TemplateEntity currentTemplate = op.orElse(null);
    Validate.notNull(currentTemplate , VALIDATE_MESSAGE);

    Validate.notNull(currentTemplate , VALIDATE_MESSAGE);
    Validate.isTrue(currentTemplate.getTstatus() == 1 , "当前指定模板的状态不正确，请检查（模板状态不能为禁用）!");
    
    this.templateRepository.updateDefaultVersion(currentTemplate.getCode());
    this.templateRepository.updateDefaultVersion(currentTemplate.getCode(), currentTemplate.getCversion(), new Date());
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.formengine.starter.service.TemplateService#updateStatusById(java.lang.Integer)
   */
  @Override
  @Transactional
  @CacheEvict(cacheNames="template" , allEntries=true)
  public void updateStatusById(String templateId , Integer tstatus) {
    Validate.isTrue(tstatus == 1 || tstatus == 0 , "模板状态只能设定为1（正常）或者0（禁用）");
    Validate.notBlank(templateId , "指定默认版本信息时,原始指定的templateId必须传入");
    Optional<TemplateEntity> op = this.templateRepository.findById(templateId);
    TemplateEntity currentTemplate = op.orElse(null);
    Validate.notNull(currentTemplate, VALIDATE_MESSAGE);

    Validate.isTrue(currentTemplate.getDefaultVersion() && tstatus == 1 || !currentTemplate.getDefaultVersion() , "当前指定的模板[%s]为默认版本，其状态不允许被禁用." , templateId);
    currentTemplate.setTstatus(tstatus);
    currentTemplate.setModifyTime(new Date());
    this.templateRepository.save(currentTemplate);
  }

  @Override
  public JSONArray importTemplate(String templateEntityId) {
    Validate.notBlank(templateEntityId, "导入模板ID不能为空！");
    TemplateEntity template = this.findDetailsById(templateEntityId);
    Validate.notNull(template, "未找到该模板，请检查！");
    Set<TemplatePropertyEntity> properties = template.getProperties();
    Set<TemplateItemEntity> items = template.getItemRelations();
    Set<TemplateGroupEntity> groups = template.getGroupRelations();
    Set<TemplateRelationEntity> relations = template.getRelations();
    JSONArray json = new JSONArray();
    Integer index = 1;
    // 一般属性
    if (!CollectionUtils.isEmpty(properties)) {
      json.addAll(this.propertiesToJson(properties, null, index));
      index += properties.size();
    }
    // 明细属性
    if (!CollectionUtils.isEmpty(items)) {
      json.addAll(this.itemsToJson(items, null, index));
      index += items.size();
    }
    // 关联属性
    if (!CollectionUtils.isEmpty(relations)) {
      json.addAll(this.relationsToJson(relations, null, index));
      index += relations.size();
    }
    // 分组属性
    if (!CollectionUtils.isEmpty(groups)) {
      json.addAll(this.groupsToJson(groups, null, index));
    }
    return json;
  }

  /**
   * 将一般属性转化为JSON
   *
   * @param properties
   * @return
   */
  private JSONArray propertiesToJson(
      Set<TemplatePropertyEntity> properties, String oldSort, Integer currentIndex) {
    JSONArray propertyArray = new JSONArray();
    for (TemplatePropertyEntity o : properties) {
      JSONObject propertyJson = new JSONObject();
      String place =
          StringUtils.isEmpty(oldSort)
              ? currentIndex.toString()
              : String.format(FORMAT_S, oldSort, currentIndex.toString());
      propertyJson.put(KEY, place);
      propertyJson.put(FIELD_NAME, o.getPropertyName());
      propertyJson.put(DESC, o.getPropertyDesc());
      propertyJson.put(NULLABLE, o.getNullable());
      propertyArray.add(propertyJson);
      currentIndex = currentIndex + 1;
    }
    return propertyArray;
  }

  /**
   * 关联属性转化JSON
   *
   * @param relations
   * @return
   */
  private JSONArray relationsToJson(
      Set<TemplateRelationEntity> relations, String oldSort, Integer currentIndex) {
    JSONArray relationArray = new JSONArray();
    for (TemplateRelationEntity o : relations) {
      JSONObject relationJson = new JSONObject();
      String place =
          StringUtils.isEmpty(oldSort)
              ? currentIndex.toString()
              : String.format(FORMAT_S, oldSort, currentIndex.toString());
      relationJson.put(KEY, place);
      relationJson.put(FIELD_NAME, o.getPropertyName());
      relationJson.put(DESC, o.getPropertyDesc());
      relationJson.put(NULLABLE, o.getNullable());
      relationArray.add(relationJson);
      currentIndex = currentIndex + 1;
    }

    return relationArray;
  }

  /**
   * 明细转化为JSON
   *
   * @param items
   * @return
   */
  private JSONArray itemsToJson(
      Set<TemplateItemEntity> items, String oldSort, Integer currentIndex) {
    JSONArray itemArray = new JSONArray();

    for (TemplateItemEntity o : items) {
      JSONObject itemJson = new JSONObject();
      String place =
          StringUtils.isEmpty(oldSort)
              ? currentIndex.toString()
              : String.format(FORMAT_S, oldSort, currentIndex.toString());
      itemJson.put(KEY, place);
      itemJson.put(FIELD_NAME, o.getPropertyName());
      itemJson.put(DESC, o.getPropertyDesc());
      itemJson.put(NULLABLE, true);
      itemJson.put(TYPE, OBJECT);
      itemJson.put("content", this.propertiesToJson(o.getProperties(), place, 1));
      itemArray.add(itemJson);
    }
    return itemArray;
  }

  /**
   * 分组转化为JSON
   *
   * @param groups
   * @return
   */
  private JSONArray groupsToJson(
      Set<TemplateGroupEntity> groups, String oldSort, Integer currentIndex) {
    JSONArray groupArray = new JSONArray();
    for (TemplateGroupEntity o : groups) {
      JSONObject groupJson = new JSONObject();
      String place =
          StringUtils.isEmpty(oldSort)
              ? currentIndex.toString()
              : String.format(FORMAT_S, oldSort, currentIndex.toString());
      groupJson.put(KEY, place);
      groupJson.put(FIELD_NAME, o.getPropertyName());
      groupJson.put(DESC, o.getPropertyDesc());
      groupJson.put(NULLABLE, true);
      groupJson.put(TYPE, OBJECT_ARRAY);
      JSONArray groupContent = new JSONArray();
      Integer subIndex = 1;
      if(!CollectionUtils.isEmpty(o.getProperties())){
        groupContent.addAll(this.propertiesToJson(o.getProperties(), place, subIndex));
      }
      subIndex =
          CollectionUtils.isEmpty(o.getProperties())
              ? subIndex
              : subIndex + o.getProperties().size();
      if(!CollectionUtils.isEmpty(o.getRelations())){
        groupContent.addAll(this.relationsToJson(o.getRelations(), place, subIndex));
      }
      subIndex =
          CollectionUtils.isEmpty(o.getRelations()) ? subIndex : subIndex + o.getRelations().size();
      if(!CollectionUtils.isEmpty(o.getItemRelations())){
        groupContent.addAll(this.itemsToJson(o.getItemRelations(), place, subIndex));
      }
      groupJson.put("content", groupContent);
      groupArray.add(groupJson);
    }

    return groupArray;
  }

}
