package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateStyleEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.repository.InstanceRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateStyleRepository;
import com.bizunited.platform.kuiper.starter.service.TemplateStyleService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 表单样式服务的实现类
 * @author yinwenjie
 */
@Service("TemplateStyleServiceImpl")
public class TemplateStyleServiceImpl implements TemplateStyleService {

  @Autowired
  private TemplateStyleRepository templateStyleRepository;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private InstanceRepository instanceRepository;
  @Autowired
  private VenusFileService kuiperFileService;

  private static final String FORMAT = "/";
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateStyleService#create(com.bizunited.platform.kuiper.entity.TemplateStyleEntity, byte[], boolean, java.lang.String)
   */
  @Override
  @Transactional
  public TemplateStyleEntity create(TemplateStyleEntity templateStyle, byte[] fileContent , boolean systemStyle , String prefix) {
    Validate.notNull(templateStyle , "进行创建操作时，模板样式对象必须传入");
    String id = templateStyle.getId();
    Validate.isTrue(StringUtils.isBlank(id) , "进行创建操作时，不能指定数据编号!!");
    templateStyle.setId(null);
    String code = templateStyle.getCode();
    Validate.notBlank(code , "进行创建操作时，模板编号信息必须传入!!");
    Pattern pattern = Pattern.compile("[\u4E00-\u9FA5]");
    Validate.isTrue(!pattern.matcher(code).find() , "进行创建时，模板编号信息不能带有中文，请检查!!");
    TemplateStyleEntity currentTemplate = this.templateStyleRepository.findDetailsByCode(code);
    Validate.isTrue(currentTemplate == null , "进行创建操作时，当前模板业务编号已经存在，请重新填写!!");
    String name = templateStyle.getName();
    Validate.notBlank(name , "进行创建时，当前样式名必须填写!!");
    Validate.notNull(fileContent , "进行创建时，必须传入当前样式文件的内容信息");
    Validate.notBlank(prefix , "进行创建时，文件扩展信息必须指定!!");
    String styleType = StringUtils.lowerCase(prefix);
    Validate.isTrue(StringUtils.equalsAny(styleType, "css" , "less") , "进行创建时，文件后缀只能使用css或者less!!");
    
    // 1、======== 先保存文件
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String renameFile = StringUtils.join(UUID.randomUUID().toString() , "." , prefix);
    String subSystem = "style";
    String relativePath = FORMAT + subSystem + FORMAT + folderName + FORMAT + (new Random().nextInt(100) % 10);
    this.kuiperFileService.saveFile(relativePath, renameFile, renameFile, fileContent);
    
    // 2、======== 判断完成，开始创建保存文件
    templateStyle.setCreateTime(new Date());
    templateStyle.setRelativePath(relativePath);
    templateStyle.setFileName(renameFile);
    templateStyle.setStyleType(styleType);
    if(systemStyle) {
      templateStyle.setSystemStyle(1);
    } else {
      templateStyle.setSystemStyle(0);
    }
    templateStyle.setTstatus(1);
    templateStyle.setProjectName(ApplicationContextUtils.getProjectName());
    this.templateStyleRepository.save(templateStyle);
    return templateStyle;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateStyleService#update(com.bizunited.platform.kuiper.entity.TemplateStyleEntity, byte[])
   */
  @Override
  @Transactional
  public TemplateStyleEntity update(TemplateStyleEntity templateStyle, byte[] fileContent , String prefix) {
    Validate.notNull(templateStyle , "进行修改操作时，模板样式对象必须传入");
    String id = templateStyle.getId();
    Optional<TemplateStyleEntity> op = this.templateStyleRepository.findById(id);
    TemplateStyleEntity currentTemplateStyle = op.orElse(null);
    Validate.notNull(currentTemplateStyle , "进行修改操作时，模板样式对应必须传入!!");

    Validate.isTrue(currentTemplateStyle.getSystemStyle() != 1 , "当前样式信息为系统样式，不允许进行修改!!");
    // 如果使用当前样式（formStyle）的模板已经存在了实例信息，则不允许进行样式修改了
    String code = currentTemplateStyle.getCode();
    List<TemplateEntity> templates = this.templateService.findByFormStyle(code);
    if(templates != null && !templates.isEmpty()) {
      List<String> templateIds = templates.stream().map(TemplateEntity::getId).collect(Collectors.toList());
      int insCount = this.instanceRepository.countByTemplateIds(templateIds.toArray(new String[]{}));
      Validate.isTrue(insCount == 0 , "当前模板样式[" + code + "]已经存在至少一个实例，不允许进行修改!!");
    }
    
    // 可能修改的name
    String name = templateStyle.getName();
    if(!StringUtils.isBlank(name)) {
      currentTemplateStyle.setName(name);
    }
    
    // 可能修改的文件内容
    if(fileContent != null) {
      Validate.notBlank(prefix , "进行创建时，文件扩展信息必须指定!!");
      String styleType = StringUtils.lowerCase(prefix);
      Validate.isTrue(StringUtils.equalsAny(styleType, "css" , "less") , "进行创建时，文件后缀只能使用css或者less!!");
      Date nowDate = new Date();
      String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
      String renameFile = StringUtils.join(UUID.randomUUID().toString() , "." , prefix);
      String subSystem = "style";
      String relativePath = FORMAT + subSystem + FORMAT + folderName + FORMAT + (new Random().nextInt(100) % 10);
      this.kuiperFileService.saveFile(relativePath, renameFile, renameFile, fileContent);
      
      currentTemplateStyle.setRelativePath(relativePath);
      currentTemplateStyle.setFileName(renameFile);
      currentTemplateStyle.setStyleType(styleType);
    }
    
    // 保存
    this.templateStyleRepository.save(currentTemplateStyle);
    return currentTemplateStyle;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateStyleService#updateStatusById(java.lang.String, boolean)
   */
  @Override
  @Transactional
  public void updateStatusById(String id, boolean tstatus) {
    Validate.notBlank(id , "修改状态时，指定模板样式的编号必须传入!!");
    Optional<TemplateStyleEntity> op = this.templateStyleRepository.findById(id);
    TemplateStyleEntity currentTemplateStyle = op.orElse(null);
    Validate.notNull(currentTemplateStyle , "进行修改操作时，模板样式对应必须传入!!");

    Validate.isTrue(currentTemplateStyle.getSystemStyle() != 1 , "当前样式信息为系统样式，不允许进行修改!!");
    
    if(tstatus) {
      currentTemplateStyle.setTstatus(1);
    } else {
      currentTemplateStyle.setTstatus(0);
    }
    this.templateStyleRepository.save(currentTemplateStyle);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateStyleService#findAll()
   */
  @Override
  public List<TemplateStyleEntity> findAll() {
    return this.templateStyleRepository.findAllByProjectName(ApplicationContextUtils.getProjectName());
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateStyleService#findByTstatus(java.lang.Integer)
   */
  @Override
  public List<TemplateStyleEntity> findByTstatus(Integer tstatus) {
    if(tstatus == null) {
      return Lists.newArrayList();
    }
    return this.templateStyleRepository.findByTstatusAndProjectName(tstatus, ApplicationContextUtils.getProjectName());
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateStyleService#findById(java.lang.String)
   */
  @Override
  public TemplateStyleEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.templateStyleRepository.findDetailsById(id);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateStyleService#findDetailsByCode(java.lang.String)
   */
  @Override
  public TemplateStyleEntity findDetailsByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.templateStyleRepository.findDetailsByCode(code);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateStyleService#findContentByCode(java.lang.String)
   */
  @Override
  public byte[] findContentByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return new byte[0];
    }
    
    // 读取文件
    TemplateStyleEntity currentTemplate = this.templateStyleRepository.findDetailsByCode(code);
    if(currentTemplate == null) {
      return new byte[0];
    }
    
    String relativePath = currentTemplate.getRelativePath();
    String fileName = currentTemplate.getFileName();
    return this.kuiperFileService.readFileContent(relativePath, fileName);
  }
}
