package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityAttributesEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.starter.repository.TemplateVisibilityAttributesRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityAttributeService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import static com.bizunited.platform.kuiper.starter.common.enums.TemplateLayoutTypeEnum.PC;

/**
 * 表单模版可见性属性服务接口实现
 * @Author: Paul Chan
 * @Date: 2019-07-22 14:52
 */
@Service("TemplateVisibilityAttributeServiceImpl")
public class TemplateVisibilityAttributeServiceImpl implements TemplateVisibilityAttributeService {

  @Autowired
  @Qualifier("KuiperToolkitService")
  private KuiperToolkitService kuiperToolkitService;
  @Autowired
  private TemplateVisibilityAttributesRepository templateVisibilityAttributesRepository;

  /**
   * 保存前验证属性数据
   * @param visibility
   * @param attributes
   */
  private void saveValidation(TemplateVisibilityEntity visibility, Set<TemplateVisibilityAttributesEntity> attributes) {
    Validate.notNull(visibility, "可见性不能为空");
    if(CollectionUtils.isEmpty(attributes)) {
      return;
    }
    Set<String> controlIdSet = new HashSet<>();
    for (TemplateVisibilityAttributesEntity attribute : attributes) {
      Validate.notBlank(attribute.getControllerId(), "控件在该表单模板上的唯一编号不能为空，请检查!!");
      Validate.notNull(attribute.getNullable(), "该控件在这个可见性定义下，是否可以不填写值不能为空，请检查!!");
      Validate.notNull(attribute.getVisibilityType(), "控件的可见性类型不能为空，请检查!!");
      Integer layoutType = attribute.getLayoutType() == null ? PC.getType() : attribute.getLayoutType();
      Validate.isTrue(controlIdSet.add(StringUtils.join(layoutType, "|", attribute.getControllerId())), "属性controllerId编号重复,请检查!!");
    }
  }

  @Override
  @Transactional
  public Set<TemplateVisibilityAttributesEntity> save(TemplateVisibilityEntity visibility, Set<TemplateVisibilityAttributesEntity> attributes) {
    /**
     * 1、初始化可见性属性
     * 2、验证参数
     * 3、定义结果集
     * 4、从数据库获取旧的可见性属性数据，为null则初始化
     * 5、将提交的数据根据属性名称作为key存到map中
     * 6、定义集合，用于接收新增、更新、删除的数据
     * 7、调用kuiperToolkitService.collectionDiscrepancy方法获取新增、更新、删除的数据
     * 8、保存新增的数据
     *   1、设置可见性属性对象的属性值
     *   2、保存可见性属性
     *   3、将可见性属性添加到结果集中
     * 9、更新旧的数据
     *   1、设置可见性属性对象的属性值
     *   2、保存可见性属性
     *   3、将可见性属性添加到结果集中
     * 10、删除旧的数据
     * 11、返回结果集
     */
    if(attributes == null) attributes = new HashSet<>();
    this.saveValidation(visibility, attributes);
    Set<TemplateVisibilityAttributesEntity> savedAttributes = new HashSet<>();
    Set<TemplateVisibilityAttributesEntity> oldAttributes = templateVisibilityAttributesRepository.findByVisiblId(visibility.getId());
    if(oldAttributes == null) oldAttributes = new HashSet<>();
    Map<String, TemplateVisibilityAttributesEntity> attributesMap = new HashMap<>(16);
    for (TemplateVisibilityAttributesEntity attribute : attributes) {
      Integer layoutType = attribute.getLayoutType() == null ? PC.getType() : attribute.getLayoutType();
      attribute.setLayoutType(layoutType);
      String key = StringUtils.join(layoutType, "|", attribute.getControllerId());
      attributesMap.put(key, attribute);
    }
    Set<TemplateVisibilityAttributesEntity> addAttributes = new HashSet<>();
    Set<TemplateVisibilityAttributesEntity> updateAttributes = new HashSet<>();
    Set<TemplateVisibilityAttributesEntity> deleteAttributes = new HashSet<>();
    kuiperToolkitService.collectionDiscrepancy(attributes, oldAttributes, attribute -> {
      Integer layoutType = attribute.getLayoutType() == null ? PC.getType() : attribute.getLayoutType();
      return StringUtils.join(layoutType, "|", attribute.getControllerId());
    }, deleteAttributes, updateAttributes, addAttributes);
    // 保存新增的数据
    for (TemplateVisibilityAttributesEntity attribute : addAttributes) {
      attribute.setId(null);
      attribute.setTemplateVisibility(visibility);
      attribute.setProjectName(ApplicationContextUtils.getProjectName());
      templateVisibilityAttributesRepository.save(attribute);
      savedAttributes.add(attribute);
    }
    // 更新旧的数据
    for (TemplateVisibilityAttributesEntity attribute : updateAttributes) {
      Integer layoutType = attribute.getLayoutType() == null ? PC.getType() : attribute.getLayoutType();
      String key = StringUtils.join(layoutType, "|", attribute.getControllerId());
      TemplateVisibilityAttributesEntity newAttribute = attributesMap.get(key);
      attribute.setLayoutType(layoutType);
      attribute.setTemplateVisibility(visibility);
      attribute.setNullable(newAttribute.getNullable());
      attribute.setVisibilityType(newAttribute.getVisibilityType());
      attribute.setControllerId(newAttribute.getControllerId());
      templateVisibilityAttributesRepository.save(attribute);
      savedAttributes.add(attribute);
    }
    return savedAttributes;
  }

  @Override
  @Transactional
  public void deleteByVisibilityId(String visibilityId) {
    Set<TemplateVisibilityAttributesEntity> attributes = templateVisibilityAttributesRepository.findByVisiblId(visibilityId);
    if(!CollectionUtils.isEmpty(attributes)){
      templateVisibilityAttributesRepository.deleteAll(attributes);
    }
  }

  @Override
  public Set<TemplateVisibilityAttributesEntity> findByVisibilityIdAndLayoutType(String visibilityId, Integer layoutType) {
    if(StringUtils.isBlank(visibilityId)) {
      return Sets.newHashSet();
    }
    if(layoutType == null) {
      layoutType = PC.getType();
    }
    return templateVisibilityAttributesRepository.findByVisibilityIdAndLayoutType(visibilityId, layoutType);
  }

}
