package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.entity.UuidEntity;
import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.core.service.NebulaStaticPersistentService;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityAttributesEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityButtonsEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.common.enums.TemplateLayoutTypeEnum;
import com.bizunited.platform.kuiper.starter.repository.InstanceActivityRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateVisibilityAttributesRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateVisibilityButtonRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateVisibilityRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityAttributeService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityButtonService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService;
import com.bizunited.platform.saturn.model.PersistentClass;
import com.bizunited.platform.saturn.model.PersistentProperty;
import com.bizunited.platform.saturn.model.PersistentRelation;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@Service("TemplateVisibilityServiceImpl")
public class TemplateVisibilityServiceImpl implements TemplateVisibilityService {

  private static final String CONTROLLER_ID_SPLIT = "-";
  private static final String CONTROLLER_ID_SUFFIX = "Attri";
  private static final String VISIBILITY_CREATE = "create";

  @Autowired 
  private TemplateVisibilityRepository templateVisibilityRepository;
  @Autowired 
  private TemplateVisibilityAttributesRepository templateVisibilityAttributesRepository;
  @Autowired 
  private TemplateVisibilityButtonRepository templateVisibilityButtonRepository;
  @Autowired 
  private TemplateService templateService;
  @Autowired
  private NebulaStaticPersistentService nebulaStaticPersistentService;
  @Autowired
  private InstanceActivityRepository instanceActivityRepository;
  @Autowired
  @Qualifier("KuiperToolkitService")
  private KuiperToolkitService kuiperToolkitService;
  @Autowired
  private TemplateVisibilityButtonService templateVisibilityButtonService;
  @Autowired
  private TemplateVisibilityAttributeService templateVisibilityAttributeService;

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService#initStaticTemplateVisibilityAttributes(java.lang.String, com.bizunited.platform.saturn.model.PersistentClass)
   */
  @Override
  @Transactional
  public Set<TemplateVisibilityEntity> initStaticTemplateVisibilities(String templateId, PersistentClass persistentClass) {
    Validate.notBlank(templateId, "表单模板编号必须传入!!");
    Validate.notNull(persistentClass, "静态模型的信息必须传入!!");
    TemplateEntity template = this.templateService.findById(templateId);
    Validate.notNull(template, "未找到指定的表单模板信息!!");
    Validate.isTrue("static".equals(template.getType()), "该初始化方法只能针对静态模版");

    TemplateVisibilityEntity templateVisibility = new TemplateVisibilityEntity();
    templateVisibility.setCanDelete(false);
    templateVisibility.setCreateTime(new Date());
    templateVisibility.setVisibilityName(VISIBILITY_CREATE);
    templateVisibility.setTemplate(template);
    templateVisibility.setProjectName(ApplicationContextUtils.getProjectName());
    templateVisibilityRepository.save(templateVisibility);
    Set<TemplateVisibilityAttributesEntity> attributes = this.initStaticTemplateVisibilityAttributes(templateVisibility, persistentClass);
    if(!CollectionUtils.isEmpty(attributes)){
      templateVisibilityAttributesRepository.saveAll(attributes);
    }
    templateVisibility.setAttributes(attributes);
    templateVisibility.setProjectName(ApplicationContextUtils.getProjectName());
    return Collections.singleton(templateVisibility);
  }

  /**
   * 初始化表单模版的可见性
   * 第一次执行idPrefix应为空字符
   * @param templateVisibility 可见性实体
   * @param persistentClass 持久化类描述
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initStaticTemplateVisibilityAttributes(TemplateVisibilityEntity templateVisibility, PersistentClass persistentClass){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(64);
    attributes.addAll(initStaticTemplateVisibilityPropertiesAttributes(templateVisibility, persistentClass, ""));
    attributes.addAll(initStaticTemplateVisibilityRelationsAttributes(templateVisibility, persistentClass, "", null));
    attributes.addAll(initStaticTemplateVisibilityItemsAttributes(templateVisibility, persistentClass, ""));
    attributes.addAll(initStaticTemplateVisibilityGroupAttributes(templateVisibility, persistentClass));
    return attributes;
  }

  /**
   * 初始化静态表单模版分组的可见性
   * 初始化静态模版分组下普通属性、关联属性、明细属性的可见性
   * @param templateVisibility 可见性实体
   * @param persistentClass 持久化类描述
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initStaticTemplateVisibilityGroupAttributes(TemplateVisibilityEntity templateVisibility, PersistentClass persistentClass){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(64);
    List<PersistentRelation> relations = persistentClass.getRelations().stream().filter(relation -> PersistentRelation.RelationType.OneToOne.equals(relation.getRelationType())).collect(Collectors.toList());
    for (PersistentRelation relation : relations) {
      String propertyName = relation.getPropertyName();
      for (TemplateLayoutTypeEnum layoutTypeEnum : TemplateLayoutTypeEnum.values()) {
        TemplateVisibilityAttributesEntity attributesEntity = new TemplateVisibilityAttributesEntity();
        attributesEntity.setAttributeName(relation.getPropertyName());
        attributesEntity.setControllerId(StringUtils.join(propertyName, CONTROLLER_ID_SUFFIX));
        attributesEntity.setNullable(relation.getNullable());
        attributesEntity.setLayoutType(layoutTypeEnum.getType());
        if(relation.getCanInsert() || relation.getCanUpdate()) {
          attributesEntity.setVisibilityType(2);
        } else {
          attributesEntity.setVisibilityType(1);
        }
        attributesEntity.setTemplateVisibility(templateVisibility);
        attributes.add(attributesEntity);
      }
      String idPrefix = StringUtils.join(propertyName, CONTROLLER_ID_SPLIT);
      PersistentClass newPersistentClass = this.nebulaStaticPersistentService.findByPersistentClass(relation.getPropertyClass());
      Validate.notNull(newPersistentClass , "在初始化主静态模型时发现NULL关联信息，请联系表单引擎开发团队!!");
      attributes.addAll(initStaticTemplateVisibilityPropertiesAttributes(templateVisibility, newPersistentClass, idPrefix));
      attributes.addAll(initStaticTemplateVisibilityRelationsAttributes(templateVisibility, newPersistentClass, idPrefix, relation));
      attributes.addAll(initStaticTemplateVisibilityItemsAttributes(templateVisibility, newPersistentClass, idPrefix));
    }
    return attributes;
  }

  /**
   * 初始化静态模版的明细可见性
   * 初始化静态模版明细下的普通属性、关联属性
   * @param templateVisibility
   * @param persistentClass
   * @param idPrefix
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initStaticTemplateVisibilityItemsAttributes(TemplateVisibilityEntity templateVisibility, PersistentClass persistentClass, String idPrefix){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(16);
    List<PersistentRelation> relations = persistentClass.getRelations().stream().filter(relation -> PersistentRelation.RelationType.OneToMany.equals(relation.getRelationType())).collect(Collectors.toList());
    for (PersistentRelation relation : relations) {
      String propertyName = relation.getPropertyName();
      for (TemplateLayoutTypeEnum layoutTypeEnum : TemplateLayoutTypeEnum.values()) {
        TemplateVisibilityAttributesEntity attributesEntity = new TemplateVisibilityAttributesEntity();
        attributesEntity.setAttributeName(StringUtils.join(idPrefix, propertyName));
        attributesEntity.setControllerId(StringUtils.join(idPrefix, propertyName, CONTROLLER_ID_SUFFIX));
        attributesEntity.setNullable(relation.getNullable());
        attributesEntity.setLayoutType(layoutTypeEnum.getType());
        if(relation.getCanInsert() || relation.getCanUpdate()) {
          attributesEntity.setVisibilityType(2);
        } else {
          attributesEntity.setVisibilityType(1);
        }
        attributesEntity.setTemplateVisibility(templateVisibility);
        attributes.add(attributesEntity);
      }

      PersistentClass newPersistentClass = this.nebulaStaticPersistentService.findByPersistentClass(relation.getPropertyClass());
      Validate.notNull(newPersistentClass , "在初始化主静态模型时发现NULL关联信息，请联系表单引擎开发团队!!");
      String newIdPrefix = StringUtils.join(idPrefix, propertyName, CONTROLLER_ID_SPLIT);
      attributes.addAll(initStaticTemplateVisibilityPropertiesAttributes(templateVisibility, newPersistentClass, newIdPrefix));
      attributes.addAll(initStaticTemplateVisibilityRelationsAttributes(templateVisibility, newPersistentClass, newIdPrefix, relation));
    }
    return attributes;
  }

  /**
   * 初始化表单模版的关联关系的可见性
   * controllerId前缀不为空时以_结尾
   * 过滤对象与对象间关联的“回溯”字段
   * @param templateVisibility 可见性实体
   * @param persistentClass 持久化类描述
   * @param idPrefix controllerId前缀
   * @param parentRelation 上级的关联信息
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initStaticTemplateVisibilityRelationsAttributes(TemplateVisibilityEntity templateVisibility, PersistentClass persistentClass, String idPrefix, PersistentRelation parentRelation){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(64);
    List<PersistentRelation> persistentRelations = persistentClass.getRelations().stream().filter(relation -> (PersistentRelation.RelationType.ManyToOne.equals(relation.getRelationType())
        || PersistentRelation.RelationType.ManyToMany.equals(relation.getRelationType()))
        && (parentRelation == null || !parentRelation.getPersistentClassName().equals(relation.getPropertyClass()))).collect(Collectors.toList());
    for (PersistentRelation relation : persistentRelations) {
      String propertyName = relation.getPropertyName();
      for (TemplateLayoutTypeEnum layoutTypeEnum : TemplateLayoutTypeEnum.values()) {
        TemplateVisibilityAttributesEntity attributesEntity = new TemplateVisibilityAttributesEntity();
        attributesEntity.setAttributeName(StringUtils.join(idPrefix, propertyName));
        attributesEntity.setControllerId(StringUtils.join(idPrefix, propertyName, CONTROLLER_ID_SUFFIX));
        attributesEntity.setNullable(relation.getNullable());
        attributesEntity.setLayoutType(layoutTypeEnum.getType());
        if(relation.getCanInsert() || relation.getCanUpdate()) {
          attributesEntity.setVisibilityType(2);
        } else {
          attributesEntity.setVisibilityType(1);
        }
        attributesEntity.setTemplateVisibility(templateVisibility);
        attributes.add(attributesEntity);
      }
    }
    return attributes;
  }

  /**
   * 初始化表单模版的基础属性
   * @param templateVisibility 可见性实体
   * @param persistentClass 持久化类描述
   * @param idPrefix controllerId前缀
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initStaticTemplateVisibilityPropertiesAttributes(TemplateVisibilityEntity templateVisibility, PersistentClass persistentClass, String idPrefix){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(16);
    List<PersistentProperty> properties = persistentClass.getProperties();
    for (PersistentProperty property : properties) {
      for (TemplateLayoutTypeEnum layoutTypeEnum : TemplateLayoutTypeEnum.values()) {
        TemplateVisibilityAttributesEntity attribute = new TemplateVisibilityAttributesEntity();
        attribute.setAttributeName(StringUtils.join(idPrefix, property.getPropertyName()));
        attribute.setControllerId(StringUtils.join(idPrefix, property.getPropertyName(), CONTROLLER_ID_SUFFIX));
        attribute.setNullable(property.getNullable());
        attribute.setLayoutType(layoutTypeEnum.getType());
        if(isReadOnly(property, attribute)) {
          attribute.setVisibilityType(1);
        } else {
          attribute.setVisibilityType(2);
        }
        attribute.setTemplateVisibility(templateVisibility);
        attribute.setProjectName(ApplicationContextUtils.getProjectName());
        attributes.add(attribute);
      }
    }
    return attributes;
  }


  /**
   * 判断属性是否是只读的
   * @param property
   * @param attribute
   * @return
   */
  private boolean isReadOnly(PersistentProperty property, TemplateVisibilityAttributesEntity attribute){
    // 说明是主键信息、或者是forminstanceId、或者设置为了不可编辑
    return property.getPrimaryKey()
        || StringUtils.equals(attribute.getAttributeName(), "formInstanceId")
        || (!property.getCanInsert() && !property.getCanUpdate());
  }

  /**
   * 判断属性是否是只读的
   * @param templateProperty
   * @param attribute
   * @return
   */
  private boolean isReadOnly(TemplatePropertyEntity templateProperty, TemplateVisibilityAttributesEntity attribute){
    // 说明是主键信息、或者是forminstanceId、或者设置为了不可编辑
    boolean canInsert = templateProperty.getCanInsert() == null?true:templateProperty.getCanInsert();
    boolean canUpdate = templateProperty.getCanUpdate() == null?true:templateProperty.getCanUpdate();
    return templateProperty.getPrimaryKey()
        || StringUtils.equals(attribute.getAttributeName(), "formInstanceId")
        || (!canInsert && !canUpdate);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService#initDynamicTemplateVisibilities(com.bizunited.platform.kuiper.entity.TemplateEntity)
   */
  @Override
  @Transactional
  public Set<TemplateVisibilityEntity> initDynamicTemplateVisibilities(TemplateEntity template) {
    Validate.notNull(template, "未找到指定的表单模板信息!!");
    Validate.isTrue(StringUtils.equals(template.getType(), "dynamic") , "该初始化过程只能针对动态模板!!");
    
    /*
     * 实际上整个过程和静态模板可见性的初始化类似：
     * 初始化的可见性，名称都是create
     * 1、首先完成一般属性的可见性初始化
     * 2、然后完成关联属性的可见性初始化
     * 其它分组属性的初始化暂不必进行，前端页面会完成，后续工作中补齐即可
     * */
    TemplateVisibilityEntity templateVisibility = new TemplateVisibilityEntity();
    templateVisibility.setCanDelete(false);
    templateVisibility.setCreateTime(new Date());
    templateVisibility.setVisibilityName(VISIBILITY_CREATE);
    templateVisibility.setTemplate(template);
    templateVisibilityRepository.save(templateVisibility);
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(64);
    attributes.addAll(initDynamicTemplateVisibilityPropertyAttributes(template.getProperties(), templateVisibility, ""));
    attributes.addAll(initDynamicTemplateVisibilityRelationAttributes(template.getRelations(), templateVisibility, ""));
    attributes.addAll(initDynamicTemplateVisibilityItemAttributes(template.getItemRelations(), templateVisibility, ""));
    attributes.addAll(initDynamicTemplateVisibilityGroupAttributes(template.getGroupRelations(), templateVisibility));
    if (!CollectionUtils.isEmpty(attributes)) {
      templateVisibilityAttributesRepository.saveAll(attributes);
    }
    templateVisibility.setAttributes(attributes);

    // 不需要建立button的可见性，因为在表单模板初始化的时候，还没有按钮
    templateVisibility.setButtons(null);
    return Collections.singleton(templateVisibility);
  }

  /**
   * 初始化动态模版明细的可见性(OneToMany)
   * 初始化动态模版明细下普通属性和关联属性的可见性
   * @param items 模版明细
   * @param templateVisibility 可见性
   * @param idPrefix controllerId的前缀
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initDynamicTemplateVisibilityItemAttributes(Set<TemplateItemEntity> items, TemplateVisibilityEntity templateVisibility, String idPrefix){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(32);
    if(CollectionUtils.isEmpty(items)){
      return attributes;
    }
    for (TemplateItemEntity item : items) {
      for (TemplateLayoutTypeEnum layoutTypeEnum : TemplateLayoutTypeEnum.values()) {
        TemplateVisibilityAttributesEntity attribute = new TemplateVisibilityAttributesEntity();
        attribute.setAttributeName(StringUtils.join(idPrefix, item.getPropertyName()));
        attribute.setControllerId(StringUtils.join(idPrefix, item.getPropertyName(), CONTROLLER_ID_SUFFIX));
        attribute.setVisibilityType(2);
        attribute.setNullable(true);
        attribute.setLayoutType(layoutTypeEnum.getType());
        attribute.setTemplateVisibility(templateVisibility);
        attributes.add(attribute);
      }
      String newIdPrefix = idPrefix + item.getPropertyName() + CONTROLLER_ID_SPLIT;
      attributes.addAll(initDynamicTemplateVisibilityPropertyAttributes(item.getProperties(), templateVisibility, newIdPrefix));
      attributes.addAll(initDynamicTemplateVisibilityRelationAttributes(item.getRelations(), templateVisibility, newIdPrefix));
    }
    return attributes;
  }

  /**
   * 初始化动态模版分组的可见性(OneToOne)
   * 初始化动态模版分组下普通属性、关联属性、明细属性的可见性
   * @param groups 分组信息
   * @param templateVisibility 可见性
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initDynamicTemplateVisibilityGroupAttributes(Set<TemplateGroupEntity> groups, TemplateVisibilityEntity templateVisibility){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(32);
    if(CollectionUtils.isEmpty(groups)){
      return attributes;
    }
    for (TemplateGroupEntity group : groups) {
      for (TemplateLayoutTypeEnum layoutTypeEnum : TemplateLayoutTypeEnum.values()) {
        TemplateVisibilityAttributesEntity attribute = new TemplateVisibilityAttributesEntity();
        attribute.setAttributeName(group.getPropertyName());
        attribute.setControllerId(StringUtils.join(group.getPropertyName(), CONTROLLER_ID_SUFFIX));
        attribute.setVisibilityType(2);
        attribute.setNullable(true);
        attribute.setLayoutType(layoutTypeEnum.getType());
        attribute.setTemplateVisibility(templateVisibility);
        attributes.add(attribute);
      }
      String newIdPrefix = group.getPropertyName() + CONTROLLER_ID_SPLIT;
      attributes.addAll(initDynamicTemplateVisibilityPropertyAttributes(group.getProperties(), templateVisibility, newIdPrefix));
      attributes.addAll(initDynamicTemplateVisibilityRelationAttributes(group.getRelations(), templateVisibility, newIdPrefix));
      attributes.addAll(initDynamicTemplateVisibilityItemAttributes(group.getItemRelations(), templateVisibility, newIdPrefix));
    }
    return attributes;
  }

  /**
   * 初始化动态模版的关联属性的可见性(ManyToOne和ManyToMany)
   * @param relations 关联属性信息
   * @param templateVisibility 可见性
   * @param idPrefix controllerId的前缀
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initDynamicTemplateVisibilityRelationAttributes(Set<TemplateRelationEntity> relations, TemplateVisibilityEntity templateVisibility, String idPrefix){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(16);
    if(CollectionUtils.isEmpty(relations)){
      return attributes;
    }
    for (TemplateRelationEntity relation : relations) {
      for (TemplateLayoutTypeEnum layoutTypeEnum : TemplateLayoutTypeEnum.values()) {
        TemplateVisibilityAttributesEntity attribute = new TemplateVisibilityAttributesEntity();
        attribute.setAttributeName(StringUtils.join(idPrefix, relation.getPropertyName()));
        attribute.setControllerId(StringUtils.join(idPrefix, relation.getPropertyName(), CONTROLLER_ID_SUFFIX));
        boolean nullable = relation.getNullable() == null?true:relation.getNullable();
        boolean canInsert = relation.getCanInsert() == null?true:relation.getCanInsert();
        boolean canUpdate = relation.getCanUpdate() == null?true:relation.getCanUpdate();
        attribute.setNullable(nullable);
        attribute.setLayoutType(layoutTypeEnum.getType());
        if(canInsert || canUpdate) {
          attribute.setVisibilityType(2);
        } else {
          attribute.setVisibilityType(1);
        }
        attribute.setTemplateVisibility(templateVisibility);
        attributes.add(attribute);
      }
    }
    return attributes;
  }

  /**
   * 初始化动态模版下普通属性的可见性
   * @param templateProperties 模版普通属性信息
   * @param templateVisibility 可见性
   * @param idPrefix controllerId的前缀
   * @return
   */
  private Set<TemplateVisibilityAttributesEntity> initDynamicTemplateVisibilityPropertyAttributes(Set<TemplatePropertyEntity> templateProperties, TemplateVisibilityEntity templateVisibility, String idPrefix){
    Set<TemplateVisibilityAttributesEntity> attributes = new HashSet<>(16);
    for (TemplatePropertyEntity property : templateProperties) {
      for (TemplateLayoutTypeEnum layoutTypeEnum : TemplateLayoutTypeEnum.values()) {
        TemplateVisibilityAttributesEntity attribute = new TemplateVisibilityAttributesEntity();
        attribute.setAttributeName(StringUtils.join(idPrefix, property.getPropertyName()));
        attribute.setControllerId(StringUtils.join(idPrefix, property.getPropertyName(), CONTROLLER_ID_SUFFIX));
        attribute.setNullable(property.getNullable());
        attribute.setLayoutType(layoutTypeEnum.getType());
        if(isReadOnly(property, attribute)) {
          attribute.setVisibilityType(1);
        } else {
          attribute.setVisibilityType(2);
        }
        attribute.setTemplateVisibility(templateVisibility);
        attributes.add(attribute);
      }
    }
    return attributes;
  }

  /**
   * 保存前验证数据
   * @param template
   * @param visibilities
   */
  private void saveValidation(TemplateEntity template, Set<TemplateVisibilityEntity> visibilities){
    Validate.notNull(template, "模板不存在，请检查!!");
    Validate.isTrue(!CollectionUtils.isEmpty(visibilities), "未发现模板控件可见性数据 , 请检查!!");
    Set<String> visibilityNameSet = new HashSet<>();
    for (TemplateVisibilityEntity visibility : visibilities) {
      Validate.notNull(visibility.getCanDelete(), "可见性是否能够删除不能为空,请检查!!");
      Validate.notBlank(visibility.getVisibilityName(), "可见性名称不能为空,请检查!!");
      Validate.notNull(visibility.getCreateTime(), "创建时间不能为空,请检查!!");
      Validate.isTrue(visibilityNameSet.add(visibility.getVisibilityName()), "可见性名称:%s ,重复,请检查!!" , visibility.getVisibilityName());
    }
  }


  /**
   * 以visibilityName作为比较，新增、更新、删除可见性
   * @param template
   * @param visibilities
   * @return
   */
  @Override
  @Transactional
  public Set<TemplateVisibilityEntity> save(TemplateEntity template, Set<TemplateVisibilityEntity> visibilities){
    /**
     * 1、定义可见性结果集
     * 2、验证参数
     * 3、从数据库查询旧的可见性数据，如果为null，则初始化集合，并倾倒如set集合中
     * 4、将提交的数据根据可见性名称作为key存到map中
     * 5、定义集合，用于接收新增、更新、删除的数据
     * 6、调用kuiperToolkitService.collectionDiscrepancy方法获取新增、更新、删除的数据
     * 7、定义当前时间变量
     * 8、处理需要删除的可见性
     *   1、验证可见性是否可以删除
     *   2、删除可见性的按钮
     *   3、删除可见性的字段属性
     *   4、删除可见性
     * 9、处理需要添加的可见性
     *   1、设置可见性对象的属性值
     *   2、保存可见性
     *   3、保存可见性的属性
     *   4、保存可见性的按钮
     *   5、将可见性添加到结果集中
     * 10、处理需要更新的可见性
     *   1、设置可见性对象的属性值
     *   2、保存可见性
     *   3、保存可见性的属性
     *   4、保存可见性的按钮
     *   5、将可见性添加到结果集中
     * 11、返回结果集
     */
    Set<TemplateVisibilityEntity> savedVisibilities = new HashSet<>();
    // 验证数据正确性
    this.saveValidation(template, visibilities);
    // 查询旧的可见性数据
    Set<TemplateVisibilityEntity> oldVisibilityList = templateVisibilityRepository.findByTemplateId(template.getId());
    if(oldVisibilityList == null) oldVisibilityList = new HashSet<>();
    // 将旧的可见性数据倾倒到集合中
    Set<TemplateVisibilityEntity> oldVisibilities = new HashSet<>(oldVisibilityList);
    // 将提交的数据根据可见性名称作为key存到map中
    Map<String, TemplateVisibilityEntity> visibilitiesMap = visibilities.stream().collect(Collectors.toMap(TemplateVisibilityEntity::getVisibilityName, visibility -> visibility, (a, b) -> b, () -> new HashMap<>(16)));
    // 定义集合接收新增、更新、删除的可见性
    Set<TemplateVisibilityEntity> addVisibilities = new HashSet<>();
    Set<TemplateVisibilityEntity> updateVisibilities = new HashSet<>();
    Set<TemplateVisibilityEntity> deleteVisibilities = new HashSet<>();
    // 调用工具获取新增、更新、删除的可见性
    kuiperToolkitService.collectionDiscrepancy(visibilities, oldVisibilities, TemplateVisibilityEntity::getVisibilityName, deleteVisibilities, updateVisibilities, addVisibilities);
    Date date = new Date();
    // 处理需要删除的可见性
    for (TemplateVisibilityEntity visibility : deleteVisibilities) {
      Validate.isTrue(visibility.getCanDelete(), "可见性【%s】不能删除", visibility.getVisibilityName());
      int count = instanceActivityRepository.countByVisibilityId(visibility.getId());
      Validate.isTrue(count == 0, "可见性【%s】已产生活动信息，不运行进行删除操作，请升级模板进行变更！！", visibility.getVisibilityName());
      // 删除可见性的按钮
      templateVisibilityButtonService.deleteByVisibilityId(visibility.getId());
      // 删除可见性的字段属性
      templateVisibilityAttributeService.deleteByVisibilityId(visibility.getId());
      templateVisibilityRepository.delete(visibility);
    }
    // 处理需要添加的可见性
    for (TemplateVisibilityEntity visibility : addVisibilities) {
      visibility.setId(null);
      visibility.setTemplate(template);
      visibility.setCreateTime(date);
      templateVisibilityRepository.save(visibility);
      // 保存可见性的属性
      Set<TemplateVisibilityAttributesEntity> attributes = templateVisibilityAttributeService.save(visibility, visibility.getAttributes());
      // 保存可见性的按钮
      Set<TemplateVisibilityButtonsEntity> buttons = templateVisibilityButtonService.save(visibility, visibility.getButtons());
      visibility.setAttributes(attributes);
      visibility.setButtons(buttons);
      savedVisibilities.add(visibility);
    }
    // 处理需要更新的可见性
    for (TemplateVisibilityEntity visibility : updateVisibilities) {
      TemplateVisibilityEntity newVisibility = visibilitiesMap.get(visibility.getVisibilityName());
      visibility.setCanDelete(newVisibility.getCanDelete());
      templateVisibilityRepository.save(visibility);
      // 保存可见性的属性
      Set<TemplateVisibilityAttributesEntity> attributes = templateVisibilityAttributeService.save(visibility, newVisibility.getAttributes());
      // 保存可见性的按钮
      Set<TemplateVisibilityButtonsEntity> buttons = templateVisibilityButtonService.save(visibility, newVisibility.getButtons());
      visibility.setAttributes(attributes);
      visibility.setButtons(buttons);
      savedVisibilities.add(visibility);
    }
    return savedVisibilities;
  }


  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService#upgrade(java.lang.String, java.util.Set)
   */
  @Override
  @Transactional
  public Set<TemplateVisibilityEntity> upgrade(
      String templateId, Set<TemplateVisibilityEntity> templateVisibilities) {
    Validate.notBlank(templateId, "升级时,原始指定的templateId必须传入");
    TemplateEntity template = this.templateService.findById(templateId);
    Validate.notNull(template, "未找到指定的模板信息，请检查!!");
    if (templateVisibilities == null || templateVisibilities.isEmpty()) {
      return Sets.newHashSet();
    }

    Set<TemplateVisibilityEntity> newTemplateVisibilities = new LinkedHashSet<>();
    for (TemplateVisibilityEntity templateVisibility : templateVisibilities) {
      TemplateVisibilityEntity newTemplateVisibility = new TemplateVisibilityEntity();
      newTemplateVisibility.setId(null);
      newTemplateVisibility.setCanDelete(templateVisibility.getCanDelete());
      newTemplateVisibility.setCreateTime(new Date());
      newTemplateVisibility.setVisibilityName(templateVisibility.getVisibilityName());
      newTemplateVisibility.setTemplate(template);
      this.templateVisibilityRepository.save(newTemplateVisibility);

      // 接着处理可见性中的元素信息
      Set<TemplateVisibilityAttributesEntity> visibilityAttributes = templateVisibility.getAttributes();
      if (visibilityAttributes != null && !visibilityAttributes.isEmpty()) {
        Set<TemplateVisibilityAttributesEntity> newVisibilityAttributes = new LinkedHashSet<>();
        for (TemplateVisibilityAttributesEntity visibilityAttribute : visibilityAttributes) {
          TemplateVisibilityAttributesEntity newVisibilityAttribute = new TemplateVisibilityAttributesEntity();
          newVisibilityAttribute.setId(null);
          newVisibilityAttribute.setAttributeName(visibilityAttribute.getAttributeName());
          newVisibilityAttribute.setControllerId(visibilityAttribute.getControllerId());
          newVisibilityAttribute.setNullable(visibilityAttribute.getNullable());
          newVisibilityAttribute.setVisibilityType(visibilityAttribute.getVisibilityType());
          newVisibilityAttribute.setTemplateVisibility(newTemplateVisibility);
          newVisibilityAttribute.setLayoutType(visibilityAttribute.getLayoutType());
          newVisibilityAttributes.add(newVisibilityAttribute);
        }
        newTemplateVisibility.setAttributes(newVisibilityAttributes);
        this.templateVisibilityAttributesRepository.saveAll(newVisibilityAttributes);
      }

      // 最后处理可见性中的按钮信息
      Set<TemplateVisibilityButtonsEntity> visibilityButtons = templateVisibility.getButtons();
      if (visibilityButtons != null && !visibilityButtons.isEmpty()) {
        Set<TemplateVisibilityButtonsEntity> newVisibilityButtons = new LinkedHashSet<>();
        for (TemplateVisibilityButtonsEntity visibilityButton : visibilityButtons) {
          TemplateVisibilityButtonsEntity newVisibilityButton = new TemplateVisibilityButtonsEntity();
          newVisibilityButton.setId(null);
          newVisibilityButton.setControllerId(visibilityButton.getControllerId());
          newVisibilityButton.setVisible(visibilityButton.getVisible());
          newVisibilityButton.setTemplateVisibility(newTemplateVisibility);
          newVisibilityButton.setLayoutType(visibilityButton.getLayoutType());
          newVisibilityButtons.add(newVisibilityButton);
        }
        newTemplateVisibility.setButtons(newVisibilityButtons);
        this.templateVisibilityButtonRepository.saveAll(newVisibilityButtons);
      }
      newTemplateVisibilities.add(newTemplateVisibility);
    }
    return newTemplateVisibilities;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService#deleteByTemplateId(java.lang.String)
   */
  @Override
  @Transactional
  public void deleteByTemplateId(String templateId) {
    if (StringUtils.isBlank(templateId)) {
      return;
    }

    TemplateEntity currentTemplate = this.templateService.findById(templateId);
    if (currentTemplate == null) {
      return;
    }
    Set<TemplateVisibilityEntity> templateVisibilities = this.templateVisibilityRepository.findByTemplateId(templateId);
    if (templateVisibilities == null || templateVisibilities.isEmpty()) {
      return;
    }

    for (TemplateVisibilityEntity templateVisibilityItem : templateVisibilities) {
      this.templateVisibilityAttributesRepository.deleteByTemplateVisibilityId(templateVisibilityItem.getId());
      this.templateVisibilityButtonRepository.deleteByTemplateVisibilityId(templateVisibilityItem.getId());
      this.templateVisibilityRepository.delete(templateVisibilityItem);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService#findDetailsByTemplateId(java.lang.String)
   */
  @Override
  public Set<TemplateVisibilityEntity> findDetailsByTemplateId(String templateId) {
    if (StringUtils.isBlank(templateId)) {
      return Sets.newHashSet();
    }
    return this.templateVisibilityRepository.findDetailsByTemplateId(templateId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService#findByTemplateIdAndVisibilityName(java.lang.String, java.lang.String)
   */
  @Override
  public TemplateVisibilityEntity findByTemplateIdAndVisibilityName(String templateId, String visibilityName) {
    if (StringUtils.isBlank(templateId) || StringUtils.isBlank(visibilityName)) {
      return null;
    }
    return this.templateVisibilityRepository.findByTemplateIdAndVisibilityName(templateId, visibilityName);
  }

  @Override
  public Set<TemplateVisibilityEntity> findDetailsByTemplateIdAndLayoutType(String templateId, Integer layoutType) {
    if (StringUtils.isBlank(templateId)) {
      return Sets.newHashSet();
    }
    if(layoutType == null) {
      layoutType = TemplateLayoutTypeEnum.PC.getType();
    }
    Set<TemplateVisibilityEntity> visibilities = templateVisibilityRepository.findVisibilityAndAttributesByTemplateIdAndLayoutType(templateId, layoutType);
    Set<TemplateVisibilityEntity> visibilitiesAndButtons = templateVisibilityRepository.findVisibilityAndButtonsByTemplateIdAndLayoutType(templateId, layoutType);
    this.merge(visibilities, visibilitiesAndButtons);
    return visibilities;
  }

  /**
   * 将集合visibilitiesAndButtons的数据合并到集合visibilities中
   * @param visibilities
   * @param visibilitiesAndButtons
   * @return
   */
  private Set<TemplateVisibilityEntity> merge(Set<TemplateVisibilityEntity> visibilities, Set<TemplateVisibilityEntity> visibilitiesAndButtons) {
    if(CollectionUtils.isEmpty(visibilities) || CollectionUtils.isEmpty(visibilitiesAndButtons)) {
      return visibilities;
    }
    Map<String, TemplateVisibilityEntity> tmpMap = visibilitiesAndButtons.stream().collect(Collectors.toMap(UuidEntity::getId, visibility -> visibility, (a, b) -> b, () -> new HashMap<>(8)));
    for (TemplateVisibilityEntity visibility : visibilities) {
      TemplateVisibilityEntity entity = tmpMap.get(visibility.getId());
      if(entity != null) {
        visibility.setButtons(entity.getButtons());
      }
    }
    return visibilities;
  }

  @Override
  public TemplateVisibilityEntity findDetailsByTemplateIdAndVisibilityNameAndLayoutType(String templateId, String visibilityName, Integer layoutType) {
    TemplateVisibilityEntity visibility = this.findByTemplateIdAndVisibilityName(templateId, visibilityName);
    this.loadDetails(visibility, layoutType);
    return visibility;
  }

  /**
   * 加载明细
   * @param visibility
   * @param layoutType
   * @return
   */
  private TemplateVisibilityEntity loadDetails(TemplateVisibilityEntity visibility, Integer layoutType) {
    if(visibility == null) {
      return visibility;
    }
    if(layoutType == null) {
      layoutType = TemplateLayoutTypeEnum.PC.getType();
    }
    Set<TemplateVisibilityAttributesEntity> attributes = templateVisibilityAttributeService.findByVisibilityIdAndLayoutType(visibility.getId(), layoutType);
    Set<TemplateVisibilityButtonsEntity> buttons = templateVisibilityButtonService.findByVisibilityIdAndLayoutType(visibility.getId(), layoutType);
    visibility.setAttributes(attributes);
    visibility.setButtons(buttons);
    return visibility;
  }

  @Override
  public Set<TemplateVisibilityEntity> findByTemplateCodeAndVersion(String templateCode, String version) {
    if(StringUtils.isBlank(templateCode)) {
      return Sets.newHashSet();
    }
    if(StringUtils.isNotBlank(version)) {
      return templateVisibilityRepository.findByTemplateCodeAndVersion(templateCode, version);
    } else {
      return templateVisibilityRepository.findByTemplateCodeAndDefaultVersion(templateCode);
    }
  }
}
