package com.bizunited.platform.kuiper.starter.common.page;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.Nullable;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

public class PageResult<T> extends Chunk<T> implements Page<T>{
  private static final long serialVersionUID = 8677559092943666L;
  private final long total;
  private Map<String,Object> extData;

  public PageResult(List<T> content, Pageable pageable, long total) {
    super(content, pageable);
    this.total = pageable.toOptional().filter(e -> !content.isEmpty())
            .filter(it -> it.getOffset() + (long)it.getPageSize() > total)
            .map(it -> it.getOffset() + (long)content.size())
            .orElse(total);
  }

  public PageResult(List<T> content, Pageable pageable, long total, Map<String,Object> extData) {
    super(content, pageable);
    this.extData = extData;
    this.total = pageable.toOptional().filter(e -> !content.isEmpty())
            .filter(it -> it.getOffset() + (long)it.getPageSize() > total)
            .map(it -> it.getOffset() + (long)content.size())
            .orElse(total);
  }

  public PageResult(List<T> content) {
    this(content, Pageable.unpaged(), null == content ? 0L : (long)content.size());
  }

  public int getTotalPages() {
    return this.getSize() == 0 ? 1 : (int)Math.ceil((double)this.total / (double)this.getSize());
  }

  public long getTotalElements() {
    return this.total;
  }

  public boolean hasNext() {
    return this.getNumber() + 1 < this.getTotalPages();
  }

  public boolean isLast() {
    return !this.hasNext();
  }

  public Map<String, Object> getExtData() {
    return extData;
  }

  public <U> Page<U> map(Function<? super T, ? extends U> converter) {
    return new PageImpl<>(getConvertedContent(converter), getPageable(), total);
  }

  public String toString() {
    String contentType = "UNKNOWN";
    List<T> content = this.getContent();
    if (content.size() > 0) {
      contentType = content.get(0).getClass().getName();
    }

    return String.format("Page %s of %d containing %s instances", this.getNumber() + 1, this.getTotalPages(), contentType);
  }

  public boolean equals(@Nullable Object obj) {
    if (this == obj) {
      return true;
    } else if (!(obj instanceof PageResult)) {
      return false;
    } else {
      PageResult<?> that = (PageResult<?>)obj;
      return this.total == that.total && super.equals(obj);
    }
  }

  public int hashCode() {
    int result = 17;
    result += 31 * (int) (total ^ total >>> 32);
    result += 31 * super.hashCode();
    return result;
  }
}
