package com.bizunited.platform.kuiper.starter.controller;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.kuiper.entity.InstanceActivityEntity;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;
import java.util.Set;

@RestController
@RequestMapping("/v1/kuiper/activities")
public class InstanceActivityController extends BaseController {
  
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(InstanceActivityController.class);
  
  @Autowired
  private InstanceActivityService instanceActivityService;
  @Autowired
  private RedisMutexService redisMutexService;
  private static final String PREVIOUS_ACTIVITY = "previousActivity";
  
  @ApiOperation(value = "在指定的表单实例下创建一个新的活动信息")
  @PostMapping("")
  @CacheEvict(cacheNames={"instance","activity"} , allEntries=true)
  public ResponseModel create(@ApiParam(name = "activity", value = "新创建的表单实例活动对象") @RequestBody InstanceActivityEntity activity) {
    Validate.notNull(activity , "错误的活动参数信息");
    InstanceEntity currentInstance = activity.getInstance();
    Validate.notNull(currentInstance , "创建活动时，必须有相关的实例信息，请检查!!");
    String instanceId = currentInstance.getId();
    Validate.notBlank(instanceId , "创建活动时，未发现相关实例的数据编号，请检查!!");
    
    try {
      Principal principal = this.getPrincipal();
      String account = principal.getName();
      // 这是因为同一个实例下不能同一时间创建活动实例
      redisMutexService.lock(instanceId);
      InstanceActivityEntity currentActivity = this.instanceActivityService.create(activity, account);
      return this.buildHttpResultW(currentActivity);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    } finally {
      redisMutexService.unlock(instanceId);
    }
  }
  
  @ApiOperation(value = "检查并视情况创建一个指定表单实例下的活动，如果taskCode指定的信息已经存在，则直接返回活动基本信息；如果不存在则根据instanceId和visibilityName进行创建"
      + " 入参为json接口，需要传递的参数包括taskCode：指定的外部业务编号（可能在本系统存在也可能不在本系统存在）；instanceId：当前活动对应的实例编号；visibilityName：指定的可见性名称（在instanceId对应的模板中visibilityName是唯一的）,例如：update、view等" 
      , notes="返回可能存在也可能是依据instanceId、visibilityName新建的活动编号。")
  @PostMapping("/checkCreate")
  @CacheEvict(cacheNames={"instance","activity"} , allEntries=true)
  public ResponseModel create(@RequestBody JSONObject paramJson) {
    Validate.notNull(paramJson , "必须进行参数传递，请查看api文档");
    String taskCode = paramJson.getString("taskCode");
    String instanceId = paramJson.getString("instanceId");
    String visibilityName = paramJson.getString("visibilityName");
    Validate.notBlank(instanceId , "在创建活动时，实例编号名必须传入!!");
    try {
      Principal principal = this.getPrincipal();
      String account = principal.getName();
      // 这是因为同一个实例下不能同一时间创建活动实例
      redisMutexService.lock(instanceId);
      InstanceActivityEntity currentActivity = this.instanceActivityService.create(taskCode, instanceId, visibilityName , account);
      return this.buildHttpResultW(currentActivity);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    } finally {
      redisMutexService.unlock(instanceId);
    }
  }
  
  @ApiOperation(value = "按照活动编号，查询指定的活动信息详情——包括关联信息")
  @GetMapping("/findDetailsById")
  @Cacheable(cacheNames="activity" , key="'findDetailsById_' + #id" , condition="#id != ''"  , sync=true)
  public ResponseModel findDetailsById(@ApiParam(name = "id", value = "指定的活动编号信息") String id) {
    try {
      InstanceActivityEntity currentActivity = this.instanceActivityService.findDetailsById(id);
      return this.buildHttpResultW(currentActivity, "template" , "creator" , "modifyer" , PREVIOUS_ACTIVITY , "instance" , "templateVisibility");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "按照指定的表单实例编号，查询其下已经发生的活动，并按照活动创建实例反序排列")
  @GetMapping("/findDetailsByInstanceId")
  @Cacheable(cacheNames="activity" , key="'findDetailsByInstanceId' + #instanceId" , condition="#instanceId != ''" , sync=true)
  public ResponseModel findDetailsByInstanceId(@ApiParam(name = "instanceId", value = "指定的表单实例编号") String instanceId) {
    try {
      Set<InstanceActivityEntity> instanceActivities = this.instanceActivityService.findDetailsByInstanceId(instanceId);
      return this.buildHttpResultW(instanceActivities, "template" , "creator" , "modifyer" , PREVIOUS_ACTIVITY , "instance" , "templateVisibility" , PREVIOUS_ACTIVITY);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
