package com.bizunited.platform.kuiper.starter.controller;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.ListTemplateEntity;
import com.bizunited.platform.kuiper.starter.service.ListTemplateService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 列表模板管理业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @author yinwenjie
 */
@RestController
@RequestMapping("/v1/kuiper/listTemplates")
public class ListTemplateController extends BaseController { 
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ListTemplateController.class);
  @Autowired
  private ListTemplateService listTemplateEntityService;
  private static final String CREATOR = "creator";

  /**
   * 初始化一个新的列表模板模型对象，初始化的列表模板将只包括基本信息，不包括列表模型JSON/XML形式的内容描述
   * */
  @ApiOperation(value = "初始化一个新的列表模板模型对象，初始化的列表模板将只包括基本信息，不包括列表模型JSON/XML形式的内容描述")
  @PostMapping(value="")
  public ResponseModel init(@RequestBody @ApiParam(name="listTemplate" , value="相关的创建过程，http接口。请注意该创建过程除了可以创建listTemplateEntity中的基本信息以外，还可以对listTemplateEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的listTemplateEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") ListTemplateEntity listTemplateEntity) {
    try {
      Principal principal = this.getPrincipal();
      ListTemplateEntity current = this.listTemplateEntityService.init(listTemplateEntity , principal);
      return this.buildHttpResultW(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的listTemplateEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   * */
  @ApiOperation(value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的listTemplateEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PatchMapping(value="")
  public ResponseModel update(@RequestBody @ApiParam(name="listTemplateEntity" , value="相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的listTemplateEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") ListTemplateEntity listTemplateEntity) {
    try {
      ListTemplateEntity current = this.listTemplateEntityService.update(listTemplateEntity);
      return this.buildHttpResultW(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "按照列表模板的数据层编号，更新这个列表模板的布局内容（JSON描述结构）")
  @PatchMapping("/updateContentById")
  public ResponseModel updateContentById(@RequestParam("id") @ApiParam("指定的列表模板编号") String id , @RequestBody @ApiParam("需要在文件中保存的JSON形式") JSONObject content) {
    try {
      ListTemplateEntity current = this.listTemplateEntityService.updateContentById(id, content);
      return this.buildHttpResultW(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "按照指定的列表模板数据层编号，查询这个列表模板的布局内容（JSON描述结构）")
  @GetMapping("/findConentById")
  public ResponseModel findContentById(@RequestParam("id") @ApiParam("指定的列表模板编号") String id) {
    try {
      JSONObject data = this.listTemplateEntityService.findContentById(id);
      return this.buildHttpResultW(data);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  /**
   * 按照主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   * @param id 主键
   */
  @ApiOperation(value = "按照主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @GetMapping("/findDetailsById")
  public ResponseModel findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try { 
      ListTemplateEntity result = this.listTemplateEntityService.findDetailsById(id); 
      return this.buildHttpResultW(result, CREATOR);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    } 
  }
  /**
   * 按照列表模板的编码进行查询，并按照创建时间顺序进行反序排列
   * @param code 指定的编码
   */
  @ApiOperation(value = "按照列表模板的编码进行查询，并按照创建时间顺序进行反序排列")
  @GetMapping("/findByCode")
  public ResponseModel findByCode(@RequestParam("code") @ApiParam("指定的编码") String code) {
    try { 
      Set<ListTemplateEntity> result = this.listTemplateEntityService.findByCode(code); 
      return this.buildHttpResultW(result, CREATOR);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    } 
  }
  /**
   * 这里是针对列表模板的数据库层分页查询定义
   * @param pageable 分页信息
   */
  @ApiOperation(value = "这里是针对列表模板的数据库层分页查询定义")
  @GetMapping("/findByConditions")
  public ResponseModel findByConditions(@ApiParam(name = "code", value = "可能的列表模板编号") String code,
                                        @ApiParam(name = "cversion", value = "模板指定的域信息") String cversion,
                                        @ApiParam(name = "name", value = "模板指定的域信息") String name,
                                        @ApiParam(name = "defaultVersion", value = "默认模板") Boolean defaultVersion,
                                        @ApiParam(name = "modify_start" , value="修改时间范围开始点，精确到yyyy-MM-dd") String modifyStart,
                                        @ApiParam(name = "modify_end" , value="修改时间查询范围结束点，精确到yyyy-MM-dd") String modifyEnd,
                                        @PageableDefault(value = 50) @ApiParam(name = "pageable", value = "分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50") Pageable pageable) {
    try {
      Map<String, Object> conditions = new HashMap<>();
      if (StringUtils.isNotBlank(code)) {
        conditions.put("code", code);
      }
      if (StringUtils.isNotBlank(cversion)) {
        conditions.put("domain", cversion);
      }
      if (name != null) {
        conditions.put("name", name);
      }
      conditions.put("modifyStart", modifyStart);
      conditions.put("modifyEnd", modifyEnd);
      conditions.put("defaultVersion", defaultVersion);
      Page<ListTemplateEntity> page = this.listTemplateEntityService.findByConditions(pageable, conditions);
      return this.buildHttpResultW(page, CREATOR);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照列表模板的编码和版本号进行查询,查询这个列表模板的布局内容（JSON描述结构）
   * @param code 列表编码
   * @param cversion 列表编码的版本号
   * @return
   */
  @ApiOperation(value = "按照列表模板的编码和版本号进行查询,查询这个列表模板的布局内容（JSON描述结构）")
  @GetMapping("/findContentByCodeAndCversion")
  public ResponseModel findContentByCodeAndCversion(@RequestParam(value = "code") @ApiParam("必传参数指定的列表模板编码") String code,
                                         @RequestParam(value = "cversion") @ApiParam("必传参数指定的列表编码的版本号") String cversion){
    try {
      JSONObject data = this.listTemplateEntityService.findContentByCodeAndCversion(code, cversion);
      return this.buildHttpResultW(data, CREATOR);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照列表模板的编码和版本号进行更新,更新这个列表模板的布局内容（JSON描述结构）
   * @param code 列表编码
   * @param cversion 列表编码的版本号
   * @param content 需要在文件中保存的JSON形式
   * @return
   */
  @ApiOperation(value = "按照列表模板的编码，更新这个列表模板的布局内容（JSON描述结构）")
  @PatchMapping("/updateContentByCodeAndCversion")
  public ResponseModel updateContentByCodeAndCversion(@RequestParam(value = "code") @ApiParam("必传参数指定的列表模板编码") String code,
                                           @RequestParam(value = "cversion") @ApiParam("必传参数指定的列表编码的版本号") String cversion,
                                           @RequestBody @ApiParam("需要在文件中保存的JSON形式") JSONObject content){
    try {
      ListTemplateEntity result = this.listTemplateEntityService.updateContentByCodeAndCversion(code, cversion, content);
      return this.buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照列表模板的编码查询默认版本,查询这个列表模板的布局内容（JSON描述结构）
   * @param code 列表编码
   * @return
   */
  @ApiOperation(value = "按照列表模板的编码查询默认版本")
  @GetMapping("/findDefaultContentByCode")
  public ResponseModel findDefaultContentByCode(@RequestParam(value = "code") @ApiParam("必传参数指定的列表模板编码") String code){
    try {
      JSONObject result = this.listTemplateEntityService.findDefaultContentByCode(code);
      return this.buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 根据表编码和列表编码的版本号指定当前版本为默认版本
   * @param code 列表编码
   * @param cversion 列表编码的版本号
   * @return
   */
  @ApiOperation(value = "该方法设定指定的列表模板版本为这个表单code下唯一默认的表单模板。" ,
          notes="该方法设定指定的列表模板版本为这个模板code下唯一默认的列表模板。在列表模板的信息中，相同code下只有一个指定的版本可以是默认的版本号（也就是defaultVersion属性为true）")
  @PatchMapping("/updateDefaultVersion")
  public ResponseModel updateDefaultVersion(@RequestParam(value = "code") @ApiParam("必传参数指定的列表模板编码") String code,
                                            @RequestParam(value = "cversion") @ApiParam("必传参数指定的列表编码的版本号") String cversion) {
    try {
      this.listTemplateEntityService.updateDefaultVersion(code, cversion);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照列表模板的编码查询默认版本
   * @param code 列表编码
   * @return
   */
  @ApiOperation(value = "按照列表模板的编码查询默认版本")
  @GetMapping("/findDefaultByCode")
  public ResponseModel findDefaultByCode(@RequestParam(value = "code") @ApiParam("必传参数指定的列表模板编码") String code){
    try {
      ListTemplateEntity result = this.listTemplateEntityService.findDefaultByCode(code);
      return this.buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

}