package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.PageEntity;
import com.bizunited.platform.kuiper.starter.service.PageService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

/**
 * 页面流页面数据功能业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @Author: TanBoQiuYun
 * @Date: 2019/12/31 17:23
 */
@Api(value = "PageController", tags = "页面流页面数据功能业务模型的MVC Controller层实现")
@RestController
@RequestMapping("/v1/kuiper/pages")
public class PageController extends BaseController {
  /** 日志 */
  private static final Logger LOGGER = LoggerFactory.getLogger(PageController.class);
  @Autowired
  private PageService pageService;

  /**
   * 页面输入的属性白名单
   */
  private static final String[] PAGE_PROPERTIES_W = {"pageFlow", "events"};

  /**
   * 按照页面流编码查询页面列表详情
   * @param pageFlowCode
   * @return
   */
  @GetMapping("findDetailsByPageFlowCode")
  @ApiOperation(value = "按照页面流编码进查询页面列表详情")
  public ResponseModel findDetailsByPageFlowCode(@ApiParam(name = "pageFlowCode", value = "指定的页面编码", required = true) @RequestParam("pageFlowCode") String pageFlowCode){
    try {
      Set<PageEntity> result = this.pageService.findDetailsByPageFlowCode(pageFlowCode);
      return buildHttpResultW(result, PAGE_PROPERTIES_W);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 按照数页面ID进行查询明细查询，查询的明细包含页面下的所有关联信息
   * @param id
   * @return
   */
  @GetMapping("findDetailsById")
  @ApiOperation(value = "按照数页面ID进行查询明细查询，查询的明细包含页面下的所有关联信息")
  public ResponseModel findDetailsById(@ApiParam(value = "页面ID", required = true) @RequestParam("id") String id){
    try {
      PageEntity result = this.pageService.findDetailsById(id);
      return buildHttpResultW(result, PAGE_PROPERTIES_W);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }


  /**
   * 根据页面流code和页面code查询详情
   * @param code
   * @param pageFlowCode
   * @return
   */
  @GetMapping("findDetailsByCodeAndPageFlowCode")
  @ApiOperation(value = "根据页面流code和页面code查询详情，查询的明细包含页面下的所有关联信息")
  public ResponseModel findDetailsByCodeAndPageFlowCode(@ApiParam(name = "code", value = "指定的页面编码", required = true) @RequestParam("code") String code,
                                                        @ApiParam(name = "pageFlowCode", value = "指定的页面流编码", required = true) @RequestParam("pageFlowCode") String pageFlowCode){
    try {
      PageEntity result = this.pageService.findDetailsByCodeAndPageFlowCode(code, pageFlowCode);
      return buildHttpResultW(result, PAGE_PROPERTIES_W);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 按照数页面流编码进行查询，查询的明细不包含关联信息以及文件内容
   * @param pageFlowCode
   * @return
   */
  @GetMapping("findByPageFlowCode")
  @ApiOperation(value = "按照数页面流编码进行查询，查询的明细不包含关联信息以及文件内容")
  public ResponseModel findByPageFlowCode(@ApiParam(value = "页面流编码", required = true) @RequestParam("pageFlowCode") String pageFlowCode){
    try {
      Set<PageEntity> result = this.pageService.findByPageFlowCode(pageFlowCode);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

}
