package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.PageEventEntity;
import com.bizunited.platform.kuiper.starter.service.PageEventService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

/**
 * 页面事件控制类MVC Controller层实现，基于HTTP Restful风格
 * @Author: Paul Chan
 * @Date: 2020-01-02 11:22
 */
@RestController
@Api(tags = " 页面事件控制类")
@RequestMapping("/v1/kuiper/page/events")
public class PageEventController extends BaseController {
  private static final Logger LOGGER = LoggerFactory.getLogger(PageEventController.class);

  @Autowired
  private PageEventService pageEventService;

  /**
   * 根据页面ID查询
   * @param pageId
   * @return
   */
  @GetMapping("findByPage")
  @ApiOperation("根据页面ID查询")
  public ResponseModel findByPage(@RequestParam @ApiParam(value = "页面ID", required = true) String pageId) {
    try {
      Set<PageEventEntity> events = pageEventService.findByPageId(pageId);
      return buildHttpResultW(events);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据页面ID查询详情
   * @param pageId
   * @return
   */
  @GetMapping("findDetailsByPage")
  @ApiOperation("根据页面ID查询详情")
  public ResponseModel findDetailsByPage(@RequestParam @ApiParam(value = "页面ID", required = true) String pageId) {
    try {
      Set<PageEventEntity> events = pageEventService.findDetailsByPageId(pageId);
      return buildHttpResultW(events);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据ID查询详情
   * @param pageId
   * @return
   */
  @GetMapping("findDetailById")
  @ApiOperation("根据ID查询详情")
  public ResponseModel findDetailById(@RequestParam @ApiParam(value = "主键ID", required = true) String id) {
    try {
      PageEventEntity event = pageEventService.findDetailById(id);
      return buildHttpResultW(event);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据页面流编码和页面编码和事件ID查询
   * @param pageFlowCode
   * @param pageCode
   * @param eventId
   * @return
   */
  @GetMapping("findDetailByPageFlowCodeAndPageCodeAndEventId")
  @ApiOperation("根据页面流编码和页面编码和事件ID查询")
  public ResponseModel findDetailByPageFlowCodeAndPageCodeAndEventId(@RequestParam @ApiParam(value = "页面流编码", required = true) String pageFlowCode,
                                                                     @RequestParam @ApiParam(value = "页面编码", required = true) String pageCode,
                                                                     @RequestParam @ApiParam(value = "事件ID", required = true) String eventId) {
    try {
      PageEventEntity event = pageEventService.findDetailByPageFlowCodeAndPageCodeAndEventId(pageFlowCode, pageCode, eventId);
      return buildHttpResultW(event, "page", "page.pageFlow");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

}
