package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.PageFlowEntity;
import com.bizunited.platform.kuiper.starter.service.PageFlowService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;

/**
 * 页面流功能业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @Author: TanBoQiuYun
 * @Date: 2019/12/31 15:09
 */
@Api(value = "PageFlowController", tags = "页面流功能业务模型的MVC Controller层实现")
@RestController
@RequestMapping("/v1/kuiper/pageFlows")
public class PageFlowController extends BaseController {
  /** 日志 */
  private static final Logger LOGGER = LoggerFactory.getLogger(PageFlowController.class);

  @Autowired
  private PageFlowService pageFlowService;

  /**
   * 页面流输入的属性白名单
   */
  private static final String[] PAGE_FLOW_PROPERTIES_W = {"pages", "createUser", "modifyUser", "pages.events"};
  /**
   * 创建页面流
   * @param pageFlow
   * @return
   */
  @PostMapping("")
  @ApiOperation(value = "创建页面流", notes = "创建页面流")
  public ResponseModel create(@RequestBody PageFlowEntity pageFlow){
    try {
      Principal principal = this.getPrincipal();
      PageFlowEntity result = this.pageFlowService.create(pageFlow, principal);
      return buildHttpResultW(result, PAGE_FLOW_PROPERTIES_W);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 更新页面流
   * @param pageFlow
   * @return
   */
  @PatchMapping("")
  @ApiOperation(value = "更新页面流", notes = "更新页面流")
  public ResponseModel update(@RequestBody PageFlowEntity pageFlow){
    try {
      Principal principal = this.getPrincipal();
      PageFlowEntity result = this.pageFlowService.update(pageFlow, principal);
      return buildHttpResultW(result, PAGE_FLOW_PROPERTIES_W);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据ID查询页面流信息（注意只有基本信息，不包括任何的关联信息）
   * @param id
   * @return
   */
  @GetMapping("findById")
  @ApiOperation(value = "根据ID查询页面流信息（注意只有基本信息，不包括任何的关联信息）",
          notes = "根据ID查询页面流信息（注意只有基本信息，不包括任何的关联信息）")
  public ResponseModel findById(@ApiParam(name = "id", value = "页面流ID", required = true) @RequestParam("id")  String id){
    try {
      PageFlowEntity result = this.pageFlowService.findById(id);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 按照数页面流ID进行查询明细查询，查询的明细包含页面流下的所有关联信息
   * @param id
   * @return
   */
  @GetMapping("findDetailsById")
  @ApiOperation(value = "按照数页面流ID进行查询明细查询，查询的明细包含页面流下的所有关联信息")
  public ResponseModel findDetailsById(@ApiParam(value = "页面流ID", required = true) @RequestParam("id") String id){
    try {
      PageFlowEntity result = this.pageFlowService.findDetailsById(id);
      return buildHttpResultW(result, PAGE_FLOW_PROPERTIES_W);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据编码查询页面流信息, 不包含关联信息
   * @param code
   * @return
   */
  @GetMapping("findByCode")
  @ApiOperation(value = "根据编码查询页面流信息, 不包含关联信息", notes = "根据编码查询页面流信息, 不包含关联信息")
  public ResponseModel findByCode(@RequestParam("code") @ApiParam(value = "指定的页面流编码", required = true) String code){
    try {
      PageFlowEntity result = this.pageFlowService.findByCode(code);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }


  /**
   * 按照数页面流编码进行查询明细查询，查询的明细包含页面流下的所有关联信息
   * @param code
   * @return
   */
  @GetMapping("findDetailsByCode")
  @ApiOperation(value = "按照数页面流编码进行查询明细查询，查询的明细包含页面流下的所有关联信息")
  public ResponseModel findDetailsByCode(@ApiParam(name = "code", value = "指定的页面流编码", required = true) @RequestParam("code") String code){
    try {
      PageFlowEntity result = this.pageFlowService.findDetailsByCode(code);
      return buildHttpResultW(result, PAGE_FLOW_PROPERTIES_W);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 分页查询-按条件搜索，页面流名称和页面流编码2个条件，支持模糊查询
   * @param pageable
   * @param name
   * @param code
   * @return
   */
  @GetMapping("findByConditions")
  @ApiOperation(value = "分页查询,支持模糊查询", notes = "页数参数为page，第一页从0开始，分页大小参数为size（size默认为50）")
  public ResponseModel findByConditions(@PageableDefault(value = 50) Pageable pageable,
                                        @ApiParam(name = "name", value = "页面流名称") @RequestParam(value = "name", required = false) String name,
                                        @ApiParam(name = "code", value = "页面流编码") @RequestParam(value = "code", required = false) String code){
    try {
      PageFlowEntity pageFlow = new PageFlowEntity();
      pageFlow.setName(name);
      pageFlow.setCode(code);
      Page<PageFlowEntity> result = this.pageFlowService.findByConditions(pageFlow, pageable);
      return buildHttpResultW(result, "pages");
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据主键删除页面流以及关联信息
   * 此方法为真删除
   * @param id
   * @return
   */
  @DeleteMapping("/{id}")
  @ApiOperation(value = "根据主键删除页面流以及关联信息")
  public ResponseModel deleteById(@ApiParam(value = "主键", required = true) @PathVariable("id") String id){
    try {
      this.pageFlowService.deleteById(id);
      return this.buildHttpResult();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
