package com.bizunited.platform.kuiper.starter.controller;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

@Api(value = "TemplateController")
@RestController
@RequestMapping("/v1/kuiper/templates")
public class TemplateController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(TemplateController.class);
  @Autowired
  private TemplateService templateService;
  
  @RequestMapping(value = "/findDetailsById", method = RequestMethod.GET)
  public ResponseModel findDetailsById(@ApiParam(name="id" , value="表单模板的数据库编号") String id) {
    try {
      TemplateEntity result = this.templateService.findDetailsById(id);
      return this.buildHttpResultW(result, "properties", "relations","creator",
              "groupRelations","groupRelations.properties","groupRelations.relations","groupRelations.itemRelations",
              "groupRelations.itemRelations.properties","groupRelations.itemRelations.relations",
              "itemRelations","itemRelations.properties","itemRelations.relations");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据模版编码和版本号查询模版详情
   * @param code
   * @param version
   * @return
   */
  @GetMapping("/findDetailsByCodeAndVersion")
  @ApiOperation("根据模版编码和版本号查询模版详情")
  public ResponseModel findDetailsByCodeAndVersion(@RequestParam @ApiParam("模版编码") String code, @RequestParam @ApiParam("版本号") String version) {
    try {
      TemplateEntity template = this.templateService.findDetailsByCodeAndVersion(code, version);
      return this.buildHttpResultW(template, "properties", "relations","creator",
          "groupRelations","groupRelations.properties","groupRelations.relations","groupRelations.itemRelations",
          "groupRelations.itemRelations.properties","groupRelations.itemRelations.relations",
          "itemRelations","itemRelations.properties","itemRelations.relations");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据模版编码和版本号查询模版详情
   * @param code
   * @return
   */
  @GetMapping("/findDetailsByCodeAndDefaultVersion")
  @ApiOperation("根据模版编码查询默认版本号的模版详情")
  public ResponseModel findDetailsByCodeAndDefaultVersion(
          @RequestParam @ApiParam("模版编码") String code) {
    try {
      TemplateEntity template = this.templateService.findDetailsByCodeAndDefaultVersion(code);
      return this.buildHttpResultW(template, "properties", "relations","creator",
          "groupRelations","groupRelations.properties","groupRelations.relations","groupRelations.itemRelations",
          "groupRelations.itemRelations.properties","groupRelations.itemRelations.relations",
          "itemRelations","itemRelations.properties","itemRelations.relations");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "该接口按照表单模板的数据库编号，返回表单模板的基本信息（注意只有基本信息，不包括任何的关联信息），无论是静态模板还是动态模板")
  @RequestMapping(value = "/findById", method = RequestMethod.GET)
  public ResponseModel findById(@ApiParam(name = "id", value = "表单模板的数据库编号")  String id) {
    try {
      TemplateEntity result = this.templateService.findById(id);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "按条件搜索，目前支持表单模板名、业务域名和项目名三个条件，无论是静态模板还是动态模板，不支持模糊查询")
  @RequestMapping(value = "/findByConditions", method = RequestMethod.GET)
  public ResponseModel findByConditions(
      @RequestParam(required = false) @ApiParam("模板名称") String name,
      @RequestParam(required = false) @ApiParam("模板指定的域信息") String domain,
      @RequestParam(required = false) @ApiParam("是否指定的默认版本") Boolean isDefault,
      @RequestParam(required = false) @ApiParam("模板状态。1：正常；0：作废") Integer tstatus,
      @RequestParam(required = false) @ApiParam("表单模板来源类型，目前有三种：static：静态模型;dynamic：动态模型;dsl：dsl模型，其它值无效") String type,
      @RequestParam(required = false) @ApiParam("创建时间范围开始点，精确到秒yyyy-MM-dd HH:mm:ss") @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss") Date createStart,
      @RequestParam(required = false) @ApiParam("创建时间查询范围结束点，精确到秒yyyy-MM-dd HH:mm:ss")  @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss") Date createEnd,
      @RequestParam(required = false) @ApiParam("修改时间范围开始点，精确到yyyy-MM-dd") String modifyStart,
      @RequestParam(required = false) @ApiParam("修改时间查询范围结束点，精确到yyyy-MM-dd") String modifyEnd,
      @RequestParam(required = false) @ApiParam("模板工程名") String projectName,
      @PageableDefault(value = 50) @ApiParam("分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50") Pageable pageable) {
    try {
      Map<String, Object> conditions = new HashMap<>();
      if (StringUtils.isNotBlank(name)) {
        conditions.put("name", name);
      }
      if (StringUtils.isNotBlank(domain)) {
        conditions.put("domain", domain);
      }
      if (projectName != null) {
        conditions.put("projectName", projectName);
      }
      if (isDefault != null) {
        conditions.put("isDefault", isDefault);
      }
      if (tstatus != null) {
        conditions.put("tstatus", tstatus);
      }
      if(type != null) {
        conditions.put("type", type);
      }
      conditions.put("createStart", createStart);
      conditions.put("createEnd", createEnd);
      conditions.put("modifyStart", modifyStart);
      conditions.put("modifyEnd", modifyEnd);
      Page<TemplateEntity> page = this.templateService.findByConditions(conditions, pageable);
      return this.buildHttpResultW(page, "maintainers", "creator");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "按条件搜索所有可用的模板信息，无论是静态模板还是动态模板，支持模板名称模糊查询")
  @RequestMapping(value = "/findAbleByConditions", method = RequestMethod.GET)
  public ResponseModel findAbleByConditions(
      @RequestParam(required = false) @ApiParam(name = "name", value = "模板名称") String name,
      @RequestParam(required = false) @ApiParam(name = "cversion", value = "模板版本号") String cversion,
      @RequestParam(required = false) @ApiParam(name = "code", value = "模板code") String code,
      @PageableDefault(value = 50) @ApiParam(name = "pageable", value = "分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50") Pageable pageable) { 
    try {
      Map<String, Object> conditions = new HashMap<>();
      conditions.put("tstatus", 1);
      if (StringUtils.isNotBlank(name)) {
        conditions.put("name", name);
      }
      if (StringUtils.isNotBlank(cversion)) {
        conditions.put("cversion", cversion);
      }
      if (StringUtils.isNotBlank(code)) {
        conditions.put("code", code);
      }
      Page<TemplateEntity> page = this.templateService.findByConditions(conditions, pageable);
      return this.buildHttpResultW(page);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "保存一个新的表单模板，因为表单模板有初始化过程，所以这里的保存功能肯定都是修改功能。首先就是说表单模板对应的模型是不能改变的，所以表单模板保存功能对表单模型无效；另外表单模板保存功能主要就是对数据源、事件、布局、可见性信息进行修改"  
      , notes="另外表单模板的保存功能会自动判断数据源、事件、布局、可见性的信息变化，自行进行这些数据项的新增、修改、删除操作。")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(@RequestBody @ApiParam(name = "template", value = "新的表单模板信息，必须至少包括、数据源、事件、布局、可见性等信息，另外注意，在添加是表单模板的id（主键信息）必须传入") TemplateEntity template) {
    try {
      TemplateEntity result = this.templateService.update(template);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "该方法设定指定的表单模板版本为这个表单code下唯一默认的表单模板。" , 
      notes="该方法设定指定的表单模板版本为这个表单code下唯一默认的表单模板。在表单模板的信息中，相同code下只有一个指定的版本可以是默认的版本号（也就是defaultVersion属性为true）")
  @RequestMapping(value = "/updateDefaultVersion", method = RequestMethod.PATCH)
  public ResponseModel updateDefaultVersion(@ApiParam(name="templateId", value = "指定的版本信息") @RequestParam(name="templateId") String templateId) {
    try {
      this.templateService.updateDefaultVersion(templateId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "更新模板状态信息，如果当前模板被禁用，那么就不能继续使用该模板创建表单实例了" ,  notes="另一个方面是，某一个模板编号下的默认版本不允许被禁用")
  @RequestMapping(value = "/updateStatusById", method = RequestMethod.PATCH)
  public ResponseModel updateStatusById(@ApiParam(name="templateId", value = "指定的模板编号") @RequestParam(name="templateId") String templateId ,
                                        @ApiParam(name="tstatus", value = "状态符1：正常；0：禁用/废除") @RequestParam(name="tstatus") Integer tstatus) {
    try {
      this.templateService.updateStatusById(templateId , tstatus);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据表单ID，获取表单属性的特定JSON结构（备注：此接口针对远端服务源-出参快速设定）
   *
   * @param templateEntityId
   * @return
   */
  @ApiOperation(value = "导入模板")
  @RequestMapping(
      value = "/getTemplateRemote",
      method = {RequestMethod.GET})
  public ResponseModel importTemplate(
      @RequestParam(name = "templateEntityId") @ApiParam("templateEntityId")
          String templateEntityId) {
    try {
      JSONArray templateJsonArray = this.templateService.importTemplate(templateEntityId);
      return this.buildHttpResultW(templateJsonArray);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

}
