package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.TemplateMaintainerEntity;
import com.bizunited.platform.kuiper.starter.service.TemplateMaintainerService;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;
import java.util.List;
import java.util.Set;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_ADMIN_ACCOUNT;


@Api(value = "TemplateMaintainerController")
@RestController
@RequestMapping("/v1/kuiper/templates/maintainer")
public class TemplateMaintainerController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(TemplateMaintainerController.class);

  @Value("${rbac.template.maintainers}")
  private String[] maintainerAccounts;
  @Autowired
  private TemplateMaintainerService templateMaintainerService;

  /**
   * 通过一次授权操作，可以对不同版本号的模板，统一指定维护人员
   *
   * @param code
   * @param maintainerIds
   * @param principal
   * @return
   */
  @ApiOperation(value = "具有授权权限的人，授权用户能维护其他用户创建的模板")
  @PatchMapping("/binding")
  public ResponseModel updateMaintainers(@ApiParam(name = "code", value = "指定的模板code") @RequestParam(name = "code") String code,
                                         @ApiParam(name = "maintainers", value = "维护人员的账号信息(数组)") @RequestBody String[] maintainers) {
    try {
      Principal principal = this.getPrincipal();
      this.templateMaintainerService.binding(code, maintainers, maintainerAccounts, principal);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "获取配置文件中设置的，具有授权变更模板信息的人员")
  @GetMapping("/loadAuthorProps")
  public ResponseModel findAuthorMaintainers() {
    try {
      if (ArrayUtils.isEmpty(maintainerAccounts)) {
        maintainerAccounts = new String[]{DEFAULT_ADMIN_ACCOUNT};
      }
      List<String> maintainerAccountList = Lists.newArrayList(maintainerAccounts);
      return this.buildHttpResultW(maintainerAccountList);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据模版ID查询详情信息
   * @param templateId
   * @return
   */
  @ApiOperation("根据模版ID查询详情信息")
  @GetMapping("findDetailsByTemplateId")
  public ResponseModel findDetailsByTemplateId(@RequestParam @ApiParam("模版ID") String templateId) {
    try {
      Set<TemplateMaintainerEntity> maintainer = templateMaintainerService.findDetailsByTemplateId(templateId);
      return this.buildHttpResultW(maintainer, "user", "template");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

}
