package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

@Api(value = "TemplateVisibilityController")
@RestController
@RequestMapping("/v1/kuiper/templatevisibilities")
class TemplateVisibilityController extends BaseController {
  private static final Logger LOGGER = LoggerFactory.getLogger(TemplateVisibilityController.class);
  @Autowired
  private TemplateVisibilityService templateVisibilityService;
  private static final String[] FILTERS_PROPERTIES = new String[] {"template", "attributes", "buttons"};

  @ApiOperation(value = "根据指定模板查询可见性")
  @GetMapping("/findDetailsByTemplateId")
  public ResponseModel findDetailsByTemplateId(@ApiParam(name = "templateId", value = "模板id") @RequestParam("templateId") String templateId) {
    try {
      Set<TemplateVisibilityEntity> templateVisibility = this.templateVisibilityService.findDetailsByTemplateId(templateId);
      return this.buildHttpResultW(templateVisibility, FILTERS_PROPERTIES);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据指定模板和布局查询可见性
   * @param templateId
   * @param layoutType
   * @return
   */
  @ApiOperation(value = "根据指定模板和布局查询可见性")
  @GetMapping("/findDetailsByTemplateIdAndLayoutType")
  public ResponseModel findDetailsByTemplateIdAndLayoutType(@RequestParam() @ApiParam(value = "模板id") String templateId,
                                                            @RequestParam(defaultValue = "1") @ApiParam(value = "布局 1：PC、2：移动端、3：打印端", required = true) Integer layoutType) {
    try {
      Set<TemplateVisibilityEntity> templateVisibility = this.templateVisibilityService.findDetailsByTemplateIdAndLayoutType(templateId, layoutType);
      return this.buildHttpResultW(templateVisibility, FILTERS_PROPERTIES);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据指定模板、可见性名称和布局查询可见性
   * @param templateId
   * @param layoutType
   * @param visibilityName
   * @return
   */
  @ApiOperation(value = "根据指定模板、可见性名称和布局查询可见性")
  @GetMapping("/findDetailsByTemplateIdAndVisibilityNameAndLayoutType")
  public ResponseModel findDetailsByTemplateIdAndVisibilityNameAndLayoutType(@RequestParam @ApiParam(value = "模板id", required = true) String templateId,
                                                                             @RequestParam @ApiParam(value = "可见性名称", required = true) String visibilityName,
                                                                             @RequestParam(defaultValue = "1") @ApiParam(value = "布局 1：PC、2：移动端、3：打印端", required = true) Integer layoutType) {
    try {
      TemplateVisibilityEntity visibility = this.templateVisibilityService.findDetailsByTemplateIdAndVisibilityNameAndLayoutType(templateId, visibilityName, layoutType);
      return this.buildHttpResultW(visibility, FILTERS_PROPERTIES);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据模版code和版本号查询可见性,如果不传版本号，则使用默认的版本号
   * @param templateCode
   * @param version
   * @return
   */
  @ApiOperation(value = "根据模版code和版本号查询可见性", notes = "如果不传版本号，则使用默认的版本号")
  @GetMapping("/findByTemplateCodeAndVersion")
  public ResponseModel findByTemplateCodeAndVersion(@RequestParam @ApiParam(required = true, value = "模版编码") String templateCode,
                                                    @RequestParam(required = false) @ApiParam("模版版本") String version) {
    try {
      Set<TemplateVisibilityEntity> visibilities = templateVisibilityService.findByTemplateCodeAndVersion(templateCode, version);
      return this.buildHttpResultW(visibilities);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
