package com.bizunited.platform.kuiper.starter.repository;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.hibernate.query.internal.NativeQueryImpl;
import org.hibernate.transform.Transformers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;


/**
 *动态模型的SQL拼接逻辑和SQL执行逻辑
 *处理一般属性，关联属性，以及分组的SQL
 */
@Repository("DynamicInstanceCustomRepositoryImpl")
public class DynamicInstanceCustomRepositoryImpl implements DynamicInstanceCustomRepository{
  
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicInstanceCustomRepositoryImpl.class);
  
  
  @Autowired
  @PersistenceContext
  private EntityManager entityManager;


  /**
   * 基准的SQL执行逻辑
   */
  @Override
  public JSONArray execute(String sql) {
    Validate.notBlank(sql, "传入SQL不能为空，请检查!!");
    LOGGER.debug("动态模型执行sql ：{}", sql);
    Query query = entityManager.createNativeQuery(sql);
    query.unwrap(NativeQueryImpl.class).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
    List<?> list = query.getResultList();
    JSONArray jsonArr = JSON.parseArray(JSON.toJSONString(list));
    LOGGER.debug(jsonArr == null ? StringUtils.EMPTY : JSON.toJSONString(jsonArr));
    return jsonArr;
  }
  
  

  /**
   * 拼接ManyToManySQL
   */
  @Override
  public JSONArray findManyToMany(String parentTableName, String targetTableName , String parentId) {
    List<String> sortedNames = Arrays.asList(parentTableName , targetTableName).stream().sorted().collect(Collectors.toList());
    //求得中间表
    String mappingTable = String.format("%s_%s_mapping", sortedNames.get(0) , sortedNames.get(1));
    String mtmFormat = "select %s_id id from %s where %s_id = '%s'";
    String mtmSQL = String.format(mtmFormat, targetTableName,mappingTable,parentTableName,parentId);
    return this.execute(mtmSQL);
  }


  /**
   * 拼接主表、分组关系下的一般属性SQL
   */
  @Override
  public JSONArray findProperties(Set<TemplatePropertyEntity> properties, String mainTableName, String formInstanceId) {
    String columnsPattern = "%s %s,";
    StringBuilder sqlBuilder = new StringBuilder();
    for(TemplatePropertyEntity prop : properties) {
      Validate.notBlank(prop.getPropertyDbName(), "在动态模板中，主表一般属性对应的数据库字段信息必须有值，请检查!!");
      Validate.notBlank(prop.getPropertyName(), "在动态模板中，主表一般属性的名称必须有值，请检查!!");
      sqlBuilder.append(String.format(columnsPattern, prop.getPropertyDbName() , prop.getPropertyName()));
    }
    
    sqlBuilder.insert(0, "select ")
      .replace(sqlBuilder.length() - 1, sqlBuilder.length(), StringUtils.EMPTY)
      .append(" from ")
      .append(mainTableName)
      .append(String.format(" where form_instance_id = '%s'", formInstanceId));
    return this.execute(sqlBuilder.toString());
  }


  /**
   * 拼接明细关系下的一般属性SQL
   */
  @Override
  public JSONArray findProperties(Set<TemplatePropertyEntity> properties, String targetTableName, String parentTableName, String parentId, String targetPropertyDesc) {
    String columnsPattern = "target.%s %s,";
    StringBuilder sqlBuilder = new StringBuilder();
    
    for(TemplatePropertyEntity prop : properties) {
      Validate.notBlank(prop.getPropertyDbName(), "%s的一般属性对应的数据库字段信息必须有值，请检查!!",targetPropertyDesc);
      Validate.notBlank(prop.getPropertyName(), "%s一般属性的名称必须有值，请检查!!",targetPropertyDesc);
      sqlBuilder.append(String.format(columnsPattern, prop.getPropertyDbName() , prop.getPropertyName()));
    }
    
    sqlBuilder.insert(0, "select ")
      .replace(sqlBuilder.length() - 1, sqlBuilder.length(), StringUtils.EMPTY)
      .append(" from ")
      .append(String.format("%s target ", targetTableName))
      .append(String.format("left join %s parent on target.%s_id = parent.id ",parentTableName,parentTableName))
      .append(String.format("where parent.id = '%s'", parentId));
    return this.execute(sqlBuilder.toString());
  }


  /**
   * 拼接ManyToOneSQL
   */
  @Override
  public JSONArray findManyToOne(String parentTableName, String targetTableName, String parentPropertyDbName, String parentId) {
    String mtoFormat = "select target.id from %s target "
        + "left join %s parent on target.id = parent.%s "
        + "where parent.id = '%s'";
    String mtoSQL = String.format(mtoFormat, targetTableName,parentTableName,parentPropertyDbName,parentId);
    return this.execute(mtoSQL);
  }

  @Override
  public void executeNativeSQL(String pattern, Object... params) {
    String sql = String.format(pattern, params);
    Query sqlQuery = entityManager.createNativeQuery(sql);
    sqlQuery.executeUpdate();
  }
  
  @Override
  public List<?> executeQuerySql(String pattern, Object... params) {
    String querySql = String.format(pattern, params);
    Query query = entityManager.createNativeQuery(querySql);
    return query.getResultList();
  }
}
