package com.bizunited.platform.kuiper.starter.repository;

import java.util.Set;

import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import com.bizunited.platform.kuiper.entity.InstanceActivityEntity;

@Repository("InstanceActivityRepository")
public interface InstanceActivityRepository
    extends
      JpaRepository<InstanceActivityEntity, String>,
      JpaSpecificationExecutor<InstanceActivityEntity> {
  
  /**
   * 按照活动编号，查询指定的活动信息详情——包括关联信息
   * @param id 
   */
  @Query("from InstanceActivityEntity insa "
      + " left join fetch insa.instance ins "
      + " left join fetch insa.previousActivity pa "
      + " left join fetch insa.template tmp "
      + " left join fetch insa.templateVisibility tmpv "
      + " where insa.id = :id ")
  public InstanceActivityEntity findDetailsById(@Param("id") String id);
  
  /**
   * 按照第三方业务编号查询指定活动的详细信息
   * @param taskCode
   * @return
   */
  @Query("from InstanceActivityEntity insa "
      + " left join fetch insa.instance ins "
      + " left join fetch insa.previousActivity pa "
      + " left join fetch insa.template tmp "
      + " left join fetch insa.templateVisibility tmpv "
      + " where insa.taskCode = :taskCode ")
  public InstanceActivityEntity findDetailsByTaskCode(@Param("taskCode") String taskCode);
  
  /**
   * 按照指定的表单实例编号，查询其下已经发生的活动(包括每一条活动的明细情况)，并按照活动创建实例反序排列
   * @param instanceId 指定的表单实例编号
   * @return 
   */
  @Query("select distinct insa from InstanceActivityEntity insa "
      + " left join fetch insa.instance ins "
      + " left join fetch insa.previousActivity pa "
      + " left join fetch insa.template tmp "
      + " left join fetch insa.templateVisibility tmpv "
      + " where ins.id = :instanceId  order by insa.createTime desc ")
  public Set<InstanceActivityEntity> findDetailsByInstanceId(@Param("instanceId") String instanceId);
  
  /**
   * 按照指定的表单实例编号，查询其下已经发生的活动(不包括活动的明细情况)，并按照活动创建实例反序排列
   * @param instanceId 指定的表单实例编号
   * @return 
   */
  @Query("select distinct insa from InstanceActivityEntity insa "
      + " left join fetch insa.instance ins "
      + " where ins.id = :instanceId order by insa.createTime desc ")
  public Set<InstanceActivityEntity> findByInstanceId(@Param("instanceId") String instanceId);
  
  /**
   * 按照指定的表单实例编号，查询这个表单实例下创建时间最新的那条活动信息id
   * @param instanceId 指定的表单实例编号
   * @return 注意返回值在有查询结果的情况下是一个数组，数组的第一个值就是符合条件的表单实例id
   */
  @Query(value="select id,create_time from engine_form_instance_activity where instance_id = :instanceId  order by create_time desc limit 1 " , nativeQuery=true)
  public Object[][] findByInstanceIdAndMaxCreateTime(@Param("instanceId") String instanceId);


  /**
   * 根据实例ID和任务code查询
   * @param instanceId
   * @param taskCode
   * @return
   */
  @Query("select distinct insa from InstanceActivityEntity insa "
      + " left join fetch insa.instance ins "
      + " where ins.id = :instanceId and insa.taskCode = :taskCode ")
  InstanceActivityEntity findByInstanceIdAndTaskCode(@Param("instanceId") String instanceId, @Param("taskCode") String taskCode);
  /**
   * 查询当前可见性是否已经存在活动信息
   * @param visibilityId
   * @return
   */
  @Query(value="select count(*) from engine_form_instance_activity where template_visibility_id = :visibilityId ",nativeQuery=true)
  public int countByVisibilityId(@Param("visibilityId") String visibilityId);

  /**
   * 根据实例ID和可见性查询
   * @param instanceId
   * @param visibility
   * @return
   */
  @Query("select ia from InstanceActivityEntity ia inner join ia.instance i " +
      "inner join ia.templateVisibility tv " +
      "where i.id = :instanceId and tv.visibilityName = :visibility ")
  Set<InstanceActivityEntity> findByInstanceIdAndVisibility(@Param("instanceId") String instanceId, @Param("visibility") String visibility);

  /**
   * 根据任务的taskcode（外部业务编号），查询实例活动的基本信息（不包括关联信息）
   * @param taskCode 
   */
  public InstanceActivityEntity findByTaskCode(String taskCode);
  /**
   * 根据实例ID和可见性查询详情
   * @param instanceId
   * @param visibilityName
   * @return
   */
  @Query("select ia from InstanceActivityEntity ia inner join ia.instance i "
      + " inner join fetch ia.templateVisibility tv "
      + " left join fetch ia.instance ins "
      + " left join fetch ia.previousActivity pa "
      + " left join fetch ia.template tmp "
      + "where i.id = :instanceId and tv.visibilityName = :visibility ")
  Set<InstanceActivityEntity> findDetailsByInstanceIdAndVisibility(@Param("instanceId") String instanceId, @Param("visibility") String visibility);
}