package com.bizunited.platform.kuiper.starter.service;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 这是一个工具包性质的服务类，主要为了减少生成代码的工作量以及基于表单引擎完成的后端业务开发的工作量。
 * @author yinwenjie
 */
public class KuiperToolkitService extends NebulaToolkitService {
  /**
   * 在表单引擎中，针对模型的业务处理过程非常多，也经常涉及需要基于模型提取当前关联属性过滤的白名单。
   * 例如在静态模型类的对象进行JSON信息转换时，我们需要指定当前对象下哪些关联属性需要一同转换成JSON，哪些不需要。</p>
   * 该工具方法提供了一种提取当前模型所有关联属性白名单的功能，以便开发人员在需要的时候能够完成这样的以上的需求。
   * 举个列子，当前静态模型类A中的属性b关联了类B，属性c关联了类C，接着类B中的属性d，关联了类D。
   * 那么通过该方法构建的白名单列表，类似为[b,b.d,c]</p>
   * notes：目前该方法仅支持静态模板
   * @param template 传入完整的静态模板描述
   * @return 如果当前模板没有任何满足规定的关联属性需要形成白名单，这里也会返回一个空集合
   */
  public Set<String> buildAllWhiteList(TemplateEntity template) {
    Validate.notNull(template, "传入的模板信息不能为空，请检查!!");
    Validate.isTrue(StringUtils.equals(template.getType(), "static"), "该服务只支持静态模板提取白名单信息，请检查!!");
    Set<String> whiteList = new HashSet<>();
    //处理关联关系数据
    Set<TemplateRelationEntity> relations = template.getRelations();
    this.fetchRelationsWhiteList(relations , whiteList , null);
    //处理分组数据
    Set<TemplateGroupEntity> groups = template.getGroupRelations();
    this.fetchGroupsWhiteList(groups , whiteList , null);
    //处理明细关系数据
    Set<TemplateItemEntity> items = template.getItemRelations();
    this.fetchItemsWhiteList(items , whiteList , null);
    return whiteList;
  }
  
  /**
   * 和buildAllWhiteList方法不一样的是，虽然该方法也负责基于指定的静态模板构建关联属性的白名单，
   * 但是该方法只会对当前静态模板中描述的直接关联信息做出相应，忽略掉基于当前模板下分组、明细项的白名单处理。</p>
   * 举个列子，当前静态模型类A中的属性b关联了类B，属性c关联了类C，接着类B中的属性d，关联了类D。
   * 那么通过该方法构建的白名单列表，类似为[b,c]，且b、c两个属性都不是明细项或者分组项</p>
   * notes：目前该方法仅支持静态模板
   * @see #buildAllWhiteList(TemplateEntity);
   * @return 如果当前模板没有任何满足规定的关联属性需要形成白名单，这里也会返回一个空集合
   */
  public Set<String> buildWhiteList(TemplateEntity template) {
    Validate.notNull(template, "传入的模板信息不能为空，请检查!!");
    Validate.isTrue(StringUtils.equals(template.getType(), "static"), "该服务只支持静态模板提取白名单信息，请检查!!");
    Set<String> whiteList = new HashSet<>();
    //处理关联关系数据
    Set<TemplateRelationEntity> relations = template.getRelations();
    this.fetchRelationsWhiteList(relations , whiteList , null);
    return whiteList;
  }
  
  /**
   * 处理分组关系，从中提取白名单信息
   */
  private void fetchGroupsWhiteList(Set<TemplateGroupEntity> groups , Set<String> whiteList , String lastName) {
    if(CollectionUtils.isEmpty(groups)) {
      return;
    }
    //处理分组逻辑
    for(TemplateGroupEntity group : groups) {
      String currentName = group.getPropertyName();
      String currentWhiteName = this.buildCurrentWhiteName(lastName, currentName);
      whiteList.add(currentWhiteName);
      //处理分组下的关联关系情况
      if(!CollectionUtils.isEmpty(group.getRelations())) {
        this.fetchRelationsWhiteList(group.getRelations() , whiteList , currentWhiteName);
      }
      //处理分组下的明细关系情况
      if(!CollectionUtils.isEmpty(group.getItemRelations())) {
        this.fetchItemsWhiteList(group.getItemRelations() , whiteList , currentWhiteName);
      }
    }
  }
  
  /**
   * 处理关联关系，从中提取白名单信息
   */
  private void fetchRelationsWhiteList(Set<TemplateRelationEntity> relations , Set<String> whiteList , String lastName) {
    if(CollectionUtils.isEmpty(relations)) {
      return;
    }
    //过滤回嗍属性
    Set<TemplateRelationEntity> needRelations = relations.stream().filter(e -> !e.getBackProperty()).collect(Collectors.toSet());
    if(CollectionUtils.isEmpty(needRelations)) {
      return;
    }
    for(TemplateRelationEntity relation : needRelations) {
      String currentName = relation.getPropertyName();
      String currentWhiteName = this.buildCurrentWhiteName(lastName, currentName);
      whiteList.add(currentWhiteName);
    }
  }
  
  /**
   * 处理明细关系，从中提取白名单信息
   */
  private void fetchItemsWhiteList(Set<TemplateItemEntity> items , Set<String> whiteList , String lastName) {
    if(CollectionUtils.isEmpty(items)) {
      return;
    }
    for(TemplateItemEntity item : items) {
      String currentName = item.getPropertyName();
      String currentWhiteName = this.buildCurrentWhiteName(lastName, currentName);
      whiteList.add(currentWhiteName);
      //处理明细下的关联关系情况
      if(!CollectionUtils.isEmpty(item.getRelations())) {
        this.fetchRelationsWhiteList(item.getRelations() , whiteList , currentWhiteName);
      }
    }
  }
  
  
  /**
   * 根据当前属性名称和之前的属性名称，拼接成最终由层级关系的白名单数据
   * @param lastName
   * @param currentName
   * @return
   */
  private String buildCurrentWhiteName(String lastName , String currentName) {
    return StringUtils.isBlank(lastName) ? currentName : lastName + "." + currentName;
  }
}
