package com.bizunited.platform.kuiper.starter.service.init;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.enums.MigrateDataTypeEnum;
import com.bizunited.platform.common.service.init.InitProcessService;
import com.bizunited.platform.common.util.FileUtils;
import com.bizunited.platform.kuiper.entity.FromDetailsImportEntity;
import com.bizunited.platform.kuiper.entity.ListTemplateEntity;
import com.bizunited.platform.kuiper.entity.MigrateImportEntity;
import com.bizunited.platform.kuiper.starter.common.enums.ListTemplateImportEnum;
import com.bizunited.platform.kuiper.starter.service.FromDetailsImportService;
import com.bizunited.platform.kuiper.starter.service.ListTemplateService;
import com.bizunited.platform.kuiper.starter.service.migrate.MigrateImportService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_ADMIN_ACCOUNT;

/**
 * 列表模板初始化
 *
 * @Author: chenrong
 * @Date: 2020/11/5 20:29
 */
@Component
public class ListTemplateInitProcess implements InitProcessService {

  private static final Logger LOGGER = LoggerFactory.getLogger(ListTemplateInitProcess.class);
  private static final String INIT_FILE_PATH = ResourceLoader.CLASSPATH_URL_PREFIX.concat("templates/list/");
  private static final String INIT_EXCEL_PATH = ResourceLoader.CLASSPATH_URL_PREFIX.concat("templates/excel/");

  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private ListTemplateService listTemplateService;
  @Autowired
  private FromDetailsImportService fromDetailsImportService;
  @Autowired
  private MigrateImportService listImportService;
  @Autowired
  private VenusFileService fileService;

  @Override
  public boolean doProcess() {
    return true;
  }

  @Override
  public void init() {
    SecurityUtils.setPrincipal(DEFAULT_ADMIN_ACCOUNT);
    ListTemplateImportEnum[] enums = ListTemplateImportEnum.values();
    if (enums == null) {
      return;
    }
    for (ListTemplateImportEnum en : enums) {
      this.init(en);
    }
  }

  /**
   * 根据枚举初始化列表
   *
   * @param en
   */
  private void init(ListTemplateImportEnum en) {
    if (en == null) {
      return;
    }
    Set<ListTemplateEntity> oldList = this.listTemplateService.findByCode(en.getCode());
    if (!CollectionUtils.isEmpty(oldList)) {
      return;
    }
    //导入excel文件
    Resource excel = this.applicationContext.getResource(INIT_EXCEL_PATH.concat(en.getExcelName()));
    Resource template = this.applicationContext.getResource(INIT_FILE_PATH.concat(en.getFileName()));
    if (template == null) {
      return;
    }
    String templateCode = null;
    String templateVersion = null;
    ListTemplateEntity listTemplateEntity = null;
    //初始化列表
    try (InputStream inTemplate = template.getInputStream()) {
      if (inTemplate == null) {
        return;
      }
      MultipartFile multipartFile;
      multipartFile = FileUtils.transferMultipartFile(inTemplate, en.getFileName());
      MigrateImportEntity migrateImportEntity = listImportService.upload(multipartFile, MigrateDataTypeEnum.LIST_TEMPLATE.getType());
      if (migrateImportEntity == null) {
        return;
      }
      listImportService.execute(migrateImportEntity.getId(), ImportExecuteModeEnum.ADD.getMode());
      Set<ListTemplateEntity> nList = this.listTemplateService.findByCode(en.getCode());
      if (CollectionUtils.isEmpty(nList)) {
        return;
      }
      listTemplateEntity = nList.iterator().next();
      templateCode = listTemplateEntity.getCode();
      templateVersion = listTemplateEntity.getCversion();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
    }
    if(StringUtils.isBlank(en.getExcelName())) {
      return;
    }
    //初始化列表导入模板
    try (InputStream inExcel = excel.getInputStream()) {
      if (StringUtils.isBlank(templateCode) || StringUtils.isBlank(templateVersion)) {
        return;
      }
      FromDetailsImportEntity fromDetailsImportEntity = fromDetailsImportService.importTemplate(FileUtils.readBytes(inExcel, 10240),
              templateCode, templateVersion, en.getImportBtnCode(), en.getExcelViewName());
      if(fromDetailsImportEntity == null) {
        return;
      }
      String relativePath = listTemplateEntity.getRelativePath();
      String fileName = listTemplateEntity.getFileName();
      byte[] bytes = fileService.readFileContent(relativePath, fileName);
      JSONObject jsonObject = this.readBytesToJson(bytes);
      this.handleImportExcel(jsonObject, fromDetailsImportEntity);
      byte[] nBytes = this.writeObjectToBytes(jsonObject);
      fileService.saveFile(relativePath, listTemplateEntity.getFileName(), listTemplateEntity.getFileName(), nBytes);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
    }
  }

  /**
   * 替换json中的导入按钮配置
   * @param jsonObject
   * @param fromDetailsImportEntity
   */
  private void handleImportExcel(JSONObject jsonObject, FromDetailsImportEntity fromDetailsImportEntity) {
    if (jsonObject == null || fromDetailsImportEntity ==null || StringUtils.isBlank(fromDetailsImportEntity.getRelativePath()) || StringUtils.isBlank(fromDetailsImportEntity.getFileName())) {
      return;
    }
    System.out.println(jsonObject);
    JSONObject content = jsonObject.getJSONObject("content");
    JSONArray btnOperates = content.getJSONArray("btnOperates");
    JSONObject btnOperate = null;
    for (Object btnOperateObj : btnOperates) {
      JSONObject tmp = (JSONObject) btnOperateObj;
      if (StringUtils.equals(tmp.getString("code"), fromDetailsImportEntity.getButtonCode())) {
        btnOperate = tmp;
        break;
      }
    }
    Validate.notNull(btnOperate, "没有获取到导入按钮");
    JSONArray events = btnOperate.getJSONArray("events");
    List<Object> comment = events.stream().filter(e -> "导入".equals(((JSONObject) e).get("comment"))).collect(Collectors.toList());
    Validate.notEmpty(comment);
    JSONObject commentJson = (JSONObject) comment.get(0);
    JSONObject importTemplate = commentJson.getJSONObject("content").getJSONObject("importTemplate");
    importTemplate.put("url", "/kuiper/fromDetailsImports/downloadTemplate?relativeLocal=".concat(fromDetailsImportEntity.getRelativePath()).concat("&fileName=").concat(fromDetailsImportEntity.getFileName()));
  }

  /**
   * 将object读成bytes内容
   * @param obj
   * @return
   */
  private byte[] writeObjectToBytes(Object obj){
    byte[] bytes;
    try (ByteArrayOutputStream out = new ByteArrayOutputStream();
         ObjectOutputStream sOut = new ObjectOutputStream(out);){
      sOut.writeObject(obj);
      sOut.flush();
      bytes= out.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      return new byte[] {};
    }
    return bytes;
  }
  /**
   * 将bytes内容读取成json信息
   * @param bytes 文件字节
   * @return
   */
  private JSONObject readBytesToJson(byte[] bytes) {
    Object layoutObject;
    try (ByteArrayInputStream in = new ByteArrayInputStream(bytes);
         ObjectInputStream sIn = new ObjectInputStream(in);){
      layoutObject = sIn.readObject();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return null;
    }
    if(layoutObject == null) {
      return null;
    }

    return JSON.parseObject(JSON.toJSONString(layoutObject));
  }
}
