package com.bizunited.platform.kuiper.starter.service.instances.handle;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.bizunited.platform.core.service.invoke.InvokeResponseHandle;
import com.bizunited.platform.kuiper.entity.InstanceActivityEntity;
import com.bizunited.platform.kuiper.entity.InstanceActivityLogEntity;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.common.enums.RelationsTypeEnum;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * 该链职责节点主要是对本次Instance表单的变化进行追踪，并存储相关变化日志到数据库和服务器文件系统上<br>
 * 注：只对之前责任链中没有出现异常的，且调用方法是写操作，且是WRITE性质的，且有相应的实例及活动信息的数据进行日志记录</p>
 * 
 * 一个标准的数据差异化描述的json结构如下：</br>
 * {</br>
 * // 主模型下的一般属性变化记录在这里</br>
 * "properties":[{</br>
 * &nbsp;&nbsp; // 属性类型</br>
 * &nbsp;&nbsp;  "propertyClassName":"java.lang.String",</br>
 * &nbsp;&nbsp;  // 属性名</br>
 * &nbsp;&nbsp;  "propertyName":"X",</br>
 * &nbsp;&nbsp;  // 可能的数据字段名</br>
 * &nbsp;&nbsp;  "propertyDbName":"X",</br>
 * &nbsp;&nbsp;  // 变化类型，包括三种“UPDATA”、“NEW”和“DELETE”</br>
 * &nbsp;&nbsp;  // 如果当前属性为一般属性或者选择性关联属性，则只可能有一种变化类型“UPDATA”</br>
 * &nbsp;&nbsp;  "changeType":"UPDATE",</br>
 * &nbsp;&nbsp;  // 变化前的值，可能没有</br>
 * &nbsp;&nbsp;  "beforeValue":"false",</br>
 * &nbsp;&nbsp;  // 变化后的值，可能没有</br>
 * &nbsp;&nbsp;  "afterValue":"true"</br>
 * },{</br>
 * &nbsp;&nbsp;  "propertyClassName":"java.lang.String",</br>
 * &nbsp;&nbsp;  "propertyName":"Y",</br>
 * &nbsp;&nbsp;  "propertyDbName":"Y",</br>
 * &nbsp;&nbsp;  "changeType":"UPDATE",</br>
 * &nbsp;&nbsp;  "beforeValue":"XXXXX",</br>
 * &nbsp;&nbsp;  "afterValue":"YYYYY"</br>
 * }],</br>
 * // 主模型下的选择性关联属性的变化记录在这里</br>
 * // 根据关联类型的不同（ManyToMany或者ManyToOne）,只是beforeValue和afterValue的值不一样</br>
 * // ManyToMany的值是数组</br>
 * "relations":[{</br>
 * &nbsp;&nbsp;  // 属性名</br>
 * &nbsp;&nbsp;  "propertyName":"Z",</br>
 * &nbsp;&nbsp;  // 可能的数据字段名</br>
 * &nbsp;&nbsp;  "propertyDbName":"Z",</br>
 * &nbsp;&nbsp;  // 只可能是“UPDATE”</br>
 * &nbsp;&nbsp;  "changeType":"UPDATE",</br>
 * &nbsp;&nbsp;  // 变化前的值，可能没有</br>
 * &nbsp;&nbsp;  "beforeValue":"XXXXXX",</br>
 * &nbsp;&nbsp;  // 变化后的值，可能没有</br>
 * &nbsp;&nbsp;  "afterValue":"YYYYYYY"</br>
 * },{</br>
 * &nbsp;&nbsp;  // 属性名</br>
 * &nbsp;&nbsp;  "propertyName":"Z",</br>
 * &nbsp;&nbsp;  // 关联的类型，只可能是UPDATE</br>
 * &nbsp;&nbsp;  "changeType":"UPDATE",</br>
 * &nbsp;&nbsp;  "beforeValue":["xxx1","xxxx2","xxxx3","xxxx4"],</br>
 * &nbsp;&nbsp;  "afterValue":["yyyy1","yyyy2","yyyy3","xxxx3"]</br>
 * }],</br>
 * // 主模型下的明细项变化信息，在这里进行描述</br>
 * // 一个主模型下可能关联多个明细信息，</br>
 * "items":[{</br>
 * &nbsp;&nbsp;  // 当前明细的属性名</br>
 * &nbsp;&nbsp;  "propertyName":"A",</br>
 * &nbsp;&nbsp;  // 当前变化的类型，只可能是UPDATE</br>
 * &nbsp;&nbsp;  "changeType":"UPDATE",</br>
 * &nbsp;&nbsp;  // 这里记录当前这个明细项“A”的每一条明细的变化情况，</br>
 * &nbsp;&nbsp;  // 可能的变化类型就包括“UPDATA”、“NEW”和“DELETE”</br>
 * &nbsp;&nbsp;  "datas":[{</br>
 * &nbsp;&nbsp;  // 当前明细项之前的id信息</br>
 * &nbsp;&nbsp;  "id":"123456789",</br>
 * &nbsp;&nbsp;  // 明细项变化的类型</br>
 * &nbsp;&nbsp;  "changeType":"DELETE",</br>
 * &nbsp;&nbsp;  // 当前明细项条目每一个一般属性的变化情况记录在这里（当然，只记录变化的）</br>
 * &nbsp;&nbsp;  "properties":[{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;   // 当前属性的类型</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyClassName":"java.lang.String",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    // 当前属性名</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyName":"A1",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    // 可能的数据库字段名</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyDbName":"A1",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    // 当前字段的变化类型</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "changeType":"DELETE",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    // 变化前的值，可能没有</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "beforeValue":"false"</br>
 * &nbsp;&nbsp;  },{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyClassName":"java.lang.String",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyName":"A2",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyDbName":"A2",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "changeType":"DELETE",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "beforeValue":"999999999"</br>
 * &nbsp;&nbsp;  }],</br>
 * &nbsp;&nbsp;  // 当前明细性条目每一个关联属性的变化情况记录在这里（当前，只记录变化的）</br>
 * &nbsp;&nbsp;  "relations":[{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyName":"A3",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyDbName":"A3",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "changeType":"DELETE",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "beforeValue":"XXXXXX"</br>
 * &nbsp;&nbsp;   },{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;     "propertyName":"A4",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;     "changeType":"DELETE",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;     "beforeValue":["xxx1","xxxx2","xxxx3","xxxx4"]</br>
 * &nbsp;&nbsp;   }]</br>
 * &nbsp;&nbsp; },</br>
 *  // 这里记录当前这个明细项“A”的第2条明细的变化情况，</br>
 * &nbsp;&nbsp; {</br>
 * &nbsp;&nbsp; // 当前明细项的ID主键信息</br>
 * &nbsp;&nbsp; "id":"567898764",</br>
 * &nbsp;&nbsp; // 变化类型，包括三种“UPDATA”、“NEW”和“DELETE”</br>
 * &nbsp;&nbsp; "changeType":"NEW",</br>
 * &nbsp;&nbsp; "properties":[{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;   "propertyClassName":"java.lang.String",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;   "propertyName":"A1",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;   "propertyDbName":"A1",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;   "changeType":"NEW",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;   "afterValue":"false"</br>
 * &nbsp;&nbsp;  },{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyClassName":"java.lang.String",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyName":"A2",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyDbName":"A2",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "changeType":"NEW",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "afterValue":"999999999"</br>
 * &nbsp;&nbsp;  }],</br>
 * &nbsp;&nbsp;  "relations":[{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyName":"A3",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyDbName":"A3",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "changeType":"NEW",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "afterValue":"XXXXXX"</br>
 * &nbsp;&nbsp;  },{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "propertyName":"A4",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "changeType":"NEW",</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    "afterValue":["xxx1","xxxx2","xxxx3","xxxx4"]</br>
 * &nbsp;&nbsp;  }]</br>
 *  }]</br>
 * },</br>
 * // 这里记录了主模型下，第二个明细属性的变化内容</br>
 * {</br>
 *   ………………</br>
 * }],</br>
 * // 主模型下，分组信息的变化情况记录在这里</br>
 * "groups":[{</br>
 * &nbsp;&nbsp;  // 当前分组的属性名</br>
 * &nbsp;&nbsp;  "propertyName":"G1",</br>
 * &nbsp;&nbsp;  // 当前分组变化的类型，只可能是UPDATE</br>
 * &nbsp;&nbsp;  "changeType":"UPDATE",</br>
 * &nbsp;&nbsp;  // 这里记录当前这个分组项“G1”的变化情况——一般属性的变化情况</br>
 * &nbsp;&nbsp;  "properties":[],</br>
 * &nbsp;&nbsp;  // 这里记录当前这个分组项“G1”的变化情况——关联属性的变化情况</br>
 * &nbsp;&nbsp;  "items":[{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    ....... // 这是分组信息下第一个明细属性的变化情况，</br>
 * &nbsp;&nbsp;  },{</br>
 * &nbsp;&nbsp;&nbsp;&nbsp;    ....... // 这是分组信息下第二个明细属性的变化情况，</br>
 * &nbsp;&nbsp;  }]</br>
 *  }]</br>
 * }</br>
 */
public abstract class AbstractInstanceDetailsLogUpdateHandle  implements InvokeResponseHandle {
  private static final String PROPERTIES = "properties";
  private static final String RELATIONS = "relations";
  private static final String PROPERTY_NAME = "propertyName";
  private static final String CHANGE_TYPE = "changeType";
  private static final String UPDATE = "UPDATE";
  private static final String PROPERTY_CLASS_NAME = "propertyClassName";
  private static final String SPECIAL_EXPRESSION = "$[?(@.id = '";
  private static final String BEFORE_VALUE = "beforeValue";
  private static final String AFTER_VALUE = "afterValue";
  @Autowired
  private TemplateService templateService;
  @Autowired
  private InstanceActivityLogService instanceActivityLogService;
  @Autowired
  private InstanceActivityService instanceActivityService;
  
  /**
   * 该私有关方法是一个公用方法，用来通过jsonPath表单式的方式，在指定的json段落位置获取满足条件的json属性数组信息</br>
   * 如果没有获取到任何结果，则返回一个空的json数组
   * @param details 指定的json段落
   * @param propertyName 指定的json段落中的属性
   * @return
   */
  private JSONArray getPropertyArray(JSONObject details , String propertyName) {
    Object relationValues = JSONPath.eval(details, "$['" + propertyName + "']");
    JSONArray relations = null;
    if(relationValues instanceof JSONArray) {
      relations = (JSONArray)relationValues;
    } else {
      relations = new JSONArray();
    }
    return relations;
  }

  /**
   * 该私有关方法是一个公用方法，用来通过jsonPath表单式的方式，在指定的json段落位置获取满足条件的json属性信息</br>
   * 如果没有获取到任何结果，则返回一个空的json对象
   * @param details 指定的json段落
   * @param propertyName 指定的json段落中的属性
   * @return
   */
  private JSONObject getPropertyObject(JSONObject details , String propertyName) {
    Object relationValue = JSONPath.eval(details, "$['" + propertyName + "']");
    JSONObject relation = null;
    if(relationValue instanceof JSONObject) {
      relation = (JSONObject)relationValue;
    } else {
      relation = new JSONObject();
    }
    return relation;
  }

  /**
   * 本过滤节点的主要处理过程在这里
   * @param context 代理器上下文
   * @param templateId 当前的模板id
   * @param creator 当前操作者
   * @return
   */
  void handle(JSONObject beforeDetails , JSONObject afterDetails , String templateId , String creator , String activityId) {
    /*
     * 处理过程如下：
     * 1、首先确认工作条件，只有当前是WRITE性质的方法，且带有InstanceActivyId的情况才进行处理（在doFilter方法中已经完成）
     * 2、准备工作主要包括从当前线程上下文读取Write操作之前的信息，以及从当前上下文读取Write操作之后的信息
     * 3、对比开始：
     *   3.1、首先对比主表单一般性数据的变化
     *   3.2、接着是主表单上的ManyToOne、ManyToMany关联性质属性的变化
     *   3.2、然后对比主表单下OneToMany性质的明细数据的变化，包括了一般数据、ManyToOne、ManyToMany性质的变化
     *   3.3、最后对比主表单下OneToOne性质的分组数据表单，包括一般数据变化、 ManyToOne、ManyToMany性质和OneToMany性质的变化
     *   注意，以上这些方法都是可以重用的
     * 4、完成后，保存到文件系统和数据库系统
     * */

    // 2、========
    TemplateEntity currentTemplate = this.templateService.findDetailsById(templateId);
    Validate.notNull(currentTemplate, "没有发现指定的模板信息，请检查。");

    // 3.1、========
    JSONObject changeTemplate = new JSONObject();
    Set<TemplatePropertyEntity> templateProperties = currentTemplate.getProperties();
    JSONArray changeProperties = this.analysisProperties(beforeDetails, afterDetails, templateProperties);
    if(changeProperties != null && !changeProperties.isEmpty()) {
      changeTemplate.put(PROPERTIES, changeProperties);
    }

    // 3.2、========
    Set<TemplateRelationEntity> relations = currentTemplate.getRelations();
    JSONArray changeRelations = new JSONArray();
    if(!CollectionUtils.isEmpty(relations)) {
      for (TemplateRelationEntity relationItem : relations) {
        String propertyName = relationItem.getPropertyName();
        JSONObject relationObject = null;
        if(StringUtils.equals(relationItem.getRelationType(), RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
          // 取得json数据结构对应的数据位置：更新前——propertyName属性
          JSONArray beforeRelations = this.getPropertyArray(beforeDetails, propertyName);
          // 取得json数据结构对应的数据位置：更新后——propertyName属性
          JSONArray afterRelations = this.getPropertyArray(afterDetails, propertyName);
          relationObject = this.analysisManyToManyRelation(beforeRelations, afterRelations, relationItem);
        } else if(StringUtils.equals(relationItem.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())) {
          // 取得json数据结构对应的数据位置
          JSONObject beforeRelation = this.getPropertyObject(beforeDetails, propertyName);
          // 取得json数据结构对应的数据位置
          JSONObject afterRelation = this.getPropertyObject(afterDetails, propertyName);
          relationObject = this.analysisManyToOneRelation(beforeRelation, afterRelation, relationItem);
        }
        if(relationObject != null) {
          changeRelations.add(relationObject);
        }
      }
    }
    if(!changeRelations.isEmpty()) {
      changeTemplate.put(RELATIONS, changeRelations);
    }

    // 3.3、=======
    Set<TemplateItemEntity> items = currentTemplate.getItemRelations();
    JSONArray changeItems = new JSONArray();
    if(!CollectionUtils.isEmpty(items)) {
      for (TemplateItemEntity item : items) {
        String itemName = item.getPropertyName();
        // 取得json数据结构对应的数据位置：更新前——itemName属性
        JSONArray beforeItems = this.getPropertyArray(beforeDetails, itemName);
        // 取得json数据结构对应的数据位置：更新后——itemName属性
        JSONArray afterItems = this.getPropertyArray(afterDetails, itemName);
        JSONObject itemObject = this.analysisItem(beforeItems, afterItems, item);
        if(itemObject != null) {
          changeItems.add(itemObject);
        }
      }
    }
    if(!changeItems.isEmpty()) {
      changeTemplate.put("items", changeItems);
    }

    // 3.4、======
    Set<TemplateGroupEntity> groups = currentTemplate.getGroupRelations();
    JSONArray changeGroups = new JSONArray();
    if(!CollectionUtils.isEmpty(groups)) {
      for (TemplateGroupEntity groupItem : groups) {
        String groupName = groupItem.getPropertyName();
        // 取得json数据结构对应的数据位置
        JSONObject beforeGroup = this.getPropertyObject(beforeDetails, groupName);
        // 取得json数据结构对应的数据位置
        JSONObject afterGroup = this.getPropertyObject(afterDetails, groupName);
        JSONObject groupObject = this.analysisGroup(beforeGroup, afterGroup, groupItem);
        if(groupObject != null) {
          changeGroups.add(groupObject);
        }
      }
    }
    if(!changeGroups.isEmpty()) {
      changeTemplate.put("groups", changeGroups);
    }

    // 4、=======
    // 如果没有相关的数据不一致，则不用记录本次变更日志
    if(changeTemplate.isEmpty()) {
      return;
    }
    InstanceActivityLogEntity instanceActivityLog = new InstanceActivityLogEntity();
    instanceActivityLog.setCreateTime(new Date());
    instanceActivityLog.setCreator(creator);
    InstanceActivityEntity currentActivity = new InstanceActivityEntity();
    currentActivity.setId(activityId);
    instanceActivityLog.setInstanceActivity(currentActivity);
    this.instanceActivityLogService.create(instanceActivityLog, changeTemplate);
    //更新实例活动的操作人和日期
    this.instanceActivityService.updateModifyer(activityId, creator);
  }

  /**
   * 该私有方法用于分析两个指定的分组属性对象的差异，并生成一个结果性质的json数组对象返回。
   * 返回的每一个对象结构，参见类定义：XXXX
   * @param beforeProperties 进行“WRITE”操作前的一般属性信息的详情
   * @param afterProperties 进行“WRITE”操作后的一般属性信息的详情
   * @param templateItems 当前描述的模板信息明细
   */
  private JSONObject analysisGroup(JSONObject beforeGroup , JSONObject afterGroup , TemplateGroupEntity templateGroup) {
    /*
     * 分组下可以包括一般属性、关联属性、明细属性。
     * 所以处理过程如下所示：
     *
     * 1、首先分析当前分组下，一般属性的变化情况
     * 2、然后分析当前分组下，可能的关联属性的变化情况
     * 3、接着分析当前分组下，可能的明细项的变化情况
     * */
    JSONObject changeGroup = new JSONObject();
    String propertyName = templateGroup.getPropertyName();
    changeGroup.put(PROPERTY_NAME, propertyName);
    changeGroup.put(CHANGE_TYPE, UPDATE);

    // 1、======
    Set<TemplatePropertyEntity> templateProperties = templateGroup.getProperties();
    JSONArray changeProperties =  this.analysisProperties(beforeGroup, afterGroup, templateProperties);
    if(!CollectionUtils.isEmpty(changeProperties)) {
      changeGroup.put(PROPERTIES, changeProperties);
    }

    // 2、======
    Set<TemplateRelationEntity> templateRelations = templateGroup.getRelations();
    JSONArray changeRelations = new JSONArray();
    if(!CollectionUtils.isEmpty(templateRelations)) {
      for (TemplateRelationEntity templateRelation : templateRelations) {
        String relationPropertyName = templateRelation.getPropertyName();
        JSONObject relationObject = null;
        if(StringUtils.equals(templateRelation.getRelationType(), RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
          // 取得json数据结构对应的数据位置：更新前——propertyName属性
          JSONArray beforeRelations = this.getPropertyArray(beforeGroup, propertyName);
          // 取得json数据结构对应的数据位置：更新后——propertyName属性
          JSONArray afterRelations = this.getPropertyArray(afterGroup, propertyName);
          relationObject = this.analysisManyToManyRelation(beforeRelations, afterRelations, templateRelation);
        } else if(StringUtils.equals(templateRelation.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())) {
          // 取得json数据结构对应的数据位置
          JSONObject beforeRelation = this.getPropertyObject(beforeGroup, relationPropertyName);
          // 取得json数据结构对应的数据位置
          JSONObject afterRelation = this.getPropertyObject(afterGroup, relationPropertyName);
          relationObject = this.analysisManyToOneRelation(beforeRelation, afterRelation, templateRelation);
        }
        if(relationObject != null) {
          changeRelations.add(relationObject);
        }
      }
    }
    if(!changeRelations.isEmpty()) {
      changeGroup.put(RELATIONS, changeRelations);
    }

    // 3、======
    Set<TemplateItemEntity> templateItems = templateGroup.getItemRelations();
    JSONArray changeItems = new JSONArray();
    if(!CollectionUtils.isEmpty(templateItems)) {
      for (TemplateItemEntity templateItem : templateItems) {
        String itemName = templateItem.getPropertyName();
        // 取得json数据结构对应的数据位置
        JSONArray beforeItems = this.getPropertyArray(beforeGroup, itemName);
        // 取得json数据结构对应的数据位置
        JSONArray afterItems = this.getPropertyArray(afterGroup, itemName);
        JSONObject itemObject = this.analysisItem(beforeItems, afterItems, templateItem);
        if(itemObject != null) {
          changeItems.add(itemObject);
        }
      }
    }
    if(!changeItems.isEmpty()) {
      changeGroup.put("items", changeItems);
    }

    // 注意，如果没有任何变化，则返回null，表示这个分组不需要记录变化
    if(CollectionUtils.isEmpty(changeProperties) || !changeRelations.isEmpty() || !changeItems.isEmpty()) {
      return changeGroup;
    }
    return null;
  }

  /**
   * 该私有方法用于分析两个指定的明细属性对象的差异，并生成一个结果性质的json数组对象返回。
   * 返回的每一个对象结构，参见类定义：XXXX
   * @param beforeItem 进行“WRITE”操作前的明细属性信息的详情
   * @param afterItem 进行“WRITE”操作后的明细属性信息的详情
   * @param templateItem 当前描述的模板信息明细
   */
  private JSONObject analysisItem(JSONArray beforeItems , JSONArray afterItems , TemplateItemEntity templateItem) {
    /*
     * 明细的变化情况分为三种：
     * 如果在操作前有明细项，但是操作后没有了明细项，那么说明明细项是删除操作
     * 如果在操作前没有明细项，但是操作后有了明细项，那么说明明细项是添加操作
     * 出去以上两种情况，明细项都是修改操作。而明细项中的所有一般属性字段，只有“修改”这种操作类型
     *
     * 那么操作过程如下：
     * 1、首先构造这个item的基本信息，包括propertyName、changeType等信息
     * 并判定当前明细项是以上哪一种操作类型，并分裂成三个集合：
     * 明细项删除集合、明细项新增集合、明细项修改集合
     *
     * 2、对明细项删除集合中的每一条元素进行变化日志记录操作
     * （包括所有一般性属性和关联属性的变化情况）
     * 3、对明细项新增集合中的每一条元素进行变化日志记录操作
     * 4、对明细项修改集合中的每一条元素进行变化日志记录操作
     *
     * (注意：如果1.3所使用的集合，其中没有任何属性发生变化，那么说明这条明细项没有变化)
     * */
    String propertyClassName = templateItem.getPropertyClassName();
    String propertyName = templateItem.getPropertyName();

    // 1、======
    JSONObject changeObjects = new JSONObject();
    changeObjects.put(PROPERTY_CLASS_NAME, propertyClassName);
    changeObjects.put(PROPERTY_NAME, propertyName);
    // 判定的依据就是明细项中的id属性
    Set<String> beforeItemIdArrays = Sets.newHashSet();
    Set<String> afterItemIdArrays = Sets.newHashSet();
    if(beforeItems != null) {
      Object beforeItemArrayValue = JSONPath.eval(beforeItems, "$['id']");
      JSONArray beforeItemArray = null;
      if(beforeItemArrayValue instanceof JSONArray) {
        beforeItemArray = (JSONArray)beforeItemArrayValue;
      }
      beforeItemIdArrays = beforeItemArray != null?Sets.newHashSet(beforeItemArray.toArray(new String[]{})):Sets.newHashSet();
    }
    if(afterItems != null) {
      Object afterItemArrayObject = JSONPath.eval(afterItems, "$['id']");
      JSONArray afterItemArray = null;
      if(afterItemArrayObject instanceof JSONArray) {
        afterItemArray = (JSONArray)afterItemArrayObject;
      }
      afterItemIdArrays = afterItemArray != null?Sets.newHashSet(afterItemArray.toArray(new String[]{})):Sets.newHashSet();
    }
    // 以下集合为删除的集合
    SetView<String> deleteIds = Sets.difference(beforeItemIdArrays, afterItemIdArrays);
    Set<JSONObject> deleteItems = Sets.newLinkedHashSet();
    if(deleteIds != null && !deleteIds.isEmpty()) {
      deleteIds.stream().forEach(itemId -> {
        JSONArray items = (JSONArray)JSONPath.eval(beforeItems, SPECIAL_EXPRESSION + itemId + "')]");
        deleteItems.add(items.getJSONObject(0));
      });
    }
    // 以下集合为新增的集合
    SetView<String> newIds = Sets.difference(afterItemIdArrays , beforeItemIdArrays);
    Set<JSONObject> newItems = Sets.newLinkedHashSet();
    if(newIds != null && !newIds.isEmpty()) {
      newIds.stream().forEach(itemId -> {
        JSONArray items = (JSONArray)JSONPath.eval(afterItems, SPECIAL_EXPRESSION + itemId + "')]");
        newItems.add(items.getJSONObject(0));
      });
    }
    // 以下集合为修改的集合
    SetView<String> updateIds = Sets.intersection(afterItemIdArrays , beforeItemIdArrays);
    Set<JSONObject> updateItems = Sets.newLinkedHashSet();
    if(updateIds != null && !updateIds.isEmpty()) {
      updateIds.stream().forEach(itemId -> {
        JSONArray items = (JSONArray)JSONPath.eval(afterItems, SPECIAL_EXPRESSION + itemId + "')]");
        updateItems.add(items.getJSONObject(0));
      });
    }

    // 2、=======
    Set<TemplatePropertyEntity> templateProperties = templateItem.getProperties();
    Set<TemplateRelationEntity> templateRelations = templateItem.getRelations();
    JSONArray datas = new JSONArray();
    if(deleteItems != null && !deleteItems.isEmpty()) {
      for (JSONObject deleteItem : deleteItems) {
        JSONObject deleteChange = new JSONObject();
        String id = deleteItem.getString("id");
        deleteChange.put("id", id);
        deleteChange.put(CHANGE_TYPE, "DELETE");
        // 2.1、=====首先是一般性属性
        if(templateProperties != null && !templateProperties.isEmpty()) {
          JSONArray deletePropertiesChanges = this.analysisProperties(deleteItem, new JSONObject(), templateProperties);
          // 组装这条明细记录下所有的一般属性变化情况
          if(!deletePropertiesChanges.isEmpty()) {
            deleteChange.put(PROPERTIES, deletePropertiesChanges);
          }
        }

        // 2.2、===然后是关联属性
        if(templateRelations != null && !templateRelations.isEmpty()) {
          JSONArray deleteRelationsChanges = new JSONArray();
          for (TemplateRelationEntity templateRelationItem : templateRelations) {
            JSONObject deleteRelationsChange = null;
            String itemPropertyName = templateRelationItem.getPropertyName();
            // 注意，关联情况分为两种，ManyToMany和ManyToOne
            if(StringUtils.equals(templateRelationItem.getRelationType(), RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
              JSONArray deleteJSONArray = deleteItem.getJSONArray(itemPropertyName);
              deleteRelationsChange = this.analysisManyToManyRelation(deleteJSONArray == null?new JSONArray():deleteJSONArray , new JSONArray(), templateRelationItem);
            } else if(StringUtils.equals(templateRelationItem.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())) {
              JSONObject deleteJSONObject = deleteItem.getJSONObject(itemPropertyName);
              deleteRelationsChange = this.analysisManyToOneRelation(deleteJSONObject == null?new JSONObject():deleteJSONObject, new JSONObject(), templateRelationItem);
            }
            if(deleteRelationsChange != null) {
              deleteRelationsChanges.add(deleteRelationsChange);
            }
          }
          if(!deleteRelationsChanges.isEmpty()) {
            deleteChange.put(RELATIONS, deleteRelationsChanges);
          }
        }
        datas.add(deleteChange);
      }
    }

    // 3、=====
    if(newItems != null && !newItems.isEmpty()) {
      for (JSONObject newItem : newItems) {
        JSONObject newChange = new JSONObject();
        String id = newItem.getString("id");
        newChange.put("id", id);
        newChange.put(CHANGE_TYPE, "NEW");
        // 3.1、=====首先是一般性属性
        if(templateProperties != null && !templateProperties.isEmpty()) {
          JSONArray newPropertiesChanges = this.analysisProperties(new JSONObject(), newItem, templateProperties);
          if(newPropertiesChanges != null && !newPropertiesChanges.isEmpty()) {
            newChange.put(PROPERTIES, newPropertiesChanges);
          }
        }
        // 3.2、=====然后是关联属性
        if(templateRelations != null && !templateRelations.isEmpty()) {
          JSONArray newRelationsChanges = new JSONArray();
          for (TemplateRelationEntity templateRelationItem : templateRelations) {
            JSONObject newRelationsChange = null;
            String itemPropertyName = templateRelationItem.getPropertyName();
            // 注意，关联情况分为两种，ManyToMany和ManyToOne
            if(StringUtils.equals(templateRelationItem.getRelationType(), RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
              JSONArray newJSONArray = newItem.getJSONArray(itemPropertyName);
              newRelationsChange = this.analysisManyToManyRelation(new JSONArray(), newJSONArray == null?new JSONArray():newJSONArray , templateRelationItem);
            } else if(StringUtils.equals(templateRelationItem.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())) {
              JSONObject newJSONObject = newItem.getJSONObject(itemPropertyName);
              newRelationsChange = this.analysisManyToOneRelation(new JSONObject(), newJSONObject == null?new JSONObject():newJSONObject , templateRelationItem);
            }
            if(newRelationsChange != null) {
              newRelationsChanges.add(newRelationsChange);
            }
          }
          if(!newRelationsChanges.isEmpty()) {
            newChange.put(RELATIONS, newRelationsChanges);
          }
        }
        datas.add(newChange);
      }
    }

    // 4、====
    if(updateItems != null && !updateItems.isEmpty()) {
      for (JSONObject updateItem : updateItems) {
        boolean isUpdateItem = false;
        JSONObject updateChange = new JSONObject();
        String id = updateItem.getString("id");
        JSONArray beforeUpdateItems = (JSONArray)JSONPath.eval(beforeItems, SPECIAL_EXPRESSION + id + "')]");
        updateChange.put("id", id);
        updateChange.put(CHANGE_TYPE, UPDATE);
        // 4.1、=====首先是一般性属性
        if(templateProperties != null && !templateProperties.isEmpty()) {
          JSONArray updatePropertiesChanges = this.analysisProperties(beforeUpdateItems.getJSONObject(0), updateItem, templateProperties);
          if(updatePropertiesChanges != null && !updatePropertiesChanges.isEmpty()) {
            isUpdateItem = true;
            updateChange.put(PROPERTIES, updatePropertiesChanges);
          }
        }
        // 3.2、=====然后是关联属性
        if(templateRelations != null && !templateRelations.isEmpty()) {
          JSONArray updateRelationsChanges = new JSONArray();
          for (TemplateRelationEntity templateRelationItem : templateRelations) {
            JSONObject updateRelationsChange = null;
            String itemPropertyName = templateRelationItem.getPropertyName();
            // 注意，关联情况分为两种，ManyToMany和ManyToOne
            if(StringUtils.equals(templateRelationItem.getRelationType(), RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
              JSONArray afterUpdateJSONArray = updateItem.getJSONArray(itemPropertyName);
              JSONArray beforeUpdateJSONArray = beforeUpdateItems.getJSONObject(0).getJSONArray(itemPropertyName);
              updateRelationsChange = this.analysisManyToManyRelation(beforeUpdateJSONArray == null?new JSONArray():beforeUpdateJSONArray , afterUpdateJSONArray == null?new JSONArray():afterUpdateJSONArray, templateRelationItem);
            } else if(StringUtils.equals(templateRelationItem.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())) {
              JSONObject afterUpdateJSONObject = updateItem.getJSONObject(itemPropertyName);
              JSONObject beforeUpdateJSONObject = beforeUpdateItems.getJSONObject(0).getJSONObject(itemPropertyName);
              updateRelationsChange = this.analysisManyToOneRelation(beforeUpdateJSONObject == null?new JSONObject():beforeUpdateJSONObject, afterUpdateJSONObject == null?new JSONObject():afterUpdateJSONObject, templateRelationItem);
            }
            if(updateRelationsChange != null) {
              updateRelationsChanges.add(updateRelationsChange);
            }
          }
          if(!updateRelationsChanges.isEmpty()) {
            isUpdateItem = true;
            updateChange.put(RELATIONS, updateRelationsChanges);
          }
        }

        // 如果条件成立，说明当前明细条目发生了数据变化
        if(isUpdateItem) {
          datas.add(updateChange);
        }
      }
    }

    if(!datas.isEmpty()) {
      changeObjects.put("datas", datas);
      return changeObjects;
    }
    return null;
  }

  /**
   * 该死有方法用于分析两个指定的关联属性对象（只可能是ManyToOne）的差异，并生成一个结果性质的json数组对象返回
   * @param beforeRelation 进行“WRITE”操作前的关联属性信息的详情
   * @param afterRelation 进行“WRITE”操作后的关联属性信息的详情
   * @param templateRelation
   */
  private JSONObject analysisManyToOneRelation(JSONObject beforeRelation , JSONObject afterRelation , TemplateRelationEntity templateRelation) {
    String propertyClassName = templateRelation.getPropertyClassName();
    String propertyName = templateRelation.getPropertyName();
    String propertyDbName = templateRelation.getPropertyDbName();

    /*
     * 处理过程为：
     * 如果当前关联关系是ManyToOne关系：
     * 这种情况下，只有“修改”这一种情况，直接验证并保存即可
     *
     * 分析后的结果格式可参见analysisManyToManyRelation私有方法
     * */
    String beforeRelationValue = beforeRelation.getString("id");
    String afterRelationValue = afterRelation.getString("id");
    if(!StringUtils.equals(beforeRelationValue, afterRelationValue)) {
      JSONObject changeObject = new JSONObject();
      changeObject.put(PROPERTY_CLASS_NAME, propertyClassName);
      changeObject.put(PROPERTY_NAME, propertyName);
      changeObject.put("propertyDbName", propertyDbName);
      changeObject.put(CHANGE_TYPE, UPDATE);
      changeObject.put(BEFORE_VALUE, beforeRelationValue);
      changeObject.put(AFTER_VALUE, afterRelationValue);
      return changeObject;
    }

    return null;
  }

  /**
   * 该死有方法用于分析两个指定的关联属性对象（只可能是ManyToMany）的差异，并生成一个结果性质的json数组对象返回
   * @param beforeRelation 进行“WRITE”操作前的关联属性信息的详情
   * @param afterRelation 进行“WRITE”操作后的关联属性信息的详情
   * @param templateRelation
   */
  private JSONObject analysisManyToManyRelation(JSONArray beforeRelation , JSONArray afterRelation , TemplateRelationEntity templateRelation) {
    String propertyName = templateRelation.getPropertyName();
    String propertyClassName = templateRelation.getPropertyClassName();

    /*
     * 处理过程为：
     * 如果当前关系是ManyToMany关系：
     * 则只需要记录之前所有的id到beforeValue数组
     * 记录之后的id到afterValue数组
     *
     * 分析后的结果格式类似如下：
     * "relations":[{
     *   // 属性名
     *   "propertyName":"Z",
     *   // 可能的数据字段名
     *   "propertyDbName":"Z",
     *   // 只可能是“UPDATE”
     *   "changeType":UPDATE,
     *   // 变化前的值，可能没有
     *   "beforeValue":"XXXXXX",
     *   // 变化后的值，可能没有
     *   AFTER_VALUE:"YYYYYYY"
     * },{
     *   // 属性名
     *   "propertyName":"Z",
     *   // 关联的类型，只可能是UPDATE
     *   "changeType":UPDATE,
     *   "beforeValue":["xxx1","xxxx2","xxxx3","xxxx4"],
     *   AFTER_VALUE:["yyyy1","yyyy2","yyyy3","xxxx3"]
     * }]
     * */
    List<?> beforeRelationIds = null;
    List<?> afterRelationIds = null;
    if(beforeRelation != null) {
      beforeRelationIds = (List<?>)JSONPath.eval(beforeRelation, StringUtils.join("$..id"));
      if(beforeRelationIds == null) {
        beforeRelationIds = Lists.newArrayList();
      }
    }
    if(afterRelation != null) {
      afterRelationIds = (List<?>)JSONPath.eval(afterRelation, StringUtils.join("$..id"));
      if(afterRelationIds == null) {
        afterRelationIds = Lists.newArrayList();
      }
    }
    // 比较两个集合是否不一样(使用google的Sets工具)
    Set<?> beforeRelationIdSets = Sets.newLinkedHashSet(beforeRelationIds);
    Set<?> afterRelationIdSets = Sets.newLinkedHashSet(afterRelationIds);
    // 如果条件成立，说明两个集合没有变化
    if(Sets.difference(beforeRelationIdSets, afterRelationIdSets).isEmpty() &&
        Sets.difference(afterRelationIdSets , beforeRelationIdSets).isEmpty()) {
      return null;
    }
    // 开始构造变化项描述
    JSONObject valueObject = new JSONObject();
    valueObject.put(PROPERTY_CLASS_NAME, propertyClassName);
    valueObject.put(PROPERTY_NAME, propertyName);
    valueObject.put(CHANGE_TYPE, UPDATE);
    if(beforeRelationIds != null && !beforeRelationIds.isEmpty()) {
      valueObject.put(BEFORE_VALUE, beforeRelationIds);
    }
    if(afterRelationIds != null && !afterRelationIds.isEmpty()) {
      valueObject.put(AFTER_VALUE, afterRelationIds);
    }
    return valueObject;
  }

  /**
   * 该私有方法用于分析两个指定的一般属性对象的差异，并生成一个结果性质的json数组对象返回。
   * 返回的每一个对象结构，参见类定义：XXXX
   * @param beforeProperties 进行“WRITE”操作前的一般属性信息的详情
   * @param afterProperties 进行“WRITE”操作后的一般属性信息的详情
   * @param templateProperties 当前描述的模板信息基本信息
   */
  private JSONArray analysisProperties(JSONObject beforeProperties , JSONObject afterProperties , Set<TemplatePropertyEntity> templateProperties) {
    /*
     * 处理过程：
     * 1、取值过程，由于表单引擎只支持有限的数据类型，所以只需要按照这些类型分别取beforeProperties和afterProperties中的值
     * 2、比较过程，有分为：
     *   如果beforeProperties和afterProperties中都有值，且值没有变化，则认为该属性没有修改
     *   如果beforeProperties和afterProperties中的值不一致，则认为该值被修改。这就包括了beforeProperties有值/没值又或者afterProperties有值/没值的情况
     * 注意：一般属性的比较结果中，没有“删除”、“新增”这样的说法，只有“修改”的定义
     *
     * 组装的json格式类似如下：
     * "properties":[{
     *   // 属性类型
     *   "propertyClassName":"java.lang.String",
     *   // 属性名
     *   PROPERTY_NAME:"X",
     *   // 可能的数据字段名
     *   "propertyDbName":"X",
     *   // 变化类型，包括三种“UPDATA”、“NEW”和“DELETE”
     *   // 如果当前属性为一般属性或者选择性关联属性，则只可能有一种变化类型“UPDATA”
     *   "changeType":UPDATE,
     *   // 变化前的值，可能没有
     *   "beforeValue":"false",
     *   // 变化后的值，可能没有
     *   AFTER_VALUE:"true"
     * },{
     *   "propertyClassName":"java.lang.String",
     *   PROPERTY_NAME:"Y",
     *   "propertyDbName":"Y","changeType":UPDATE,
     *   "beforeValue":"XXXXX",
     *   AFTER_VALUE:"YYYYY"
     * }]
     * */

    JSONArray changeResults = new JSONArray();
    for (TemplatePropertyEntity templateProperty : templateProperties) {
      String propertyClassName = templateProperty.getPropertyClassName();
      String propertyName = templateProperty.getPropertyName();
      String propertyDbName = templateProperty.getPropertyDbName();

      boolean isChange = false;
      Object beforeValue = null;
      Object afterValue = null;
      if(StringUtils.equals(propertyClassName, "java.lang.String")) {
        beforeValue = beforeProperties.getString(propertyName);
        afterValue = afterProperties.getString(propertyName);
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        if(StringUtils.equals(beforeValue == null?"":beforeValue.toString(), afterValue == null?"":afterValue.toString())) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.lang.Long") || StringUtils.equals(propertyClassName, "long")) {
        beforeValue = beforeProperties.getLong(propertyName);
        afterValue = afterProperties.getLong(propertyName);
        long beforeLongValue = beforeValue == null?0l:(long)beforeValue;
        long afterLongValue = afterValue == null?0l:(long)afterValue;
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        if(beforeLongValue == afterLongValue || (beforeValue == null && afterValue == null)) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.lang.Integer") || StringUtils.equals(propertyClassName, "int")) {
        beforeValue = beforeProperties.getInteger(propertyName);
        afterValue = afterProperties.getInteger(propertyName);
        int beforeIntValue = beforeValue == null?0:(int)beforeValue;
        int afterIntValue = afterValue == null?0:(int)afterValue;
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        if(beforeIntValue == afterIntValue || (beforeValue == null && afterValue == null)) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.lang.Short") || StringUtils.equals(propertyClassName, "short")) {
        beforeValue = beforeProperties.getShort(propertyName);
        afterValue = afterProperties.getShort(propertyName);
        int beforeShortValue = beforeValue == null?0:(short)beforeValue;
        int afterShortValue = afterValue == null?0:(short)afterValue;
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        if(beforeShortValue == afterShortValue || (beforeValue == null && afterValue == null)) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.lang.Byte") || StringUtils.equals(propertyClassName, "byte")) {
        beforeValue = beforeProperties.getByte(propertyName);
        afterValue = afterProperties.getByte(propertyName);
        byte beforeByteValue = beforeValue == null?0:(byte)beforeValue;
        byte afterByteValue = afterValue == null?0:(byte)afterValue;
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        if(beforeByteValue == afterByteValue || (beforeValue == null && afterValue == null)) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.lang.Boolean") || StringUtils.equals(propertyClassName, "boolean")) {
        beforeValue = beforeProperties.getBoolean(propertyName);
        afterValue = afterProperties.getBoolean(propertyName);
        boolean beforeBooleanValue = beforeValue == null?false:(boolean)beforeValue;
        boolean afterBooleanValue = afterValue == null?false:(boolean)afterValue;
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        if(beforeBooleanValue == afterBooleanValue || (beforeValue == null && afterValue == null)) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.lang.Float") || StringUtils.equals(propertyClassName, "float")) {
        beforeValue = beforeProperties.getFloat(propertyName);
        afterValue = afterProperties.getFloat(propertyName);
        float beforeFloatValue = beforeValue == null?0.0f:(float)beforeValue;
        float afterFloatValue = afterValue == null?0.0f:(float)afterValue;
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        if(beforeFloatValue == afterFloatValue || (beforeValue == null && afterValue == null)) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.lang.Double") || StringUtils.equals(propertyClassName, "double")) {
        beforeValue = beforeProperties.getDouble(propertyName);
        afterValue = afterProperties.getDouble(propertyName);
        double beforeDoubleValue = beforeValue == null?0.0:(double)beforeValue;
        double afterDoubleValue = afterValue == null?0.0:(double)afterValue;
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        if(beforeDoubleValue == afterDoubleValue || (beforeValue == null && afterValue == null)) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.math.BigDecimal")) {
        beforeValue = beforeProperties.getBigDecimal(propertyName);
        afterValue = afterProperties.getBigDecimal(propertyName);
        float beforeBigDecimalValue = beforeValue == null?BigDecimal.ZERO.floatValue():((BigDecimal)beforeValue).floatValue();
        float afterBigDecimalValue = afterValue == null?BigDecimal.ZERO.floatValue():((BigDecimal)afterValue).floatValue();
        if((beforeValue == null && afterValue != null) || (beforeValue != null && afterValue == null)) {
          isChange = true;
        }
        // 只有以下情况成立，才说明没有变化
        else if(beforeBigDecimalValue == afterBigDecimalValue || (beforeValue == null && afterValue == null)) {
          continue;
        } else {
          isChange = true;
        }
      } else if(StringUtils.equals(propertyClassName, "java.util.Date")) {
        beforeValue = beforeProperties.getDate(propertyName);
        afterValue = afterProperties.getDate(propertyName);
        long beforeDateValue = beforeValue == null?0l:((Date)beforeValue).getTime();
        long afterDateValue = afterValue == null?0l:((Date)afterValue).getTime();
        // 只有以下情况成立，才说明没有变化
        if(beforeDateValue == afterDateValue || (beforeValue == null && afterValue == null)) {
          continue;
        }
        isChange = true;
      }

      // 如果条件成立，说明该字段有变化
      JSONObject changeObject = new JSONObject();
      if(isChange) {
        changeObject.put(PROPERTY_CLASS_NAME, propertyClassName);
        changeObject.put(PROPERTY_NAME, propertyName);
        changeObject.put("propertyDbName", propertyDbName);
        changeObject.put(CHANGE_TYPE, UPDATE);
        if(beforeValue != null) {
          changeObject.put(BEFORE_VALUE, beforeValue);
        } 
        if(afterValue != null) {
          changeObject.put(AFTER_VALUE, afterValue);
        }
        changeResults.add(changeObject);
      }
    }
    return changeResults;
  }
}
