package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.kuiper.entity.PageEntity;
import com.bizunited.platform.kuiper.entity.PageEventEntity;
import com.bizunited.platform.kuiper.entity.PageFlowEntity;
import com.bizunited.platform.kuiper.starter.repository.PageRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.PageEventService;
import com.bizunited.platform.kuiper.starter.service.PageService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 页面流页面数据服务的接口实现
 * @Author: TanBoQiuYun
 * @Date: 2019/12/31 17:32
 */
@Service("PageServiceImpl")
public class PageServiceImpl implements PageService {

  private static final String JSON_FILE_SUFFIX = "json";
  private static final String SAVE_LOCATION = "/page/json/";

  @Autowired
  private PageRepository pageRepository;
  @Autowired
  private PageEventService pageEventService;
  @Autowired
  private VenusFileService kuiperFileService;
  @Autowired
  private KuiperToolkitService kuiperToolkitService;
  @Autowired
  private PlatformContext platformContext;


  /**
   * 根据页面编码比较，处理可能的更新、新增、删除的情况
   * @param pageFlow
   * @param pages
   * @return
   */
  @Override
  @Transactional
  public Set<PageEntity> save(PageFlowEntity pageFlow, Set<PageEntity> pages) {
    //入参校验
    this.saveValidation(pageFlow, pages);
    //原来的页面信息集合
    Set<PageEntity> oldPages = this.pageRepository.findDetailsByPageFlowId(pageFlow.getId());
    if (CollectionUtils.isEmpty(pages)){
      this.delete(oldPages);
      return Sets.newHashSet();
    }
    //可能的新增、修改、删除
    Set<PageEntity> needInsertPages = new HashSet<>();
    Set<PageEntity> needUpdatePages = new HashSet<>();
    Set<PageEntity> needDeletePages = new HashSet<>();
    kuiperToolkitService.collectionDiscrepancy(pages, oldPages, PageEntity::getCode, needDeletePages, needUpdatePages, needInsertPages);
    this.create(pageFlow, needInsertPages);
    this.update(pages, needUpdatePages);
    this.delete(needDeletePages);
    needInsertPages.addAll(needUpdatePages);
    return needInsertPages;
  }

  /**
   * 删除页面以及页面事件
   * @param pages
   */
  private void delete(Set<PageEntity> pages) {
    if (CollectionUtils.isEmpty(pages)){
      return;
    }
    for (PageEntity page : pages) {
      pageEventService.deleteByPageId(page.getId());
      kuiperFileService.deleteFile(page.getRelativePath(), page.getFileName(), page.getFileName());
    }
    pageRepository.deleteAll(pages);
  }

  /**
   * 批量新增页面
   * @param pageFlow
   * @param needInsertPages
   */
  private void create(PageFlowEntity pageFlow, Set<PageEntity> needInsertPages) {
    //遍历新增
    needInsertPages.forEach(page -> {
      page.setPageFlow(pageFlow);
      //保存内容到文件
      this.saveFile(page);
      page.setProjectName(platformContext.getAppName());
      pageRepository.save(page);
      pageEventService.save(page, page.getEvents());
    });
  }

  /**
   * 批量更新页面
   * @param pages  前端传过来的
   * @param needUpdatePages 原始的需要更新的集合
   */
  private void update(Set<PageEntity> pages, Set<PageEntity> needUpdatePages) {
    //更新
    Map<String, PageEntity> newUpdatePageMap = pages.stream().collect(Collectors.toMap(PageEntity::getCode, e -> e));
    if (CollectionUtils.isEmpty(newUpdatePageMap)){
      return;
    }
    needUpdatePages.forEach(page -> {
      this.loadDetails(page);
      //前端传的页面
      PageEntity updatePage = newUpdatePageMap.get(page.getCode());
      //更新页面文件内容
      if (!page.getPageContent().equals(updatePage.getPageContent())){
        page.setPageContent(updatePage.getPageContent());
        byte[] pageContent = updatePage.getPageContent().getBytes(StandardCharsets.UTF_8);
        String fileName = page.getFileName();
        kuiperFileService.saveFile(page.getRelativePath(), fileName, fileName, pageContent);
      }
      page.setName(updatePage.getName());
      pageRepository.save(page);
      pageEventService.save(page, updatePage.getEvents());
    });
  }

  /**
   * 保存页面内容到文件系统中
   * @param page
   */
  private void saveFile(PageEntity page) {
    String folderName = new SimpleDateFormat("yyyyMMdd").format(new Date());
    String renameImage = UUID.randomUUID().toString();
    String fileRename = renameImage + "." + JSON_FILE_SUFFIX;
    String relativePath = StringUtils.join(SAVE_LOCATION , folderName , "/" , (new Random().nextInt(100) % 10));
    page.setFileName(fileRename);
    page.setRelativePath(relativePath);
    byte[] pageContent = page.getPageContent().getBytes(StandardCharsets.UTF_8);
    kuiperFileService.saveFile(relativePath, fileRename, fileRename, pageContent);
  }


  @Override
  public PageEntity findDetailsById(String id) {
    if (StringUtils.isBlank(id)){
      return null;
    }
    PageEntity page = pageRepository.findDetailsById(id);
    return this.loadDetails(page);
  }

  @Override
  public PageEntity findDetailsByCodeAndPageFlowCode(String code, String pageFlowCode) {
    if (StringUtils.isAnyBlank(code, pageFlowCode)){
      return null;
    }
    PageEntity page = pageRepository.findDetailsByCodeAndPageFlowCode(code, pageFlowCode);
    return this.loadDetails(page);
  }

  @Override
  public Set<PageEntity> findDetailsByPageFlowCode(String pageFlowCode) {
    if (StringUtils.isBlank(pageFlowCode)){
      return Sets.newHashSet();
    }
    Set<PageEntity> pages = pageRepository.findDetailsByPageFlowCode(pageFlowCode);
    if (!CollectionUtils.isEmpty(pages)){
      pages.forEach(page ->  this.loadDetails(page));
    }
    return pages;
  }

  @Override
  public Set<PageEntity> findDetailsByPageFlowId(String pageFlowId) {
    if (StringUtils.isBlank(pageFlowId)){
      return Sets.newHashSet();
    }
    Set<PageEntity> pages = pageRepository.findDetailsByPageFlowId(pageFlowId);
    if (!CollectionUtils.isEmpty(pages)){
      pages.forEach(page ->  this.loadDetails(page));
    }
    return pages;
  }

  @Override
  public Set<PageEntity> findByPageFlowCode(String pageFlowCode) {
    if (StringUtils.isBlank(pageFlowCode)){
      return Sets.newHashSet();
    }
    return pageRepository.findByPageFlowCode(pageFlowCode);
  }

  @Override
  @Transactional
  public void deleteByPageFlowId(String pageFlowId) {
    Validate.notBlank(pageFlowId, "页面流ID不能为空！");
    Set<PageEntity> pages = pageRepository.findDetailsByPageFlowId(pageFlowId);
    //先删除页面对应的页面事件
    if (!CollectionUtils.isEmpty(pages)){
      pages.forEach(page -> {
        kuiperFileService.deleteFile(page.getRelativePath(), page.getFileName(), page.getFileName());
        pageEventService.deleteByPageId(page.getId());
      });
      //再删除页面流对应的所有页面
      pageRepository.deleteByPageFlowId(pageFlowId);
    }
  }

  /**
   * 页面-加载页面流页面文件内容
   * @param page
   * @return
   */
  private PageEntity loadDetails(PageEntity page) {
    if (page == null){
      return null;
    }
    //加载页面事件内容
    Set<PageEventEntity> pageEvents = pageEventService.findDetailsByPageId(page.getId());
    page.setEvents(pageEvents);
    byte[] bytes = kuiperFileService.readFileContent(page.getRelativePath(), page.getFileName());
    if (null != bytes){
      //将数组中的每一个元素转换成对应的char再组合成string
      String content = new String(bytes, StandardCharsets.UTF_8);
      page.setPageContent(content);
    }
    return page;
  }

  /**
   * 保存前数据校验
   * @param pageFlow
   * @param pages
   */
  private void saveValidation(PageFlowEntity pageFlow, Set<PageEntity> pages) {
    Validate.notNull(pageFlow, "页面流对象不存在!");
    Validate.notBlank(pageFlow.getId(), "页面流对象ID不能为空!");
    if (CollectionUtils.isEmpty(pages)){
      return;
    }
    Set<String> pageCodes = new HashSet<>();
    pages.forEach(pageEntity -> {
      Validate.notBlank(pageEntity.getName(), "页面流页面名称不能为空!");
      Validate.notBlank(pageEntity.getCode(), "页面流页面编码不能为空!");
      Validate.notBlank(pageEntity.getPageContent(), "保存的页面流页面文件内容不能为空!");
      //验证Code重复性
      Validate.isTrue(!pageCodes.contains(pageEntity.getCode()), "页面流页面编码重复，请替换!");
      pageCodes.add(pageEntity.getCode());
    });
  }
}
