package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.core.service.NebulaStaticPersistentService;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplateMaintainerEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.service.InstanceService;
import com.bizunited.platform.kuiper.service.StaticTemplateService;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.common.enums.FormTemplateTypeEnum;
import com.bizunited.platform.kuiper.starter.model.StaticTemplateMergeModel;
import com.bizunited.platform.kuiper.starter.repository.InstanceRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplatePropertyRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRelationRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRepository;
import com.bizunited.platform.kuiper.starter.service.StaticTemplateMergeService;
import com.bizunited.platform.kuiper.starter.service.TemplateEventService;
import com.bizunited.platform.kuiper.starter.service.TemplateGroupService;
import com.bizunited.platform.kuiper.starter.service.TemplateItemService;
import com.bizunited.platform.kuiper.starter.service.TemplateMaintainerService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService;
import com.bizunited.platform.saturn.model.PersistentClass;
import com.bizunited.platform.saturn.model.PersistentProperty;
import com.bizunited.platform.saturn.model.PersistentRelation;
import com.bizunited.platform.saturn.model.PersistentRelation.RelationType;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author yinwenjie
 */
@Service("StaticTemplateServiceImpl")
public class StaticTemplateServiceImpl extends AbstractTemplateService implements StaticTemplateService {
  @Autowired
  private NebulaStaticPersistentService nebulaStaticPersistentService;
  @Autowired
  private TemplateRepository templateRepository;
  @Autowired
  private TemplatePropertyRepository templatePropertyRepository;
  @Autowired
  private TemplateRelationRepository templateRelationRepository;
  @Autowired
  private TemplateMaintainerService templateMaintainerService;
  @Autowired
  private TemplateEventService templateEventService;
  @Autowired
  private TemplateVisibilityService templateVisibilityService;
  @Autowired
  private TemplateItemService templateItemService;
  @Autowired
  private TemplateGroupService templateGroupService;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private InstanceRepository instanceRepository;
  @Autowired
  private InstanceService instanceService;
  @Autowired
  private UserService userService;
  @Autowired
  private PlatformContext platformContext;
  @Autowired
  private StaticTemplateMergeService staticTemplateMergeService;

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.service.StaticTemplateService#initStaticTemplate(com.bizunited.platform.kuiper.entity.TemplateEntity, java.security.Principal, boolean)
   */
  @Override
  @Transactional
  @CacheEvict(cacheNames="template" , allEntries=true)
  public TemplateEntity initStaticTemplate(TemplateEntity template , Principal principal , boolean initVisibilities) {
    String id = template.getId();
    Validate.isTrue(StringUtils.isBlank(id) , "表单模板初始化时，不能传入表单模板编号!!");
    template.setId(null);
    String cversion = template.getCversion();
    Validate.notBlank(cversion , "新的表单模板的版本号信息必须指定!!");
    String name = template.getName();
    Validate.notBlank(name , "新的表单模板的名称信息必须传入!!");
    String type = template.getType();
    Validate.notBlank(type , "表单模板类型必须传入");
    Validate.isTrue(StringUtils.equals(type, FormTemplateTypeEnum.STATIC.getType()) , "目标表单模板类型只支持（static）静态模板类型!!");
    /*
     * 1.1 版本的初始过程发生了变化，要进行注释的修改和代码的修改
     * 实际上最大的差异就是数据源不在提供“服务源的方式”，另外就是关联信息的描述只保留OneToMany和OneToOne的
     *
     * 初始化过程如下：
     * 1、首先确认其基本入参、类信息没有问题
     * 2、寻找其静态模型基本信息，组装一般属性和直接关联属性
     * 3、然后根据静态模型中描述的OneToMany关系，进行明细信息描述部分的处理（既是TemplateItemEntity信息）
     * 4、接着根据静态模型中描述的OneToOne关系，进行分组信息部分的处理（既是TemplateGroupEntity信息）
     * 5、初始化可见性信息，每一个静态表单模板都有一个不能删除的，名字叫做create的可见性信息
     * 注意，初始化的可见性中只需要由主模型中的属性，以及OneToOne关联性质的属性
     * 
     * 注意：以上各个步骤都需要保存到数据库，形成一条草稿信息（所谓草稿信息就是还不能正式发布进行表单模板实例化操作的表单）
     * */
    // 1、=======
    String persistentClassName = template.getPersistentClassName();
    PersistentClass persistentClass = this.nebulaStaticPersistentService.findByPersistentClass(persistentClassName);
    Validate.notNull(persistentClass , "没有找到指定的静态模型描述，请检查!!");
    // 自定义的code（保证唯一性）
    template.setCode(persistentClass.getClassName());
    TemplateEntity currentTemplate = this.templateRepository.findByCodeAndCversion(template.getCode(), cversion);
    Validate.isTrue(currentTemplate == null , "当前指定的表单模板编号和版本已经存在，请检查!!");
    template.setProjectName(platformContext.getAppName());
    super.saveTemplate(template, principal);
    
    // 2、======
    // 基本信息
    List<PersistentProperty> persistentProperties = persistentClass.getProperties();
    Set<TemplatePropertyEntity> templateProperties = this.initStaticProperties(template, persistentProperties);
    if(!CollectionUtils.isEmpty(templateProperties)) {
      templateProperties.forEach(p -> p.setProjectName(platformContext.getAppName()));
    }
    this.templatePropertyRepository.saveAll(templateProperties);
    template.setProperties(templateProperties);
    // 关联信息只需要关注ManyToMany和ManyToOne
    List<PersistentRelation> persistentRelations = persistentClass.getRelations();
    if(persistentRelations != null && !persistentRelations.isEmpty()) {
      List<PersistentRelation> persistentManyToRelations = persistentRelations.stream().filter(item -> item.getRelationType() == RelationType.ManyToMany || item.getRelationType() == RelationType.ManyToOne).collect(Collectors.toList());
      if(!CollectionUtils.isEmpty(persistentManyToRelations)) {
        Set<TemplateRelationEntity> templateRelations = this.initStaticRelations(template, persistentManyToRelations);
        if(!CollectionUtils.isEmpty(templateRelations)) {
          templateRelations.forEach(t -> t.setProjectName(platformContext.getAppName()));
        }
        this.templateRelationRepository.saveAll(templateRelations);
        template.setRelations(templateRelations);
      }
    }

    // 3、======
    // 这里只需要处理OneToMany关系
    if(persistentRelations != null && !persistentRelations.isEmpty()) {
      List<PersistentRelation> persistentOneToManyRelations = persistentRelations.stream().filter(item -> item.getRelationType() == RelationType.OneToMany).collect(Collectors.toList());
      if(!CollectionUtils.isEmpty(persistentOneToManyRelations)) {
        Set<TemplateItemEntity> items = new HashSet<>();
        for (PersistentRelation persistentOneToManyRelationItem : persistentOneToManyRelations) {
          String oneToManyPersistentClassName = persistentOneToManyRelationItem.getPropertyClass();
          PersistentClass oneToManyPersistentClass = nebulaStaticPersistentService.findByPersistentClass(oneToManyPersistentClassName);
          Validate.notNull(oneToManyPersistentClass , "在初始化主静态模型时发现NULL关联信息，请联系表单引擎开发团队!!");
          TemplateItemEntity currentTemplateItem = this.templateItemService.initStaticItems(template, null, persistentOneToManyRelationItem, oneToManyPersistentClass);
          currentTemplateItem.setProjectName(platformContext.getAppName());
          items.add(currentTemplateItem);
        }
        template.setItemRelations(items);
      }
    }
    
    // 4、======
    // 这里只需要处理OneToOne的关系
    if(persistentRelations != null && !persistentRelations.isEmpty()) {
      List<PersistentRelation> persistentOneToOneRelations = persistentRelations.stream().filter(item -> item.getRelationType() == RelationType.OneToOne).collect(Collectors.toList());
      if(!CollectionUtils.isEmpty(persistentOneToOneRelations)) {
        Set<TemplateGroupEntity> groups = new HashSet<>();
        for (PersistentRelation oneToOneRelationItem : persistentOneToOneRelations) {
          String oneToOnePersistentClassName = oneToOneRelationItem.getPropertyClass();
          PersistentClass oneToOnePersistentClass = nebulaStaticPersistentService.findByPersistentClass(oneToOnePersistentClassName);
          Validate.notNull(oneToOnePersistentClass , "在初始化主静态模型时发现NULL关联信息，请联系表单引擎开发团队!!");
          TemplateGroupEntity currentTemplateGroup = this.templateGroupService.initStaticGroups(template, oneToOneRelationItem , oneToOnePersistentClass);
          currentTemplateGroup.setProjectName(platformContext.getAppName());
          groups.add(currentTemplateGroup);
        }
        template.setGroupRelations(groups);
      }
    }
    
    // 5、=====
    if(initVisibilities) {
      Set<TemplateVisibilityEntity> templateVisibilities = this.templateVisibilityService.initStaticTemplateVisibilities(template.getId(), persistentClass);
      template.setVisibility(templateVisibilities);
    }
    return template;
  }

  /**
   * 根据静态模型的基础属性描述，组装模板对象中的一般属性
   * @param template
   * @param persistentProperties
   * @return
   */
  private Set<TemplatePropertyEntity> initStaticProperties(TemplateEntity template ,List<PersistentProperty> persistentProperties) {
    Set<TemplatePropertyEntity> templateProperties = new LinkedHashSet<>();
    for (PersistentProperty persistentProperty : persistentProperties) {
      TemplatePropertyEntity property = new TemplatePropertyEntity();
      property.setCanInsert(persistentProperty.getCanInsert());
      property.setCanUpdate(persistentProperty.getCanUpdate());
      property.setUnique(persistentProperty.getUnique());
      property.setIndex(persistentProperty.getIndex());
      property.setNullable(persistentProperty.getNullable());
      property.setPrimaryKey(persistentProperty.getPrimaryKey());
      property.setPropertyClassName(persistentProperty.getPropertyClass());
      String propertyDbName = persistentProperty.getPropertyDbName() == null?"":persistentProperty.getPropertyDbName();
      Validate.isTrue(propertyDbName  != null && propertyDbName.length() <= 64 , "在初始化时发现字段的数据库属性描述[%s]，长度超过了64个字符，请检查!!" , propertyDbName);
      property.setPropertyDbName(propertyDbName);
      property.setPropertyDesc(persistentProperty.getPropertyDesc());
      String propertyName = persistentProperty.getPropertyName();
      Validate.isTrue(!StringUtils.isBlank(propertyName) && propertyName.length() <= 64 , "在初始化时发现字段[%s]的长度超过了64个字符，请检查!!" , propertyName);
      property.setPropertyName(propertyName);
      property.setDefaultController(persistentProperty.getDefaultType() != null? persistentProperty.getDefaultType() :"");
      property.setDefaultKeys(persistentProperty.getDefaultKeys());
      property.setDefaultValues(persistentProperty.getDefaultValues());
      property.setMaxLen(persistentProperty.getMaxLen());

      property.setId(null);
      property.setCurrentTemplate(template);
      property.setCurrentGroup(null);
      property.setCurrentItem(null);
      property.setProjectName(platformContext.getAppName());
      templateProperties.add(property);
    }
    return templateProperties;
  }

  /**
   * 根据静态模型的直接关联属性，组装模板中的关联属性——relations部分的信息(也就是ManyToMany关系和ManyToOne关系)
   * @param template
   * @param persistentRelations
   * @return
   */
  private Set<TemplateRelationEntity> initStaticRelations(TemplateEntity template , List<PersistentRelation> persistentRelations) {
    Set<TemplateRelationEntity> templateRelations = new LinkedHashSet<>();
    for (PersistentRelation persistentRelationItem : persistentRelations) {
      TemplateRelationEntity relationModel = new TemplateRelationEntity();
      relationModel.setCanInsert(persistentRelationItem.getCanInsert());
      relationModel.setCanUpdate(persistentRelationItem.getCanUpdate());
      relationModel.setIndex(persistentRelationItem.getIndex());
      relationModel.setNullable(persistentRelationItem.getNullable());
      relationModel.setPropertyClassName(persistentRelationItem.getPropertyClass());
      relationModel.setPropertyDbName(persistentRelationItem.getPropertyDbName() == null ?"":persistentRelationItem.getPropertyDbName());
      relationModel.setPropertyDesc(persistentRelationItem.getPropertyDesc());
      relationModel.setPropertyName(persistentRelationItem.getPropertyName());
      relationModel.setRelationType(persistentRelationItem.getRelationType().toString());

      relationModel.setCurrentView(null);
      relationModel.setId(null);
      relationModel.setTargetTableName("");
      relationModel.setCurrentTemplate(template);
      relationModel.setCurrentGroup(null);
      relationModel.setCurrentItem(null);
      relationModel.setProjectName(platformContext.getAppName());
      templateRelations.add(relationModel);
    }

    return templateRelations;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.service.StaticTemplateService#upgradeStaticTemplate(java.lang.String, java.lang.String, java.security.Principal)
   */
  @Override
  @Transactional
  @CacheEvict(cacheNames="template" , allEntries=true)
  public TemplateEntity upgradeStaticTemplate(String templateId, String newVersion, boolean updateInstance, Principal principal) {
    /*
     * 0、在验证过程完成后 ，自动升级过程开始
     * 1、首先“升级”静态模板基本信息，实际上就是将之前的源模板信息进行拷贝（除了id、code、name信息以外）
     * 2、然后“升级”静态模板的一般属性和关联属性（ManyToMany关联和ManyToOne关联），
     * 实际上就是使用initStaticProperties方法和initStaticRelations方法，重新扫描初始化
     * 3、接着就是“升级”可能存在的分组信息，同样是初始化
     * 4、“升级”可能存在的明细编辑信息，同样是重新初始化
     * 5、进行布局信息的拷贝（注意布局的信息可能包括PC布局、移动端布局和打印端布局）
     * 6、进行事件信息的拷贝
     * 7、进行可见性信息的拷贝
     * */
    // 0、==========
    Validate.notBlank(templateId , "升级时,原始指定的templateId必须传入");
    Validate.notBlank(newVersion , "升级时,新的版本信息必须传入");
    TemplateEntity currentTemplate = this.templateRepository.findById(templateId).orElse(null);
    Validate.notNull(currentTemplate , "未找到指定的模板信息，请检查!!");

    Validate.isTrue(StringUtils.equals(currentTemplate.getType(), FormTemplateTypeEnum.STATIC.getType()) , "该升级方法只适用于“静态表单模板”!!");
    TemplateEntity exsitTemplate = this.templateRepository.findByCodeAndCversion(currentTemplate.getCode(), newVersion);
    Validate.isTrue(exsitTemplate == null , "指定的版本编号信息已经在指定的code[%s]中存在，请重新指定版本编号信息!!" , currentTemplate.getCode());
    Validate.notNull(principal , "必须指定操作者信息!!");
    String account = principal.getName();
    UserVo user = this.userService.findByAccount(account);
    Validate.notNull(user , "未找到任何创建者信息，请检查!!");
    // 检查静态类是否还存在
    String type = currentTemplate.getType();
    Validate.isTrue(StringUtils.equals(type , FormTemplateTypeEnum.STATIC.getType()) , "目前本接口只支持基于静态模型的静态模板升级");
    String persistentClassName = currentTemplate.getPersistentClassName();
    Validate.notBlank(persistentClassName , "未发现指定的静态模型信息，请检查数据库数据!!");
    PersistentClass currentPersistentClass = this.nebulaStaticPersistentService.findByPersistentClass(persistentClassName);
    Validate.notNull(currentPersistentClass , "未发现指定的静态模型定义，请检查[%s]" , currentPersistentClass);
    
    // 1、2、3、4、========== 都在这里
    TemplateEntity newTemplate = new TemplateEntity();
    newTemplate.setCode(currentTemplate.getCode());
    newTemplate.setCreateTime(new Date());
    newTemplate.setCreator(account);
    newTemplate.setCversion(newVersion);
    newTemplate.setDomain(currentTemplate.getDomain());
    newTemplate.setFormStyle(currentTemplate.getFormStyle());
    newTemplate.setName(currentTemplate.getName());
    newTemplate.setPersistentClassName(currentTemplate.getPersistentClassName());
    newTemplate.setProjectName(platformContext.getAppName());
    newTemplate.setRepositoryEntity(currentTemplate.getRepositoryEntity());
    newTemplate.setTstatus(currentTemplate.getTstatus());
    newTemplate.setType(currentTemplate.getType());
    newTemplate.setModifyTime(new Date());
    newTemplate = this.initStaticTemplate(newTemplate, principal, false);
    String newTemplateId = newTemplate.getId();
    
    // 5、========再是布局信息
    super.upgradeTemplateLayout(templateId, newTemplateId, newTemplate);
    // 6、======== 再是事件信息
    super.upgradeTemplateEvent(templateId, newTemplateId, newTemplate);
    // 7、======== 最后是可见性信息
    super.upgradeTemplateVisibility(templateId, newTemplateId, newTemplate);
    // 8、======== 修改为当前最新升级的template为默认的version
    this.templateService.updateDefaultVersion(newTemplateId);

    // 9、======== 查询现有的维护人员，并绑定关系
    this.bindingTemplateMaintainer(templateId,newTemplateId);
    // 10、======= 是否应用于所有表单实例
    if(updateInstance) {
      instanceService.updateTemplate(newTemplate);
    }
    return newTemplate;
  }
  
  @Override
  @Transactional
  @CacheEvict(cacheNames="template" , allEntries=true)
  public void deleteById(String templateId) {
    /*
     * 删除过程为：
     * 1、首先判断当前模板是否有实例，如果有则不允许删除。
     * 2、删除模板下的所有关联信息：
     *   a、删除可见性
     *   b、删除事件
     *   c、删除数据源
     *   d、删除模型
     * 3、最后再删除相关模板信息
     * */
    // 1、========
    Validate.notBlank(templateId , "进行删除时，模板信息编号必须填入!!");
    int insCount = this.instanceRepository.countByTemplateId(templateId);
    Validate.isTrue(insCount == 0 , "当前指定的模板信息已存在实例，不允许被删除!!");
    
    // 2、========
    // 可见性
    this.templateVisibilityService.deleteByTemplateId(templateId);
    // 事件
    this.templateEventService.deleteByTemplateId(templateId);
    
    // 3、=======最后删除模板
    this.templateRepository.deleteByTemplateId(templateId);
  }

  @Override
  @Transactional
  public TemplateEntity merge(String refTemplateId, String templateId, String name, String version, Principal principal) {
    StaticTemplateMergeModel model = new StaticTemplateMergeModel();
    model.setReferenceTemplateId(refTemplateId);
    model.setMergedTemplateId(templateId);
    model.setNewTemplateName(name);
    model.setNewTemplateVersion(version);
    Validate.notNull(principal , "必须指定操作者信息!!");
    model.setAccount(principal.getName());
    return staticTemplateMergeService.merge(model);
  }


  /**
   * 绑定表单维护人员
   */
  private void bindingTemplateMaintainer(String templateId , String newTemplateId){
    Set<TemplateMaintainerEntity> maintainers = templateMaintainerService.findByTemplateId(templateId);
    if(!CollectionUtils.isEmpty(maintainers)) {
      String[] accounts = maintainers.stream().map(TemplateMaintainerEntity::getUserAccount).collect(Collectors.toList()).toArray(new String[maintainers.size()]);
      templateMaintainerService.binding(newTemplateId, accounts);
    }
  }
}