package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateEventEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.repository.TemplateEventRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.TemplateEventService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.RegExUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

@Service("TemplateEventServiceImpl")
public class TemplateEventServiceImpl implements TemplateEventService {

  private static final String SCRIPT_PATH = "/javascript";

  @Autowired
  private TemplateEventRepository templateEventRepository;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private KuiperToolkitService kuiperToolkitService;
  @Autowired
  private PlatformContext platformContext;

  /**
   * 保存控件事件信息
   * @param templateId
   * @param eventEntityVos
   * @return
   */
  @Override
  @Transactional
  public void save(String templateId, List<TemplateEventEntity> eventEntityVos) {
    Validate.notBlank(templateId, "%s不能为空,请检查!!", "模板id");
    // 检查模板id是否存在且是否唯一 , 只有存在且唯一时,才能继续下面的逻辑
    TemplateEntity template = this.templateService.findById(templateId);
    Validate.notNull(template,"模板不存在，请检查!!");

    // 根据模板id查询数据库原始的控件事件数据
    List<TemplateEventEntity> dbTemplateEvents = templateEventRepository.findByTemplateId(templateId);
    Set<String> eventIdSet = new HashSet<>();// 用于判断是否有重复的元素
    Date now = new Date();
    //遍历，同时判断新增或更新情况
    if(!CollectionUtils.isEmpty(eventEntityVos)){
      eventEntityVos.forEach(event -> {
        this.validate(eventIdSet,template,event);
        //根据传入的templateId、controllerId、eventId 判断当前事件是否在该表单中唯一
        int count = templateEventRepository.countByTemplateIdAndEventIdAndControllerId(template.getId(), event.getControllerId() , event.getEventId());
        if (StringUtils.isNotBlank(event.getId())) {
          // 当事件id有值时,更新操作
          Validate.isTrue(count <= 1, "事件名:%s , templateId、controllerId、eventId组合编号重复,请更换!!" , event.getOnEvent() , 0);
        } else {
          // 当事件id无值时,新增操作
          Validate.isTrue(count == 0, "事件名:%s , templateId、controllerId、eventId组合编号重复,请更换!!" , event.getOnEvent() , 0);
          event.setCreateTime(now);
        }
        event.setProjectName(platformContext.getAppName());
        this.saveFile(event,templateId);
        templateEventRepository.save(event);
      });
    }

    //判断并且删除事件
    Set<String> needDeleteKeys = kuiperToolkitService.collectionDiffent(dbTemplateEvents,eventEntityVos, TemplateEventEntity::getId);
    if(!CollectionUtils.isEmpty(needDeleteKeys)){
      needDeleteKeys.forEach(e -> {
        Optional<TemplateEventEntity> op = dbTemplateEvents.stream().filter(x -> StringUtils.equals(e,x.getId())).findFirst();
        TemplateEventEntity eventEntity = op.orElse(null);
        if(eventEntity != null){
          this.deleteFile(eventEntity,templateId);
        }
        templateEventRepository.deleteById(e);
      });
    }
  }


  /**
   * 验证事件实体字段的相关信息
   * @param eventIdSet
   * @param template
   * @param event
   */
  private void validate(Set<String> eventIdSet,TemplateEntity template,TemplateEventEntity event){
    event.setTemplate(template);
    Validate.notBlank(event.getOnEvent(), "事件名不能为空,请检查!!", "控件事件名 ");
    Validate.notBlank(event.getEventId(), "事件名: %s不能为空,请检查!!","控件eventid编号" , event.getOnEvent());
    Validate.isTrue(eventIdSet.add(event.getEventId()),"事件名:%s , %s重复,请检查!!", event.getOnEvent() , "控件eventid编号");
    Validate.notBlank(event.getControllerId(), "事件名：%s , %s不能为空,请检查!!" , event.getOnEvent() ,"控件controllerid编号 ");
    Validate.notBlank(event.getControllerType(), "事件名：%s , %s不能为空,请检查!!" , event.getOnEvent(),"控件类型");
    Validate.notBlank(event.getOnEvent(), "事件名:%s, %s不能为空,请检查!!" , event.getOnEvent() ," 控件事件名");
    Validate.notBlank(event.getAttributeName(), "事件名:%s, %s不能为空,请检查!!" , event.getOnEvent(),"模板属性名");
    Validate.notNull(event.getEventType(), "事件名:%s , %s 不能为空,请检查!!" , event.getOnEvent(),"控件事件类型");
    Validate.notBlank(event.getEventDesc(), "事件名:%s , %s 不能为空,请检查!!" , event.getOnEvent() ,"控件事件描述");
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateEventService#upgrade(java.lang.String, java.util.List)
   */
  @Override
  public Set<TemplateEventEntity> upgrade(String templateId, List<TemplateEventEntity> templateEvents) {
    Validate.notBlank(templateId , "升级时,原始指定的templateId必须传入");
    TemplateEntity template = this.templateService.findById(templateId);
    Validate.notNull(template , "未找到指定的模板信息，请检查!!");
    if(templateEvents == null || templateEvents.isEmpty()) {
      return Sets.newHashSet();
    }

    Set<TemplateEventEntity> newTemplateEvents = new LinkedHashSet<>();
    for (TemplateEventEntity templateEvent : templateEvents) {
      TemplateEventEntity newTemplateEvent = new TemplateEventEntity();
      newTemplateEvent.setAttributeName(templateEvent.getAttributeName());
      newTemplateEvent.setControllerId(templateEvent.getControllerId());
      newTemplateEvent.setControllerType(templateEvent.getControllerType());
      newTemplateEvent.setCreateTime(new Date());
      newTemplateEvent.setEventDesc(templateEvent.getEventDesc());
      newTemplateEvent.setEventId(templateEvent.getEventId());
      newTemplateEvent.setEventType(templateEvent.getEventType());
      newTemplateEvent.setOnEvent(templateEvent.getOnEvent());
      newTemplateEvent.setServiceName(templateEvent.getServiceName());
      newTemplateEvent.setServiceParams(templateEvent.getServiceParams());
      newTemplateEvent.setJavascriptDesc(templateEvent.getJavascriptDesc());
      newTemplateEvent.setTemplate(template);
      newTemplateEvent.setProjectName(platformContext.getAppName());
      newTemplateEvents.add(newTemplateEvent);
      this.saveFile(newTemplateEvent,templateId);
    }
    this.templateEventRepository.saveAll(newTemplateEvents);
    return newTemplateEvents;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateEventService#deleteByTemplateId(java.lang.String)
   */
  @Override
  @Transactional
  public void deleteByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) {
      return;
    }

    this.templateEventRepository.deleteByTemplateId(templateId);
  }

  /**
   * 根据指定事件id,查询事件详情信息"
   */
  @Override
  public TemplateEventEntity findDetailById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    TemplateEventEntity event = templateEventRepository.findDetailById(id);
    this.readFile(event);
    return event;
  }

  /**
   * 根据模板id查询事件数据(不带fetch)
   * @param templateId
   * @return
   */
  @Override
  public List<TemplateEventEntity> findByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) {
      return Lists.newArrayList();
    }
    List<TemplateEventEntity> events = templateEventRepository.findByTemplateId(templateId);
    this.readFiles(events);
    return events;
  }


  /**
   * 根据指定模板id,查询所有事件详情信息
   * @param templateId
   * @return
   */
  @Override
  public Set<TemplateEventEntity> findDetailsByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) {
      return Sets.newHashSet();
    }
    Set<TemplateEventEntity> events = templateEventRepository.findDetailsByTemplateId(templateId);
    this.readFiles(events);
    return events;
  }

  /**
   * 根据提供的事件信息，保存相应的js脚本文件
   * @param event
   */
  private void saveFile(TemplateEventEntity event , String templateId) {
    if(event == null) {
      return;
    }
    String relativePath = StringUtils.join(SCRIPT_PATH , "/" , RegExUtils.replaceAll(templateId, "-", ""),"/");
    Validate.notBlank(event.getOnEvent(), "事件名:%s , %s不能为空,请检查!" , event.getOnEvent() ,"控件事件名");
    Validate.notBlank(event.getControllerId(), "事件名:%s , %s不能为空,请检查!" , event.getOnEvent() ,"控件controllerid编号");
    byte[] fileContext = event.getJavascriptDesc().getBytes(StandardCharsets.UTF_8);
    String fileRename = StringUtils.join(event.getOnEvent(),".txt");
    String controllerId = RegExUtils.replaceAll(event.getControllerId() ,"-","");
    venusFileService.saveFile(relativePath + controllerId ,fileRename, fileRename,fileContext);
    event.setFileName(fileRename);
    event.setRelativePath(relativePath + controllerId);
  }

  /**
   * 根据提供的事件信息，删除相应的js脚本文件
   * @param event
   * @param templateId
   */
  private void deleteFile(TemplateEventEntity event , String templateId) {
    if(event == null) {
      return;
    }
    String relativePath = StringUtils.join(SCRIPT_PATH , "/" , RegExUtils.replaceAll(templateId, "-", ""),"/");
    Validate.notBlank(event.getOnEvent(), "事件名:%s , %s不能为空,请检查!!" , event.getOnEvent() ,"控件事件名");
    Validate.notBlank(event.getControllerId(), "事件名:%s , %s不能为空,请检查!!" , event.getOnEvent() ,"控件controllerid编号");
    String fileRename = StringUtils.join(event.getOnEvent(),".txt");
    String controllerId = RegExUtils.replaceAll(event.getControllerId(), "-","");
    venusFileService.deleteFile(relativePath + controllerId, fileRename, fileRename);
  }

  /**
   * 根据事件信息，读取事件的脚本
   * @param event
   */
  private void readFile(TemplateEventEntity event) {
    if(event == null) {
      return;
    }
    if(StringUtils.isNotBlank(event.getRelativePath()) && StringUtils.isNotBlank(event.getFileName())) {
      byte[] contexts = venusFileService.readFileContent(event.getRelativePath(), event.getFileName());
      if(contexts != null ) {
        event.setJavascriptDesc(new String(contexts, StandardCharsets.UTF_8));
      }
    }
  }

  /**
   * 批量读取事件脚本
   * @param events
   */
  private void readFiles(Collection<TemplateEventEntity> events) {
    if(CollectionUtils.isEmpty(events)) {
      return;
    }
    for(TemplateEventEntity e : events) {
      this.readFile(e);
    }
  }

}
