package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.starter.repository.TemplateItemRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplatePropertyRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRelationRepository;
import com.bizunited.platform.kuiper.starter.service.TemplateItemService;
import com.bizunited.platform.saturn.model.PersistentClass;
import com.bizunited.platform.saturn.model.PersistentProperty;
import com.bizunited.platform.saturn.model.PersistentRelation;
import com.bizunited.platform.saturn.model.PersistentRelation.RelationType;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service("TemplateItemServiceImpl")
public class TemplateItemServiceImpl extends AbstractTemplateService implements TemplateItemService {
  @Autowired
  private TemplateItemRepository templateItemRepository;
  @Autowired
  private TemplatePropertyRepository templatePropertyRepository;
  @Autowired
  private TemplateRelationRepository templateRelationRepository;
  @Autowired
  private PlatformContext platformContext;
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateItemService#initStaticItems(com.bizunited.platform.kuiper.entity.TemplateEntity, com.bizunited.platform.kuiper.entity.TemplateGroupEntity, com.bizunited.platform.saturn.model.PersistentRelation, com.bizunited.platform.saturn.model.PersistentClass)
   */
  @Override
  @Transactional
  public TemplateItemEntity initStaticItems(TemplateEntity parentTemplate , TemplateGroupEntity parentGroup , PersistentRelation parentRelation , PersistentClass persistentClass) {
    String parentClassName = parentRelation.getPersistentClassName();
    // 基于上层是主表单模型关联，进行的明细创建
    TemplateItemEntity templateItem = new TemplateItemEntity();
    templateItem.setPropertyClassName(persistentClass.getClassName());
    // 明细信息项的属性名称
    String propertyName = parentRelation.getPropertyName();
    Validate.notBlank(propertyName , "明细项属性不能为空，请检查!!");
    Validate.isTrue(propertyName.length() <= 64 , "属性名[%s]的长度需要小于64个字符，请检查!!" , propertyName);
    
    templateItem.setParentClassName(parentClassName);
    templateItem.setPkage(persistentClass.getPkage());
    templateItem.setRepositoryEntity(persistentClass.getRepositoryEntity());
    templateItem.setSimpleClassName(persistentClass.getSimpleClassName());
    templateItem.setType("static");
    // 和动态模型相关的信息设置为空字符串
    templateItem.setParentTableName("");
    templateItem.setTableName("");
    templateItem.setParentTemplate(parentTemplate);
    templateItem.setParentGroup(parentGroup);
    templateItem.setId(null);
    templateItem.setParentTableName("");
    templateItem.setQueryService("");
    // 字段属性和字段中文描述
    templateItem.setPropertyDesc(parentRelation.getPropertyDesc());
    templateItem.setPropertyName(propertyName);
    templateItem.setProjectName(platformContext.getAppName());
    this.templateItemRepository.save(templateItem);
    
    // 明细中的基本属性
    Set<TemplatePropertyEntity> templateProperties = this.initStaticProperties(persistentClass.getProperties());
    templateItem.setProperties(templateProperties);
    if(!templateProperties.isEmpty()) {
      templateProperties.stream().forEach(item -> {
        item.setCurrentItem(templateItem);
        item.setProjectName(platformContext.getAppName());
      });
      templatePropertyRepository.saveAll(templateProperties);
      templateItem.setProperties(templateProperties);
    }
    
    // 直接关联属性(只需要处理ManyToMany关系和ManyToOne关系)
    List<PersistentRelation> relations = persistentClass.getRelations();
    if(relations == null || relations.isEmpty()) {
      return templateItem;
    }
    List<PersistentRelation> currentRelations = relations.stream()
        .filter(item -> (item.getRelationType() == RelationType.ManyToOne || item.getRelationType() == RelationType.ManyToMany)).collect(Collectors.toList());
    Set<TemplateRelationEntity> templateRelations = this.initStaticRelations(parentClassName, currentRelations);
    templateItem.setRelations(templateRelations);
    if(!templateRelations.isEmpty()) {
      templateRelations.stream().forEach(item -> {
        item.setCurrentItem(templateItem);
        item.setProjectName(platformContext.getAppName());
      });
      templateRelationRepository.saveAll(templateRelations);
      templateItem.setRelations(templateRelations);
    }
    return templateItem;
  }
  
  /**
   * 根据模型的明细关联属性，组装模板明细信息中的关联属性——relations部分的信息
   * @param persistentRelations
   * @return 
   */
  private Set<TemplateRelationEntity> initStaticRelations(String parentClassName, List<PersistentRelation> persistentRelations) {
    Set<TemplateRelationEntity> templateRelations = new LinkedHashSet<>();
    for (PersistentRelation persistentRelationItem : persistentRelations) {
      TemplateRelationEntity relationModel = super.initTemplateRelationEntity(persistentRelationItem, parentClassName);
      relationModel.setProjectName(platformContext.getAppName());
      templateRelations.add(relationModel);
    }
    return templateRelations;
  }
  
  /**
   * 根据模型的明细关联属性，组装模板明细信息中的一般属性
   * @param persistentProperties
   * @return
   */
  private Set<TemplatePropertyEntity> initStaticProperties(List<PersistentProperty> persistentProperties) {
    Set<TemplatePropertyEntity> templateProperties = new LinkedHashSet<>();
    for (PersistentProperty persistentProperty : persistentProperties) {
      TemplatePropertyEntity propertyEntity = super.initTemplatePropertyEntity(persistentProperty);
      propertyEntity.setProjectName(platformContext.getAppName());
      templateProperties.add(propertyEntity);
    }
    return templateProperties;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateItemService#initDynamicItems(com.bizunited.platform.kuiper.entity.TemplateItemEntity, com.bizunited.platform.kuiper.entity.TemplateEntity, com.bizunited.platform.kuiper.entity.TemplateGroupEntity, java.lang.String)
   */
  @Override
  @Transactional
  public TemplateItemEntity initDynamicItems(TemplateItemEntity templateItem , TemplateEntity parentTemplate, TemplateGroupEntity parentGroup, String parentTableName) {
    // 首先是这个分组表自身的信息
    templateItem.setId(null);
    templateItem.setParentGroup(parentGroup);
    templateItem.setParentTemplate(parentTemplate);
    templateItem.setParentTableName(parentTableName);
    // 然后将涉及到静态模板的属性设置为null
    templateItem.setPkage("");
    templateItem.setQueryService("");
    templateItem.setParentClassName("");
    templateItem.setSimpleClassName("");
    templateItem.setProjectName(platformContext.getAppName());
    this.templateItemRepository.save(templateItem);
    
    // 然后是明细中的一般属性
    this.initDynamicProperties(templateItem);
    
    // 接着是这个明细中的关联属性（ManyToMany和ManyToOne两种）
    this.initDynamicRelations(templateItem);
    return templateItem;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateItemService#findDetailsById(java.lang.String)
   */
  @Override
  public TemplateItemEntity findDetailsById(String templateItemId) {
    Validate.notBlank(templateItemId,"查询模版明细ID不能为空!");
    return templateItemRepository.findDetailsById(templateItemId);
  }

  @Override
  public TemplateItemEntity findParentById(String templateItemId) {
    if(StringUtils.isBlank(templateItemId)){
      return null;
    }
    return templateItemRepository.findParentById(templateItemId);
  }

  private void initDynamicProperties(TemplateItemEntity templateItem) {
    Set<TemplatePropertyEntity> templateProperties = templateItem.getProperties();
    for (TemplatePropertyEntity templateProperty : templateProperties) {
      templateProperty.setId(null);
      templateProperty.setCurrentGroup(null);
      templateProperty.setCurrentItem(templateItem);
      templateProperty.setCurrentTemplate(null);
      templateProperty.setProjectName(platformContext.getAppName());
      this.templatePropertyRepository.save(templateProperty);
    }
  }
  
  private void initDynamicRelations(TemplateItemEntity templateItem) {
    Set<TemplateRelationEntity> templateRelations = templateItem.getRelations(); 
    if(templateRelations != null && !templateRelations.isEmpty()) {
      for (TemplateRelationEntity templateRelation : templateRelations) {
        templateRelation.setCurrentGroup(null);
        templateRelation.setCurrentItem(templateItem);
        templateRelation.setCurrentTemplate(null);
        templateRelation.setId(null);
        templateRelation.setQueryService("");
        templateRelation.setProjectName(platformContext.getAppName());
        this.templateRelationRepository.save(templateRelation);
      }
    }
  }
}
