package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityButtonsEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.starter.common.enums.TemplateLayoutTypeEnum;
import com.bizunited.platform.kuiper.starter.repository.TemplateVisibilityButtonRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityButtonService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * 表单模版可见性按钮服务接口定义
 * @Author: Paul Chan
 * @Date: 2019-07-22 14:54
 */
@Service("TemplateVisibilityButtonServiceImpl")
public class TemplateVisibilityButtonServiceImpl implements TemplateVisibilityButtonService {

  @Autowired
  @Qualifier("KuiperToolkitService")
  private KuiperToolkitService kuiperToolkitService;
  @Autowired
  private TemplateVisibilityButtonRepository templateVisibilityButtonRepository;
  @Autowired
  private PlatformContext platformContext;

  /**
   * 保存前验证数据
   * @param visibility
   * @param buttons
   */
  private void saveValidation(TemplateVisibilityEntity visibility, Set<TemplateVisibilityButtonsEntity> buttons) {
    Validate.notNull(visibility, "可见性不能为空");
    if(CollectionUtils.isEmpty(buttons)) {
      return;
    }
    Set<String> controlIdSet = new HashSet<>();
    for (TemplateVisibilityButtonsEntity button : buttons) {
      Validate.notBlank(button.getControllerId(), "控件的唯一编号不能为空，请检查!!");
      Validate.notNull(button.getVisible(), "控件的可见性不能为空，请检查!!");
      Integer layoutType = button.getLayoutType() == null ? TemplateLayoutTypeEnum.PC.getType() : button.getLayoutType();
      boolean flag = controlIdSet.add(StringUtils.join(layoutType, "|", button.getControllerId()));
      Validate.isTrue(flag, "按钮controllerId编号重复,请检查!!");
    }
  }

  @Override
  @Transactional
  public Set<TemplateVisibilityButtonsEntity> save(TemplateVisibilityEntity visibility, Set<TemplateVisibilityButtonsEntity> buttons) {
    /**
     * 1、初始化可见性按钮
     * 2、验证参数
     * 3、定义结果集
     * 4、从数据库获取旧的可见性按钮数据，为null则初始化
     * 5、将提交的数据根据ControllerId作为key存到map中
     * 6、定义集合，用于接收新增、更新、删除的数据
     * 7、调用kuiperToolkitService.collectionDiscrepancy方法获取新增、更新、删除的数据
     * 8、保存新增的数据
     *   1、设置可见性按钮对象的属性值
     *   2、保存可见性按钮
     *   3、将可见性按钮添加到结果集中
     * 9、更新旧的数据
     *   1、设置可见性按钮对象的属性值
     *   2、保存可见性按钮
     *   3、将可见性按钮添加到结果集中
     * 10、删除旧的数据
     * 11、返回结果集
     */
    if(buttons == null) buttons = new HashSet<>();
    this.saveValidation(visibility, buttons);
    Set<TemplateVisibilityButtonsEntity> savedButtons = new HashSet<>();
    Set<TemplateVisibilityButtonsEntity> oldButtons = templateVisibilityButtonRepository.findByVisiblId(visibility.getId());
    if(oldButtons == null) oldButtons = new HashSet<>();
    Map<String, TemplateVisibilityButtonsEntity> buttonsMap = new HashMap<>(16);
    for (TemplateVisibilityButtonsEntity button : buttons) {
      Integer layoutType = button.getLayoutType() == null ? TemplateLayoutTypeEnum.PC.getType() : button.getLayoutType();
      String key = StringUtils.join(layoutType, "|", button.getControllerId());
      button.setLayoutType(layoutType);
      buttonsMap.put(key, button);
    }
    Set<TemplateVisibilityButtonsEntity> addButtons = new HashSet<>();
    Set<TemplateVisibilityButtonsEntity> updateButtons = new HashSet<>();
    Set<TemplateVisibilityButtonsEntity> deleteButtons = new HashSet<>();
    kuiperToolkitService.collectionDiscrepancy(buttons, oldButtons, button -> {
      Integer layoutType = button.getLayoutType() == null ? TemplateLayoutTypeEnum.PC.getType() : button.getLayoutType();
      return StringUtils.join(layoutType, "|", button.getControllerId());
    }, deleteButtons, updateButtons, addButtons);
    // 保存新增的数据
    for (TemplateVisibilityButtonsEntity button : addButtons) {
      button.setId(null);
      button.setTemplateVisibility(visibility);
      button.setProjectName(platformContext.getAppName());
      templateVisibilityButtonRepository.save(button);
      savedButtons.add(button);
    }
    // 更新就数据
    for (TemplateVisibilityButtonsEntity button : updateButtons) {
      Integer layoutType = button.getLayoutType() == null ? TemplateLayoutTypeEnum.PC.getType() : button.getLayoutType();
      String key = StringUtils.join(layoutType, "|", button.getControllerId());
      TemplateVisibilityButtonsEntity newButton = buttonsMap.get(key);
      button.setLayoutType(layoutType);
      button.setTemplateVisibility(visibility);
      button.setVisible(newButton.getVisible());
      templateVisibilityButtonRepository.save(button);
      savedButtons.add(button);
    }
    return savedButtons;
  }

  @Override
  @Transactional
  public void deleteByVisibilityId(String visibilityId) {
    Set<TemplateVisibilityButtonsEntity> buttons = templateVisibilityButtonRepository.findByVisiblId(visibilityId);
    if(!CollectionUtils.isEmpty(buttons)){
      templateVisibilityButtonRepository.deleteAll(buttons);
    }
  }

  @Override
  public Set<TemplateVisibilityButtonsEntity> findByVisibilityIdAndLayoutType(String visibilityId, Integer layoutType) {
    if(StringUtils.isBlank(visibilityId)) {
      return Sets.newHashSet();
    }
    if(layoutType == null) {
      layoutType = TemplateLayoutTypeEnum.PC.getType();
    }
    return templateVisibilityButtonRepository.findByVisibilityIdAndLayoutType(visibilityId, layoutType);
  }

}
