package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.FromDetailsImportEntity;
import com.bizunited.platform.kuiper.starter.service.FromDetailsImportService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.security.Principal;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 基于表单引擎，列表引擎，对业务数据通过xls文件的方式进行导入，并关联相关表单实例的服务的http restful接口层实现
 *
 * @author yinwenjie
 */
@RestController
@RequestMapping("/v1/kuiper/fromDetailsImports")
public class FromDetailsImportController extends BaseController {

  @Autowired
  private FromDetailsImportService fromDetailsImportService;
  @Autowired
  private VenusFileService venusFileService;

  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(FromDetailsImportController.class);

  @ApiOperation(value = "执行基于表单引擎、列表引起的业务数据导入过程，业务侧技术人员，只需要关注如何将xls/xlsx文件中的内容进行校验和对象转换")
  @PostMapping(value = "")
  public ResponseModel imports(@RequestParam(name = "file", required = true) @ApiParam(required = true, name = "file", value = "上传的xls/xlsx文件信息，目前只支持单个文件的上传") MultipartFile file,
                               @RequestParam(name = "templateCode", required = true) @ApiParam(required = true, name = "templateCode", value = "导入业务数据后，将为业务数据创建表单实例，该templateCode指向将要创建表单实例时所参照的模板") String templateCode,
                               @RequestParam(name = "templateVersion", required = false) @ApiParam(required = false, name = "templateVersion", value = "导入业务数据后，将为业务数据创建表单实例，该templateVersion指向将要创建表单实例时所参照的模板版本（该参数可以不传入，如果不传入，则会使用指定模板的默认版本，如果没有设置默认版本，则会抛出异常）") String templateVersion,
                               @RequestParam(name = "processClassName", required = true) @ApiParam(required = true, name = "processClassName", value = "用于处理本次xls业务数据导入所使用的完成导入处理类") String processClassName,
                               HttpServletRequest request,
                               Principal principal) {
    // 构建可能的params信息
    Map<String, Object> params = new HashMap<>();
    Enumeration<String> names = request.getParameterNames();
    if (names != null) {
      while (names.hasMoreElements()) {
        String name = names.nextElement();
        String values[] = request.getParameterValues(name);
        if (values != null && values.length == 1) {
          params.put(name, values[0]);
        } else if (values != null && values.length > 1) {
          params.put(name, values);
        }
      }
    }

    // 调用服务层信息
    try {
      fromDetailsImportService.imports(file.getBytes(), params, principal, templateCode, templateVersion, processClassName);
      return this.buildHttpResult();
    } catch (IOException | RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "通过该接口，系统将可以返回该进程中已经被扫描到的所有业务数据导入处理器的类名信息")
  @GetMapping(value = "/findProcessClassNames")
  public ResponseModel findProcessClassNames() {
    try {
      Set<String> results = this.fromDetailsImportService.findProcessClassNames();
      return this.buildHttpResultW(results, new String[]{});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  @ApiOperation(value = "该功能为指定的列表模板、基于指定的进行业务数据批量导入功能的按钮，上传xls、xlsx格式的模板文件使用。以保证使用该功能进行业务数据批量导入的xls、xlsx文件的统一性")
  @PostMapping(value = "/importTemplate")
  public ResponseModel importTemplate(@RequestParam(name = "file", required = true) @ApiParam(required = true, name = "file", value = "上传的xls/xlsx（模板）文件信息，目前只支持单个文件的上传") MultipartFile file,
                                      @RequestParam(name = "listTemplateCode", required = true) @ApiParam(required = true, name = "listTemplateCode", value = "对应的列表模板业务编号") String listTemplateCode,
                                      @RequestParam(name = "listTemplateVersion", required = true) @ApiParam(required = true, name = "listTemplateVersion", value = "对应的列表模板版本信息") String listTemplateVersion,
                                      @RequestParam(name = "buttonCode", required = true) @ApiParam(required = true, name = "buttonCode", value = "对应的列表模板中进行业务数据批量导入功能的按钮code") String buttonCode) {

    try {
      byte fileBytes[] = file.getBytes();
      String originalFilename = file.getOriginalFilename();
      FromDetailsImportEntity result = this.fromDetailsImportService.importTemplate(fileBytes, listTemplateCode, listTemplateVersion, buttonCode, originalFilename);
      return this.buildHttpResultW(result, new String[]{});
    } catch (IOException | RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 按照给定的文件路径和文件名，下载一般性文件。
   *
   * @param response
   * @param relativeLocal
   * @param fileName
   * @throws IOException
   */
  @ApiOperation(value = "按照给定的文件路径和文件名，下载一般性文件。")
  @RequestMapping(value = "/downloadTemplate", method = RequestMethod.GET)
  public void downloadTemplate(HttpServletRequest request, HttpServletResponse response,
                               @ApiParam(name = "relativeLocal", value = "相对路径", required = true) @RequestParam("relativeLocal") String relativeLocal,
                               @ApiParam(name = "fileName", value = "文件名（重命名后的）", required = true) @RequestParam("fileName") String fileName) throws IOException {

    FromDetailsImportEntity ordinaryFile = this.fromDetailsImportService.findByFileNameAndRelativeLocal(fileName, relativeLocal);
    if (ordinaryFile == null) {
      return;
    }
    byte[] bytes = venusFileService.readFileContent(relativeLocal, fileName);
    if (bytes == null) {
      return;
    }
    // 附件文件下载
    // modified by keller 2020-09-24 下载模板文件时候使用原文件名
    String downloadFileName = ordinaryFile.getOriginalFilename();
    if (StringUtils.isBlank(downloadFileName)) {
      //避免历史数据下载时候该字段为空的情况
      downloadFileName = ordinaryFile.getFileName();
    }
    this.writeResponseFile(request, response, bytes, downloadFileName);
  }

  /**
   * 按照给定的列表编码，列表版本以及按钮的编号，查询导入模板文件并下载
   *
   * @param listTemplateCode
   * @param listTemplateVersion
   * @param buttonCode
   */
  @ApiOperation(value = "按照给定的列表编码，列表版本以及按钮的编号，查询导入模板文件")
  @RequestMapping(value = "/downloadTemplateByCode", method = RequestMethod.GET)
  public void downloadTemplateByCode(HttpServletRequest request, HttpServletResponse response,
                                     @ApiParam(name="listTemplateCode",value = "列表编码",required = true) @RequestParam("listTemplateCode") String listTemplateCode,
                                     @ApiParam(name="listTemplateVersion",value = "列表版本号",required = true) @RequestParam("listTemplateVersion") String listTemplateVersion,
                                     @ApiParam(name="buttonCode",value = "按钮编码",required = true) @RequestParam("buttonCode") String buttonCode) {
    FromDetailsImportEntity ordinaryFile = this.fromDetailsImportService.findByCodeAndVersionAndButtonCode(listTemplateCode,listTemplateVersion,buttonCode);
    if (ordinaryFile == null) {
      return;
    }
    byte[] bytes = venusFileService.readFileContent(ordinaryFile.getRelativePath(), ordinaryFile.getFileName());
    if (bytes == null) {
      return;
    }
    // 附件文件下载
    String downloadFileName = ordinaryFile.getOriginalFilename();
    if (StringUtils.isBlank(downloadFileName)) {
      //避免历史数据下载时候该字段为空的情况
      downloadFileName = ordinaryFile.getFileName();
    }
    this.writeResponseFile(request, response, bytes, downloadFileName);
  }

  /**
   * 按照给定的列表编码，列表版本以及按钮的编号，查询导入模板文件
   * @param request
   * @param response
   * @param listTemplateCode
   * @param listTemplateVersion
   * @param buttonCode
   * @return
   */
  @ApiOperation(value = "按照给定的列表编码，列表版本以及按钮的编号，查询导入模板文件")
  @GetMapping("/findTemplateByCode")
  public ResponseModel findTemplateByCode(HttpServletRequest request, HttpServletResponse response,
                                          @ApiParam(name="listTemplateCode",value = "列表编码",required = true) @RequestParam("listTemplateCode") String listTemplateCode,
                                          @ApiParam(name="listTemplateVersion",value = "列表版本号",required = true) @RequestParam("listTemplateVersion") String listTemplateVersion,
                                          @ApiParam(name="buttonCode",value = "按钮编码",required = true) @RequestParam("buttonCode") String buttonCode) {
    try {
      FromDetailsImportEntity entity = fromDetailsImportService.findByCodeAndVersionAndButtonCode(listTemplateCode,listTemplateVersion,buttonCode);
      return this.buildHttpResultW(entity);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}