package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.entity.InstanceViewEntity;
import com.bizunited.platform.kuiper.service.InstanceService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;
import java.util.Date;

@RestController
@RequestMapping("/v1/kuiper/instances")
public class InstanceController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(InstanceController.class);
  @Autowired
  private InstanceService instanceService;
  private static final String ACTIVITIES = "activities";
  private static final String TEMPLATE = "template";
  
  @ApiOperation(value = "创建一个新的表单实例，以及表单实例中第一个名叫create的活动实例。并且这个名叫create的活动实例的taskCode按照传入的taskCode为准")
  @PostMapping("")
  @CacheEvict(cacheNames="instance" , allEntries=true)
  public ResponseModel createByTemplateId(@ApiParam(name = "templateId", value = "初始化表单实例所需要的模板数据层编号信息") @RequestParam("templateId") String templateId , 
                              @ApiParam(name = "taskCode", value = "指定的可能传入的，将要创建的create名称的活动实例的唯一外部业务编号" , required = false) @RequestParam(name = "taskCode" , required = false) String taskCode) {
    try {
      Principal principal = this.getPrincipal();
      String account = principal.getName();
      InstanceEntity currentInstance = this.instanceService.createByTemplateId(templateId , taskCode, account);
      return this.buildHttpResultW(currentInstance, TEMPLATE , ACTIVITIES);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 只需要传入一个表单模板的业务编号（code），表单引擎就可以根据这个code寻找到相关的最新版本，并基于这个最新版本的模板自动创建表单实例
   * @param templateCode 指定的实例基本信息——主要要有其中关联的模板数据层编号信息
   */
  @ApiOperation(value = "只需要传入一个表单模板的业务编号（code），表单引擎就可以根据这个code寻找到相关的最新版本，并基于这个最新版本的模板自动创建表单实例")
  @PostMapping("/{code}")
  @CacheEvict(cacheNames="instance" , allEntries=true)
  public ResponseModel createByTemplateCode(@ApiParam(name = "code", value = "指定的实例基本信息——主要要有其中关联的模板业务编号信息") @PathVariable("code") String templateCode , 
                              @ApiParam(name = "taskCode", value = "指定的可能传入的，将要创建的create名称的活动实例的唯一外部业务编号" , required = false) @RequestParam(name = "taskCode" , required = false) String taskCode) {
    try {
      Principal principal = this.getPrincipal();
      String account = principal.getName();
      InstanceEntity currentInstance = this.instanceService.createByTemplateCode(templateCode , taskCode , account);
      return this.buildHttpResultW(currentInstance, TEMPLATE , ACTIVITIES);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 只需要传入一个表单模板的业务编号（code）和版本号（cversion），表单引擎就可以根据这个code和cversion寻找到相关的模版，并基于这个模板自动创建表单实例
   * @param templateCode 指定的实例基本信息——主要要有其中关联的模板数据层编号信息
   */
  @ApiOperation(value = "只需要传入一个表单模板的业务编号（code）和版本号（cversion），表单引擎就可以根据这个code和cversion寻找到相关的模版，并基于这个模板自动创建表单实例")
  @PostMapping("/{code}/{cversion}")
  @CacheEvict(cacheNames="instance" ,allEntries=true)
  public ResponseModel createByTemplateCode(@PathVariable("code") @ApiParam(name = "code", value = "指定的实例基本信息——主要要有其中关联的模板业务编号信息") String templateCode,
                              @PathVariable("cversion") @ApiParam(name = "cversion", value = "指定的实例基本信息——主要要有其中关联的模板业务编号信息") String cversion,
                              @RequestParam(required = false) @ApiParam(name = "taskCode", value = "指定的可能传入的，将要创建的create名称的活动实例的唯一外部业务编号") String taskCode) {
    try {
      Principal principal = this.getPrincipal();
      String account = principal.getName();
      InstanceEntity currentInstance = this.instanceService.createByTemplateCode(templateCode, cversion, taskCode , account);
      return this.buildHttpResultW(currentInstance, TEMPLATE, ACTIVITIES);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
 
  @ApiOperation(value = "根据表单实例的数据层编号，查询这个实例的详情——包括实例的直接关联信息")
  @GetMapping("/findDetailsById")
  @Cacheable(cacheNames="instance" , key="'findDetailsById_' + #instanceId" , sync=true)
  public ResponseModel findDetailsById(@ApiParam(name = "instanceId", value = "指定的表单实例编号") String instanceId) {
    try {
      InstanceEntity instance = this.instanceService.findDetailsById(instanceId);
      return this.buildHttpResultW(instance, TEMPLATE ,ACTIVITIES , "creator");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "按条件搜索，目前支持表单模板名、业务域名和项目名三个条件，不支持模糊查询")
  @GetMapping("/findByConditions")
  public ResponseModel findByConditions(
    @ApiParam(name = "code", value = "实例对应模板的模板编号" , required=false) @RequestParam(name="code" , required=false) String code,
    @ApiParam(name = "instanceId", value = "实例ID" , required=false) @RequestParam(name="instanceId" , required=false) String instanceId,
    @ApiParam(name = "cversion", value = "实例对应模板的状态" , required=false) @RequestParam(name="cversion" , required=false) String cversion,
    @ApiParam(name = "domain", value = "实例域" , required=false) @RequestParam(name="domain" , required=false) String domain,
    @ApiParam(name = "templateName", value = "模板名" , required=false) @RequestParam(name="templateName" , required=false) String templateName,
    @ApiParam(name = "projectName", value = "实例工程名" , required=false) @RequestParam(name="projectName" , required=false) String projectName,
    @ApiParam(name = "beginCreateTime", value = "可能的实例创建时间（起），时间格式为yyyy-MM-dd HH:mm:ss" , required=false) @RequestParam(name="beginCreateTime" , required=false) @DateTimeFormat(pattern="yyyy-MM-dd HH:mm:ss") Date beginCreateTime,
    @ApiParam(name = "endCreateTime", value = "可能的实例创建时间（止），时间格式为yyyy-MM-dd HH:mm:ss" , required=false) @RequestParam(name="endCreateTime" , required=false) @DateTimeFormat(pattern="yyyy-MM-dd HH:mm:ss") Date endCreateTime,
    @ApiParam(name = "userName", value = "可能的创建者用户名（真实姓名）" , required=false) @RequestParam(name="userName" , required=false) String userName,
    @PageableDefault(value = 50) @ApiParam(name = "pageable", value = "分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50") Pageable pageable) { 
    try {
      Page<InstanceViewEntity> page = this.instanceService.findByConditions(code, instanceId, cversion, domain, templateName, projectName, beginCreateTime, endCreateTime, userName, pageable);
      return this.buildHttpResultW(page);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "根据指定的表单模板编号信息，查询这个表单模板下涉及的表单实例数量")
  @GetMapping("/countByTemplateId")
  public ResponseModel countByTemplateId(@ApiParam(name = "templateId", value = "指定的表单模板编号信息") @RequestParam("templateId") String templateId) {
    try {
      Integer count = this.instanceService.countByTemplateId(templateId);
      return this.buildHttpResultW(count);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据模版ID分页查询
   * @return
   */
  @GetMapping("findByTemplateId")
  @ApiOperation("根据模版ID分页查询")
  public ResponseModel findByTemplateId(@RequestParam @ApiParam("模版ID") String templateId,
                                        @PageableDefault(value = 50) @ApiParam(value = "分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50") Pageable pageable) {
    try {
      Page<InstanceEntity> page = this.instanceService.findByTemplateId(templateId, pageable);
      return this.buildHttpResultW(page, "template");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


}
