package com.bizunited.platform.kuiper.starter.repository;

import com.bizunited.platform.kuiper.entity.ListTemplateEntity;
import com.bizunited.platform.kuiper.starter.repository.internal.ListTemplateRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * ListTemplateEntity业务模型的数据库方法支持
 * @author yinwenjie
 */
@Repository("ListTemplateEntityRepository")
public interface ListTemplateRepository
    extends
      JpaRepository<ListTemplateEntity, String>
      ,JpaSpecificationExecutor<ListTemplateEntity>
      , ListTemplateRepositoryCustom {

  /**
   * 按照主键进行详情查询（包括关联信息），这里查询的是
   * @param id 主键
   * */
  @Query("select distinct listTemplateEntity from ListTemplateEntity listTemplateEntity "
      + " where listTemplateEntity.id=:id ")
  public ListTemplateEntity findDetailsById(@Param("id") String id);
  /**
   * 按照列表模板编码和版本查询指定的列表模板信息
   * @param code 列表编码
   * @param cversion 列表编码的版本号
   * @return
   */
  @Query("from ListTemplateEntity le where le.code = :code and le.cversion = :cversion ")
  public ListTemplateEntity findByCodeAndCversion(@Param("code") String code , @Param("cversion") String cversion);
  /**
   * 按照列表模板的编码进行查询，并按照创建时间顺序进行反序排泄
   * @param code 指定的编码
   * @return
   */
  @Query("from ListTemplateEntity t where t.code = :code order by t.createTime desc ")
  public Set<ListTemplateEntity> findByCode(@Param("code") String code);
  /**
   * 查询一批指定的id编号的列表模板信息在系统中真实存在的数量
   * @param ids
   * @return
   */
  @Query(value="select count(*) from engine_form_list_template where id in (:ids)" , nativeQuery = true)
  public int countByIds(@Param("ids") String[] ids);

  /**
   * 按照列表模板编码和版本查询指定的列表模板信息（包括关联信息）
   * @param code 指定的编码
   * @param cversion 版本号
   * @return
   */
  @Query("select distinct listTemplateEntity from ListTemplateEntity listTemplateEntity "
            + " where listTemplateEntity.code=:code and listTemplateEntity.cversion = :cversion ")
  ListTemplateEntity findDetailsByCodeAndCversion(@Param("code") String code, @Param("cversion") String cversion);

  /**
   * 按照列表模板的编码和版本号查询默认版本
   * @param code 指定的编码
   * @return
   */
  @Query("from ListTemplateEntity le where le.code = :code and le.defaultVersion = 1 ")
  ListTemplateEntity findDefaultByCode(@Param("code") String code);


  /**
   * 把当前code对应的所有列表模板版设置为非默认版本或默认版本
   * @param code 指定的编码* @param defaultVersion 是否默认版本
   */
  @Modifying
  @Query(value="update engine_form_list_template set default_version = :defaultVersion where code = :code ", nativeQuery=true)
  void updateDefaultVersion(@Param("code") String code, @Param("defaultVersion") Boolean defaultVersion);


  /**
   * 把当前code和cversion对应的这条列表模板版设置为非默认版本或默认版本
   * @param code 指定的编码
   * @param cversion 版本号
   * @param defaultVersion 是否默认版本
   */
  @Modifying
  @Query(value="update engine_form_list_template set default_version = :defaultVersion, modify_time = :modifyTime where code = :code and cversion = :cversion ", nativeQuery=true)
  void updateDefaultVersion(@Param("code") String code,
                            @Param("cversion") String cversion,
                            @Param("defaultVersion") Boolean defaultVersion,
                            @Param("modifyTime") Date modifyTime);

  /**
   * 根据编码统计当前code下的列表模板的数量
   * @param code 指定的编码
   * @return
   */
  @Query(value="select count(*) from engine_form_list_template where code = :code ", nativeQuery = true)
  Long countByCode(@Param("code") String code);

  /**
   * 根据ID集合查询列表数据
   * @param ids
   * @return
   */
  @Query("select lt from ListTemplateEntity lt where lt.id in :ids")
  List<ListTemplateEntity> findByIds(@Param("ids") String[] ids);

  /**
   * 根据编码和版本号统计数据
   * @param code
   * @param version
   * @return
   */
  @Query("select count(*) from ListTemplateEntity lt where lt.code = :code and lt.cversion = :version ")
  long countByCodeAndVersion(@Param("code") String code, @Param("version") String version);
}