package com.bizunited.platform.kuiper.starter.repository;

import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;

import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.Set;

@Repository("TemplateVisibilityRepository")
public interface TemplateVisibilityRepository extends JpaRepository<TemplateVisibilityEntity, String>, JpaSpecificationExecutor<TemplateVisibilityEntity> {
  /**
   * 按照指定的模板id查询表单可见性
   * @param templateId
   * @return
   */
  @Query("select tve from TemplateVisibilityEntity tve " 
   + "left join fetch tve.template t "
   + "left join fetch tve.attributes tva " 
   + "left join fetch tve.buttons tvb "
   + "where t.id = :templateId  order by tve.createTime")
  Set<TemplateVisibilityEntity> findDetailsByTemplateId(@Param("templateId") String templateId);

  /**
   * 按照指定的模板ids查询表单可见性
   * @param templateIds
   * @return
   */
  @Query("select tve from TemplateVisibilityEntity tve "
          + "left join fetch tve.template t "
          + "left join fetch tve.attributes tva "
          + "left join fetch tve.buttons tvb "
          + "where t.id in (:templateIds) ")
  Set<TemplateVisibilityEntity> findDetailsByTemplateIds(@Param("templateIds") Set<String> templateIds);

  /**
   * 按照指定的模板id，找到所有相关的可见性信息
   * @param templateId 指定的模板信息
   * @return 
   */
  @Query("from TemplateVisibilityEntity tv where tv.template.id = :templateId order by tv.createTime ")
  public Set<TemplateVisibilityEntity> findByTemplateId(@Param("templateId") String templateId);

  /**
   * 按照指定的模板数据编号和指定的可见性名称，查询指定的可见性信息（只包括可见性的基本信息）
   * @param templateId 指定的模板编号信息
   * @param visibilityName 指定的可见性名称
   */
  @Query("from TemplateVisibilityEntity tv inner join fetch tv.template t where t.id = :templateId and tv.visibilityName = :visibilityName ")
  public TemplateVisibilityEntity findByTemplateIdAndVisibilityName(@Param("templateId") String templateId , @Param("visibilityName") String visibilityName);

  /**
   * 根据模板ID和可见性名称，查询是否存在给定的可见性
   * @param templateId
   * @param visibilityName
   * @return
   */
  @Query(value = "select count(*) from engine_form_visibility et where et.template_id = :templateId and et.visibility_name = :visibilityName ", nativeQuery = true)
  int countVisibilityName(@Param("templateId") String templateId, @Param("visibilityName") String visibilityName);

  /**
   * 根据模版和布局查询可见性及其属性
   * @param templateId
   * @param layoutType
   * @return
   */
  @Query("select tve from TemplateVisibilityEntity tve "
      + "left join fetch tve.template t "
      + "left join fetch tve.attributes tva "
      + "where t.id = :templateId and tva.layoutType = :layoutType  order by tve.createTime")
  Set<TemplateVisibilityEntity> findVisibilityAndAttributesByTemplateIdAndLayoutType(@Param("templateId") String templateId, @Param("layoutType") Integer layoutType);

  /**
   * 根据模版和布局查询可见性及其按钮
   * @param templateId
   * @param layoutType
   * @return
   */
  @Query("select tve from TemplateVisibilityEntity tve "
      + "left join fetch tve.template t "
      + "left join fetch tve.buttons tvb "
      + "where t.id = :templateId and tvb.layoutType = :layoutType  order by tve.createTime")
  Set<TemplateVisibilityEntity> findVisibilityAndButtonsByTemplateIdAndLayoutType(@Param("templateId") String templateId, @Param("layoutType") Integer layoutType);


  /**
   * 根据模版和版本号获取可见性
   * @param templateCode
   * @param version
   * @return
   */
  @Query("select tv from TemplateVisibilityEntity tv inner join tv.template t where t.code = :templateCode and t.cversion = :version  order by tv.createTime asc")
  Set<TemplateVisibilityEntity> findByTemplateCodeAndVersion(@Param("templateCode") String templateCode, @Param("version") String version);

  /**
   * 根据模版编号和模版默认版本获取可见性
   * @param templateCode
   * @return
   */
  @Query("select tv from TemplateVisibilityEntity tv inner join tv.template t where t.code = :templateCode and t.defaultVersion = 1  order by tv.createTime asc")
  Set<TemplateVisibilityEntity> findByTemplateCodeAndDefaultVersion(@Param("templateCode") String templateCode);
}
