package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.kuiper.entity.DynamicTemplateDraftEntity;
import com.bizunited.platform.kuiper.starter.repository.DynamicTemplateDraftRepository;
import com.bizunited.platform.kuiper.starter.service.DynamicTemplateDraftService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;
import javax.persistence.criteria.Predicate;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;

/**
 * 动态模板草稿业务模型的服务层接口实现
 * @author saturn
 */
@Service("DynamicTemplateDraftServiceImpl")
public class DynamicTemplateDraftServiceImpl implements DynamicTemplateDraftService {

  private static final  String JSON_FILE_SUFFIX = "json";


  @Autowired
  private DynamicTemplateDraftRepository dynamicTemplateDraftRepository;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 在创建一个新的动态模板草稿模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  @Override
  public void createValidation(DynamicTemplateDraftEntity dynamicTemplateDraftEntity) {
    Validate.notNull(dynamicTemplateDraftEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dynamicTemplateDraftEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    dynamicTemplateDraftEntity.setId(null);
    Validate.notBlank(dynamicTemplateDraftEntity.getCode(), "业务编号不能为空！");
    Validate.notBlank(dynamicTemplateDraftEntity.getName(), "业务名不能为空！");
    Validate.notBlank(dynamicTemplateDraftEntity.getTableName(), "动态模板设定的数据表名不能为空！");
    Validate.notBlank(dynamicTemplateDraftEntity.getCversion(), "版本号不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(dynamicTemplateDraftEntity.getCode() == null || dynamicTemplateDraftEntity.getCode().length() < 255 , "业务名/业务编号,填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(dynamicTemplateDraftEntity.getTableName() == null || dynamicTemplateDraftEntity.getTableName().length() < 255 , "动态模板设定的数据表名,填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(dynamicTemplateDraftEntity.getCversion() == null || dynamicTemplateDraftEntity.getCversion().length() < 255 , "版本号,填入值超过了限定长度(255)，请检查!");
    DynamicTemplateDraftEntity currentDynamicTemplateDraftEntity = this.dynamicTemplateDraftRepository.findByCode(dynamicTemplateDraftEntity.getCode());
    Validate.isTrue(currentDynamicTemplateDraftEntity == null, "该业务名/业务编号的草稿已存在,请在草稿上编辑");
    currentDynamicTemplateDraftEntity = this.dynamicTemplateDraftRepository.findByTableName(dynamicTemplateDraftEntity.getTableName());
    Validate.isTrue(currentDynamicTemplateDraftEntity == null, "该表名的草稿已存在,请在草稿上编辑");
  }

  /**
   * 创建动态模版的草稿
   * @param dynamicTemplateDraftEntity
   * @param json
   * @return
   */
  @Override
  @Transactional(rollbackOn = Exception.class)
  public DynamicTemplateDraftEntity create(DynamicTemplateDraftEntity dynamicTemplateDraftEntity , String json) {
    this.createValidation(dynamicTemplateDraftEntity);
    Validate.notBlank(json, "动态模板的草稿json数据不能为空");
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String renameImage = UUID.randomUUID().toString();
    String fileRename = renameImage + "." + JSON_FILE_SUFFIX;
    String relativePath = StringUtils.join("/draft/" , folderName , "/" , (new Random().nextInt(100) % 10));
    byte[] jsonContent = json.getBytes(StandardCharsets.UTF_8);
    venusFileService.saveFile(relativePath, fileRename, fileRename,jsonContent);
    dynamicTemplateDraftEntity.setCreateTime(nowDate);
    dynamicTemplateDraftEntity.setModifyTime(nowDate);
    dynamicTemplateDraftEntity.setFileCode(relativePath);
    dynamicTemplateDraftEntity.setFileName(fileRename);
    dynamicTemplateDraftEntity.setProjectName(platformContext.getAppName());
    this.dynamicTemplateDraftRepository.save(dynamicTemplateDraftEntity);
    return dynamicTemplateDraftEntity;
  }


  /**
   * 更新动态模版的草稿
   * @param dynamicTemplateDraft
   * @param json
   * @return
   */
  @Override
  @Transactional
  public DynamicTemplateDraftEntity update(DynamicTemplateDraftEntity dynamicTemplateDraft, String json) {
    Validate.notNull(dynamicTemplateDraft, "未能获取到提交的草稿信息，请检查!!");
    Validate.notBlank(dynamicTemplateDraft.getId(), "草稿的id不能为空");
    Validate.notBlank(json, "动态模板的草稿json数据不能为空");
    String id = dynamicTemplateDraft.getId();
    DynamicTemplateDraftEntity dynamicTemplateDraftEntity = this.findDetailsById(id);
    Validate.notNull(dynamicTemplateDraftEntity, "草稿不存在");
    this.updateValidate(dynamicTemplateDraft);
    byte[] jsonContent = json.getBytes(StandardCharsets.UTF_8);
    String fileName = dynamicTemplateDraftEntity.getFileName();
    venusFileService.saveFile(dynamicTemplateDraftEntity.getFileCode(), fileName, fileName,jsonContent);
    dynamicTemplateDraftEntity.setModifyTime(new Date());
    if(StringUtils.isNotBlank(dynamicTemplateDraft.getCode())){
      dynamicTemplateDraftEntity.setCode(dynamicTemplateDraft.getCode());
    }
    if(StringUtils.isNotBlank(dynamicTemplateDraft.getTableName())){
      dynamicTemplateDraftEntity.setTableName(dynamicTemplateDraft.getTableName());
    }
    if(StringUtils.isNotBlank(dynamicTemplateDraft.getCversion())){
      dynamicTemplateDraftEntity.setCversion(dynamicTemplateDraft.getCversion());
    }
    if(StringUtils.isNotBlank(dynamicTemplateDraft.getName())){
      dynamicTemplateDraftEntity.setName(dynamicTemplateDraft.getName());
    }
    dynamicTemplateDraftEntity.setProjectName(platformContext.getAppName());
    this.dynamicTemplateDraftRepository.save(dynamicTemplateDraftEntity);
    return dynamicTemplateDraftEntity;
  }

  /**
   * 更新参数验证
   * @param dynamicTemplateDraft
   */
  private void updateValidate(DynamicTemplateDraftEntity dynamicTemplateDraft){
    if(StringUtils.isNotBlank(dynamicTemplateDraft.getCode())){
      Long count = dynamicTemplateDraftRepository.countByCodeWithoutId(dynamicTemplateDraft.getId(), dynamicTemplateDraft.getCode());
      Validate.isTrue(count == 0, "业务编号已存在");
    }
    if(StringUtils.isNotBlank(dynamicTemplateDraft.getTableName())){
      Long count = dynamicTemplateDraftRepository.countByTableNameWithoutId(dynamicTemplateDraft.getId(), dynamicTemplateDraft.getTableName());
      Validate.isTrue(count == 0, "业务表名已存在");
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.formengine.starter.service.DynamicTemplateDraftService#findById(java.lang.String)
   */
  @Override
  public DynamicTemplateDraftEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.dynamicTemplateDraftRepository.findDetailsById(id);
  }

  /**
   *
   * @param id 主键
   */
  @Override
  @Transactional(rollbackOn = Exception.class)
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DynamicTemplateDraftEntity> op = this.dynamicTemplateDraftRepository.findById(id);
    DynamicTemplateDraftEntity current = op.orElse(null);
    Validate.notNull(current, "删除的模版草稿对象不存在");

    this.dynamicTemplateDraftRepository.delete(current);
    venusFileService.deleteFile(current.getFileCode(), current.getFileName(), current.getFileName());
  }

  @Override
  @Transactional(rollbackOn = Exception.class)
  public void deleteByCode(String code){
    Validate.notBlank(code, "业务编码不能为空");
    DynamicTemplateDraftEntity draftEntity = this.dynamicTemplateDraftRepository.findByCode(code);
    if(draftEntity == null){
      return;
    }
    this.dynamicTemplateDraftRepository.delete(draftEntity);
    venusFileService.deleteFile(draftEntity.getFileCode(), draftEntity.getFileName(), draftEntity.getFileName());
  }

  @Override
  public Page<DynamicTemplateDraftEntity> findByConditions(Pageable pageable, String tableName) {
    Sort sort = new Sort(Sort.Direction.DESC, "modifyTime");
    Pageable currentPageable = PageRequest.of(pageable.getPageNumber(), pageable.getPageSize(), sort);
    Specification<DynamicTemplateDraftEntity> specification = (root, query, cb) -> {
      List<Predicate> predicates = new ArrayList<>();
      if(StringUtils.isNotBlank(tableName)){
        Predicate predicate = cb.equal(root.get("tableName").as(String.class), tableName);
        predicates.add(predicate);
      }
      return cb.and(predicates.toArray(new Predicate[0]));
    };
    return this.dynamicTemplateDraftRepository.findAll(specification, currentPageable);
  }

  @Override
  public String findJsonById(String id) {
    Validate.notBlank(id, "ID不能为空");
    DynamicTemplateDraftEntity result = this.findDetailsById(id);
    Validate.notNull(result, "未找到模版草稿");
    byte[] bytes = venusFileService.readFileContent(result.getFileCode(), result.getFileName());
    return new String(bytes, StandardCharsets.UTF_8);
  }

} 
